"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesis = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesis extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesis';
        this.actionList = {
            "AddTagsToStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html",
                "description": "Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html",
                "description": "Creates a Amazon Kinesis stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DecreaseStreamRetentionPeriod": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html",
                "description": "Decreases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html",
                "description": "Deletes a stream and all its shards and data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeregisterStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html",
                "description": "Deregisters a stream consumer with a Kinesis data stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeLimits": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html",
                "description": "Describes the shard limits and usage for the account.",
                "accessLevel": "Read"
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html",
                "description": "Describes the specified stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html",
                "description": "Gets the description of a registered stream consumer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeStreamSummary": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html",
                "description": "Provides a summarized description of the specified Kinesis data stream without the shard list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DisableEnhancedMonitoring": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html",
                "description": "Disables enhanced monitoring.",
                "accessLevel": "Write"
            },
            "EnableEnhancedMonitoring": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html",
                "description": "API_EnableEnhancedMonitoring.html",
                "accessLevel": "Write"
            },
            "GetRecords": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html",
                "description": "Gets data records from a shard.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetShardIterator": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html",
                "description": "Gets a shard iterator. A shard iterator expires five minutes after it is returned to the requester.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "IncreaseStreamRetentionPeriod": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html",
                "description": "Increases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "ListShards": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html",
                "description": "Lists the shards in a stream and provides information about each shard.",
                "accessLevel": "List"
            },
            "ListStreamConsumers": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html",
                "description": "Lists the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer.",
                "accessLevel": "List"
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html",
                "description": "Lists your streams.",
                "accessLevel": "List"
            },
            "ListTagsForStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html",
                "description": "Lists the tags for the specified Amazon Kinesis stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "MergeShards": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html",
                "description": "Merges two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "PutRecord": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html",
                "description": "Writes a single data record from a producer into an Amazon Kinesis stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "PutRecords": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html",
                "description": "Writes multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "RegisterStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html",
                "description": "Registers a stream consumer with a Kinesis data stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html",
                "description": "Description for SplitShard",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "SplitShard": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html",
                "description": "Description for SplitShard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "StartStreamEncryption": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StartStreamEncryption.html",
                "description": "Grants permission to enable or update server-side encryption using an AWS KMS key for a specified stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "kmsKey": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "StopStreamEncryption": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StopStreamEncryption.html",
                "description": "Grants permission to disable server-side encryption for a specified stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "kmsKey": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "SubscribeToShard": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html",
                "description": "Listening to a specific shard with enhanced fan-out.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateShardCount": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html",
                "description": "Updates the shard count of the specified stream to the specified number of shards.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-streams.html",
                "arn": "arn:${Partition}:kinesis:${Region}:${Account}:stream/${StreamName}",
                "conditionKeys": []
            },
            "consumer": {
                "name": "consumer",
                "url": "https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-consumers.html",
                "arn": "arn:${Partition}:kinesis:${Region}:${Account}:${StreamType}/${StreamName}/consumer/${ConsumerName}:${ConsumerCreationTimpstamp}",
                "conditionKeys": []
            },
            "kmsKey": {
                "name": "kmsKey",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html
     */
    toAddTagsToStream() {
        this.add('kinesis:AddTagsToStream');
        return this;
    }
    /**
     * Creates a Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html
     */
    toCreateStream() {
        this.add('kinesis:CreateStream');
        return this;
    }
    /**
     * Decreases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html
     */
    toDecreaseStreamRetentionPeriod() {
        this.add('kinesis:DecreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Deletes a stream and all its shards and data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html
     */
    toDeleteStream() {
        this.add('kinesis:DeleteStream');
        return this;
    }
    /**
     * Deregisters a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html
     */
    toDeregisterStreamConsumer() {
        this.add('kinesis:DeregisterStreamConsumer');
        return this;
    }
    /**
     * Describes the shard limits and usage for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        this.add('kinesis:DescribeLimits');
        return this;
    }
    /**
     * Describes the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html
     */
    toDescribeStream() {
        this.add('kinesis:DescribeStream');
        return this;
    }
    /**
     * Gets the description of a registered stream consumer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html
     */
    toDescribeStreamConsumer() {
        this.add('kinesis:DescribeStreamConsumer');
        return this;
    }
    /**
     * Provides a summarized description of the specified Kinesis data stream without the shard list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html
     */
    toDescribeStreamSummary() {
        this.add('kinesis:DescribeStreamSummary');
        return this;
    }
    /**
     * Disables enhanced monitoring.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html
     */
    toDisableEnhancedMonitoring() {
        this.add('kinesis:DisableEnhancedMonitoring');
        return this;
    }
    /**
     * API_EnableEnhancedMonitoring.html
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html
     */
    toEnableEnhancedMonitoring() {
        this.add('kinesis:EnableEnhancedMonitoring');
        return this;
    }
    /**
     * Gets data records from a shard.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html
     */
    toGetRecords() {
        this.add('kinesis:GetRecords');
        return this;
    }
    /**
     * Gets a shard iterator. A shard iterator expires five minutes after it is returned to the requester.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html
     */
    toGetShardIterator() {
        this.add('kinesis:GetShardIterator');
        return this;
    }
    /**
     * Increases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html
     */
    toIncreaseStreamRetentionPeriod() {
        this.add('kinesis:IncreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Lists the shards in a stream and provides information about each shard.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html
     */
    toListShards() {
        this.add('kinesis:ListShards');
        return this;
    }
    /**
     * Lists the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html
     */
    toListStreamConsumers() {
        this.add('kinesis:ListStreamConsumers');
        return this;
    }
    /**
     * Lists your streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.add('kinesis:ListStreams');
        return this;
    }
    /**
     * Lists the tags for the specified Amazon Kinesis stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        this.add('kinesis:ListTagsForStream');
        return this;
    }
    /**
     * Merges two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html
     */
    toMergeShards() {
        this.add('kinesis:MergeShards');
        return this;
    }
    /**
     * Writes a single data record from a producer into an Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        this.add('kinesis:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html
     */
    toPutRecords() {
        this.add('kinesis:PutRecords');
        return this;
    }
    /**
     * Registers a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html
     */
    toRegisterStreamConsumer() {
        this.add('kinesis:RegisterStreamConsumer');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html
     */
    toRemoveTagsFromStream() {
        this.add('kinesis:RemoveTagsFromStream');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html
     */
    toSplitShard() {
        this.add('kinesis:SplitShard');
        return this;
    }
    /**
     * Grants permission to enable or update server-side encryption using an AWS KMS key for a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StartStreamEncryption.html
     */
    toStartStreamEncryption() {
        this.add('kinesis:StartStreamEncryption');
        return this;
    }
    /**
     * Grants permission to disable server-side encryption for a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StopStreamEncryption.html
     */
    toStopStreamEncryption() {
        this.add('kinesis:StopStreamEncryption');
        return this;
    }
    /**
     * Listening to a specific shard with enhanced fan-out.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html
     */
    toSubscribeToShard() {
        this.add('kinesis:SubscribeToShard');
        return this;
    }
    /**
     * Updates the shard count of the specified stream to the specified number of shards.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html
     */
    toUpdateShardCount() {
        this.add('kinesis:UpdateShardCount');
        return this;
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-streams.html
     *
     * @param streamName - Identifier for the streamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(streamName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:stream/${StreamName}';
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type consumer to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-consumers.html
     *
     * @param streamType - Identifier for the streamType.
     * @param streamName - Identifier for the streamName.
     * @param consumerName - Identifier for the consumerName.
     * @param consumerCreationTimpstamp - Identifier for the consumerCreationTimpstamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConsumer(streamType, streamName, consumerName, consumerCreationTimpstamp, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:${StreamType}/${StreamName}/consumer/${ConsumerName}:${ConsumerCreationTimpstamp}';
        arn = arn.replace('${StreamType}', streamType);
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${ConsumerName}', consumerName);
        arn = arn.replace('${ConsumerCreationTimpstamp}', consumerCreationTimpstamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type kmsKey to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKmsKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kinesis = Kinesis;
//# sourceMappingURL=data:application/json;base64,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