"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesisvideo = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisvideostreams.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesisvideo extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisvideostreams.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesisvideo';
        this.actionList = {
            "ConnectAsMaster": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsMaster.html",
                "description": "Grants permission to connect as a master to the signaling channel specified by the endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "ConnectAsViewer": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsViewer.html",
                "description": "Grants permission to connect as a viewer to the signaling channel specified by the endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "CreateSignalingChannel": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateSignalingChannel.html",
                "description": "Grants permission to create a signaling channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateStream.html",
                "description": "Grants permission to create a Kinesis video stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteSignalingChannel": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteSignalingChannel.html",
                "description": "Grants permission to delete an existing signaling channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteStream.html",
                "description": "Grants permission to delete an existing Kinesis video stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeSignalingChannel": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeSignalingChannel.html",
                "description": "Grants permission to describe the specified signaling channel",
                "accessLevel": "List",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html",
                "description": "Grants permission to describe the specified Kinesis video stream",
                "accessLevel": "List",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetClip": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetClip.html",
                "description": "Grants permission to get a media clip from a video stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetDASHStreamingSessionURL": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDASHStreamingSessionURL.html",
                "description": "Grants permission to create a URL for MPEG-DASH video streaming",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetDataEndpoint": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html",
                "description": "Grants permission to get an endpoint for a specified stream for either reading or writing media data to Kinesis Video Streams",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetHLSStreamingSessionURL": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetHLSStreamingSessionURL.html",
                "description": "Grants permission to create a URL for HLS video streaming",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetIceServerConfig": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetIceServerConfig.html",
                "description": "Grants permission to get the ICE server configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetMedia": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMedia.html",
                "description": "Grants permission to return media content of a Kinesis video stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetMediaForFragmentList": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMediaForFragmentList.html",
                "description": "Grants permission to read and return media data only from persisted storage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetSignalingChannelEndpoint": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetSignalingChannelEndpoint.html",
                "description": "Grants permission to get endpoints for a specified combination of protocol and role for a signaling channel",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "ListFragments": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListFragments.html",
                "description": "Grants permission to list the fragments from archival storage based on the pagination token or selector type with range specified",
                "accessLevel": "List",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "ListSignalingChannels": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListSignalingChannels.html",
                "description": "Grants permission to list your signaling channels",
                "accessLevel": "List"
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListStreams.html",
                "description": "Grants permission to list your Kinesis video streams",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForResource.html",
                "description": "Grants permission to fetch the tags associated with your resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    }
                }
            },
            "ListTagsForStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForStream.html",
                "description": "Grants permission to fetch the tags associated with Kinesis video stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "PutMedia": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_PutMedia.html",
                "description": "Grants permission to send media data to a Kinesis video stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "SendAlexaOfferToMaster": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_SendAlexaOfferToMaster.html",
                "description": "Grants permission to send the Alexa SDP offer to the master",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagResource.html",
                "description": "Grants permission to attach set of tags to your resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TagStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagStream.html",
                "description": "Grants permission to attach set of tags to your Kinesis video streams",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags from your resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "channel": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UntagStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagStream.html",
                "description": "Grants permission to remove one or more tags from your Kinesis video streams",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDataRetention": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateDataRetention.html",
                "description": "Grants permission to update the data retention period of your Kinesis video stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateSignalingChannel": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateSignalingChannel.html",
                "description": "Grants permission to update an existing signaling channel",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "UpdateStream": {
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateStream.html",
                "description": "Grants permission to update an existing Kinesis video stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html",
                "arn": "arn:${Partition}:kinesisvideo:${Region}:${Account}:stream/${StreamName}/${CreationTime}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/kinesisvideostreams-webrtc-dg/latest/devguide/kvswebrtc-how-it-works.html",
                "arn": "arn:${Partition}:kinesisvideo:${Region}:${Account}:channel/${ChannelName}/${CreationTime}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to connect as a master to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsMaster.html
     */
    toConnectAsMaster() {
        this.add('kinesisvideo:ConnectAsMaster');
        return this;
    }
    /**
     * Grants permission to connect as a viewer to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsViewer.html
     */
    toConnectAsViewer() {
        this.add('kinesisvideo:ConnectAsViewer');
        return this;
    }
    /**
     * Grants permission to create a signaling channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateSignalingChannel.html
     */
    toCreateSignalingChannel() {
        this.add('kinesisvideo:CreateSignalingChannel');
        return this;
    }
    /**
     * Grants permission to create a Kinesis video stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateStream.html
     */
    toCreateStream() {
        this.add('kinesisvideo:CreateStream');
        return this;
    }
    /**
     * Grants permission to delete an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteSignalingChannel.html
     */
    toDeleteSignalingChannel() {
        this.add('kinesisvideo:DeleteSignalingChannel');
        return this;
    }
    /**
     * Grants permission to delete an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteStream.html
     */
    toDeleteStream() {
        this.add('kinesisvideo:DeleteStream');
        return this;
    }
    /**
     * Grants permission to describe the specified signaling channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeSignalingChannel.html
     */
    toDescribeSignalingChannel() {
        this.add('kinesisvideo:DescribeSignalingChannel');
        return this;
    }
    /**
     * Grants permission to describe the specified Kinesis video stream
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html
     */
    toDescribeStream() {
        this.add('kinesisvideo:DescribeStream');
        return this;
    }
    /**
     * Grants permission to get a media clip from a video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetClip.html
     */
    toGetClip() {
        this.add('kinesisvideo:GetClip');
        return this;
    }
    /**
     * Grants permission to create a URL for MPEG-DASH video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDASHStreamingSessionURL.html
     */
    toGetDASHStreamingSessionURL() {
        this.add('kinesisvideo:GetDASHStreamingSessionURL');
        return this;
    }
    /**
     * Grants permission to get an endpoint for a specified stream for either reading or writing media data to Kinesis Video Streams
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html
     */
    toGetDataEndpoint() {
        this.add('kinesisvideo:GetDataEndpoint');
        return this;
    }
    /**
     * Grants permission to create a URL for HLS video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetHLSStreamingSessionURL.html
     */
    toGetHLSStreamingSessionURL() {
        this.add('kinesisvideo:GetHLSStreamingSessionURL');
        return this;
    }
    /**
     * Grants permission to get the ICE server configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetIceServerConfig.html
     */
    toGetIceServerConfig() {
        this.add('kinesisvideo:GetIceServerConfig');
        return this;
    }
    /**
     * Grants permission to return media content of a Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMedia.html
     */
    toGetMedia() {
        this.add('kinesisvideo:GetMedia');
        return this;
    }
    /**
     * Grants permission to read and return media data only from persisted storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMediaForFragmentList.html
     */
    toGetMediaForFragmentList() {
        this.add('kinesisvideo:GetMediaForFragmentList');
        return this;
    }
    /**
     * Grants permission to get endpoints for a specified combination of protocol and role for a signaling channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetSignalingChannelEndpoint.html
     */
    toGetSignalingChannelEndpoint() {
        this.add('kinesisvideo:GetSignalingChannelEndpoint');
        return this;
    }
    /**
     * Grants permission to list the fragments from archival storage based on the pagination token or selector type with range specified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListFragments.html
     */
    toListFragments() {
        this.add('kinesisvideo:ListFragments');
        return this;
    }
    /**
     * Grants permission to list your signaling channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListSignalingChannels.html
     */
    toListSignalingChannels() {
        this.add('kinesisvideo:ListSignalingChannels');
        return this;
    }
    /**
     * Grants permission to list your Kinesis video streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListStreams.html
     */
    toListStreams() {
        this.add('kinesisvideo:ListStreams');
        return this;
    }
    /**
     * Grants permission to fetch the tags associated with your resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('kinesisvideo:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to fetch the tags associated with Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        this.add('kinesisvideo:ListTagsForStream');
        return this;
    }
    /**
     * Grants permission to send media data to a Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_PutMedia.html
     */
    toPutMedia() {
        this.add('kinesisvideo:PutMedia');
        return this;
    }
    /**
     * Grants permission to send the Alexa SDP offer to the master
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_SendAlexaOfferToMaster.html
     */
    toSendAlexaOfferToMaster() {
        this.add('kinesisvideo:SendAlexaOfferToMaster');
        return this;
    }
    /**
     * Grants permission to attach set of tags to your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.add('kinesisvideo:TagResource');
        return this;
    }
    /**
     * Grants permission to attach set of tags to your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagStream.html
     */
    toTagStream() {
        this.add('kinesisvideo:TagStream');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.add('kinesisvideo:UntagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagStream.html
     */
    toUntagStream() {
        this.add('kinesisvideo:UntagStream');
        return this;
    }
    /**
     * Grants permission to update the data retention period of your Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateDataRetention.html
     */
    toUpdateDataRetention() {
        this.add('kinesisvideo:UpdateDataRetention');
        return this;
    }
    /**
     * Grants permission to update an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateSignalingChannel.html
     */
    toUpdateSignalingChannel() {
        this.add('kinesisvideo:UpdateSignalingChannel');
        return this;
    }
    /**
     * Grants permission to update an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateStream.html
     */
    toUpdateStream() {
        this.add('kinesisvideo:UpdateStream');
        return this;
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html
     *
     * @param streamName - Identifier for the streamName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamName, creationTime, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisvideo:${Region}:${Account}:stream/${StreamName}/${CreationTime}';
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${CreationTime}', creationTime);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/kinesisvideostreams-webrtc-dg/latest/devguide/kvswebrtc-how-it-works.html
     *
     * @param channelName - Identifier for the channelName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelName, creationTime, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisvideo:${Region}:${Account}:channel/${ChannelName}/${CreationTime}';
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${CreationTime}', creationTime);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters requests based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSignalingChannel()
     * - .toCreateStream()
     * - .toTagResource()
     * - .toTagStream()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the stream.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - stream
     * - channel
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters requests based on the presence of mandatory tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSignalingChannel()
     * - .toCreateStream()
     * - .toTagResource()
     * - .toTagStream()
     * - .toUntagResource()
     * - .toUntagStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Kinesisvideo = Kinesisvideo;
//# sourceMappingURL=data:application/json;base64,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