"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lightsail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lightsail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lightsail';
        this.actionList = {
            "AllocateStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html",
                "description": "Creates a static IP address that can be attached to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "AttachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html",
                "description": "Attaches a disk to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    }
                }
            },
            "AttachInstancesToLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html",
                "description": "Attaches one or more instances to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html",
                "description": "Attaches a TLS certificate to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html",
                "description": "Attaches a static IP address to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "CloseInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html",
                "description": "Closes a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html",
                "description": "Copies a snapshot from one AWS Region to another in Amazon Lightsail.",
                "accessLevel": "Write"
            },
            "CreateCloudFormationStack": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html",
                "description": "Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "CreateDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html",
                "description": "Creates a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html",
                "description": "Creates a disk from snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html",
                "description": "Creates a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html",
                "description": "Creates a domain resource for the specified domain name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html",
                "description": "Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "CreateInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html",
                "description": "Creates an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html",
                "description": "Creates one or more instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstancesFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html",
                "description": "Creates one or more instances based on an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html",
                "description": "Creates a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html",
                "description": "Creates a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html",
                "description": "Creates a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "CreateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html",
                "description": "Creates a new relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html",
                "description": "Creates a new relational database from a snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html",
                "description": "Creates a relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html",
                "description": "Deletes a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html",
                "description": "Deletes a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html",
                "description": "Deletes a domain resource and all of its DNS records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html",
                "description": "Deletes a DNS record entry for a domain resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html",
                "description": "Deletes an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html",
                "description": "Deletes an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html",
                "description": "Deletes a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "DeleteKnownHostKeys": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html",
                "description": "Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html",
                "description": "Deletes a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Deletes a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html",
                "description": "Deletes a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html",
                "description": "Deletes relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                }
            },
            "DetachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html",
                "description": "Detaches a disk from an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DetachInstancesFromLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Detaches one or more instances from a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DetachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html",
                "description": "Detaches a static IP from an instance to which it is attached.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "DownloadDefaultKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html",
                "description": "Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "ExportSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html",
                "description": "Exports an Amazon Lightsail snapshot to Amazon EC2.",
                "accessLevel": "Write"
            },
            "GetActiveNames": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html",
                "description": "Returns the names of all active (not deleted) resources.",
                "accessLevel": "Read"
            },
            "GetBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html",
                "description": "Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html",
                "description": "Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetCloudFormationStackRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html",
                "description": "Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "CloudFormationStackRecord": {
                        "required": true
                    }
                }
            },
            "GetDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html",
                "description": "Returns information about a disk.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html",
                "description": "Returns information about a disk snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html",
                "description": "Returns information about all disk snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDisks": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html",
                "description": "Returns information about all disks.",
                "accessLevel": "List"
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html",
                "description": "Returns DNS records for a domain resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetDomains": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html",
                "description": "Returns DNS records for all domain resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetExportSnapshotRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html",
                "description": "Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.",
                "accessLevel": "List",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html",
                "description": "Returns information about an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceAccessDetails": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html",
                "description": "Returns temporary keys you can use to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html",
                "description": "Returns the data points for the specified metric of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstancePortStates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html",
                "description": "Returns the port states of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html",
                "description": "Returns information about an instance snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html",
                "description": "Returns information about all instance snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceState": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html",
                "description": "Returns the state of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html",
                "description": "Returns information about all instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html",
                "description": "Returns information about a key pair.",
                "accessLevel": "List",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetKeyPairs": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html",
                "description": "Returns information about all key pairs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html",
                "description": "Returns information about a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html",
                "description": "Returns the data points for the specified metric of a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerTlsCertificates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Returns information about a load balancer TLS certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancers": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html",
                "description": "Returns information about load balancers.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html",
                "description": "Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperations": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html",
                "description": "Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperationsForResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html",
                "description": "Returns operations for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                }
            },
            "GetRegions": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html",
                "description": "Returns a list of all valid AWS Regions for Amazon Lightsail.",
                "accessLevel": "List"
            },
            "GetRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html",
                "description": "Returns information about a relational database.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html",
                "description": "Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html",
                "description": "Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html",
                "description": "Returns events for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html",
                "description": "Returns events for the specified log stream of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogStreams": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html",
                "description": "Returns the log streams available for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseMasterUserPassword": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html",
                "description": "Returns the master user password of a relational database.",
                "accessLevel": "Write"
            },
            "GetRelationalDatabaseMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html",
                "description": "Returns the data points for the specified metric of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html",
                "description": "Returns the parameters of a relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html",
                "description": "Returns information about a relational database snapshot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html",
                "description": "Returns information about all relational database snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabases": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html",
                "description": "Return information about all relational databases.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html",
                "description": "Returns information about a static IP.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "GetStaticIps": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html",
                "description": "Returns information about all static IPs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html",
                "description": "Imports a public key from a key pair.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "IsVpcPeered": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html",
                "description": "Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.",
                "accessLevel": "Read"
            },
            "OpenInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html",
                "description": "Adds, or opens a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "PeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html",
                "description": "Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.",
                "accessLevel": "Write"
            },
            "PutInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html",
                "description": "Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html",
                "description": "Reboots an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html",
                "description": "Reboots a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "ReleaseStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html",
                "description": "Deletes a static IP.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html",
                "description": "Starts an instance that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StartRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html",
                "description": "Starts a relational database that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html",
                "description": "Stops an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StopRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html",
                "description": "Stops a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnpeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html",
                "description": "Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html",
                "description": "Updates a domain recordset after it is created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "UpdateLoadBalancerAttribute": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Updates a load balancer attribute, such as the health check path and session stickiness.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html",
                "description": "Updates a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html",
                "description": "Updates the parameters of a relational database.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Domain": {
                "name": "Domain",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Instance": {
                "name": "Instance",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "InstanceSnapshot": {
                "name": "InstanceSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "KeyPair": {
                "name": "KeyPair",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StaticIp": {
                "name": "StaticIp",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Disk": {
                "name": "Disk",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DiskSnapshot": {
                "name": "DiskSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LoadBalancer": {
                "name": "LoadBalancer",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PeeredVpc": {
                "name": "PeeredVpc",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}",
                "conditionKeys": []
            },
            "LoadBalancerTlsCertificate": {
                "name": "LoadBalancerTlsCertificate",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}",
                "conditionKeys": []
            },
            "ExportSnapshotRecord": {
                "name": "ExportSnapshotRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}",
                "conditionKeys": []
            },
            "CloudFormationStackRecord": {
                "name": "CloudFormationStackRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}",
                "conditionKeys": []
            },
            "RelationalDatabase": {
                "name": "RelationalDatabase",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "RelationalDatabaseSnapshot": {
                "name": "RelationalDatabaseSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a static IP address that can be attached to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html
     */
    toAllocateStaticIp() {
        this.add('lightsail:AllocateStaticIp');
        return this;
    }
    /**
     * Attaches a disk to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html
     */
    toAttachDisk() {
        this.add('lightsail:AttachDisk');
        return this;
    }
    /**
     * Attaches one or more instances to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html
     */
    toAttachInstancesToLoadBalancer() {
        this.add('lightsail:AttachInstancesToLoadBalancer');
        return this;
    }
    /**
     * Attaches a TLS certificate to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html
     */
    toAttachLoadBalancerTlsCertificate() {
        this.add('lightsail:AttachLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Attaches a static IP address to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html
     */
    toAttachStaticIp() {
        this.add('lightsail:AttachStaticIp');
        return this;
    }
    /**
     * Closes a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html
     */
    toCloseInstancePublicPorts() {
        this.add('lightsail:CloseInstancePublicPorts');
        return this;
    }
    /**
     * Copies a snapshot from one AWS Region to another in Amazon Lightsail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.add('lightsail:CopySnapshot');
        return this;
    }
    /**
     * Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html
     */
    toCreateCloudFormationStack() {
        this.add('lightsail:CreateCloudFormationStack');
        return this;
    }
    /**
     * Creates a disk.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html
     */
    toCreateDisk() {
        this.add('lightsail:CreateDisk');
        return this;
    }
    /**
     * Creates a disk from snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html
     */
    toCreateDiskFromSnapshot() {
        this.add('lightsail:CreateDiskFromSnapshot');
        return this;
    }
    /**
     * Creates a disk snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html
     */
    toCreateDiskSnapshot() {
        this.add('lightsail:CreateDiskSnapshot');
        return this;
    }
    /**
     * Creates a domain resource for the specified domain name.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html
     */
    toCreateDomain() {
        this.add('lightsail:CreateDomain');
        return this;
    }
    /**
     * Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html
     */
    toCreateDomainEntry() {
        this.add('lightsail:CreateDomainEntry');
        return this;
    }
    /**
     * Creates an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html
     */
    toCreateInstanceSnapshot() {
        this.add('lightsail:CreateInstanceSnapshot');
        return this;
    }
    /**
     * Creates one or more instances.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html
     */
    toCreateInstances() {
        this.add('lightsail:CreateInstances');
        return this;
    }
    /**
     * Creates one or more instances based on an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html
     */
    toCreateInstancesFromSnapshot() {
        this.add('lightsail:CreateInstancesFromSnapshot');
        return this;
    }
    /**
     * Creates a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html
     */
    toCreateKeyPair() {
        this.add('lightsail:CreateKeyPair');
        return this;
    }
    /**
     * Creates a load balancer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html
     */
    toCreateLoadBalancer() {
        this.add('lightsail:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html
     */
    toCreateLoadBalancerTlsCertificate() {
        this.add('lightsail:CreateLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Creates a new relational database.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html
     */
    toCreateRelationalDatabase() {
        this.add('lightsail:CreateRelationalDatabase');
        return this;
    }
    /**
     * Creates a new relational database from a snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html
     */
    toCreateRelationalDatabaseFromSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseFromSnapshot');
        return this;
    }
    /**
     * Creates a relational database snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html
     */
    toCreateRelationalDatabaseSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Deletes a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html
     */
    toDeleteDisk() {
        this.add('lightsail:DeleteDisk');
        return this;
    }
    /**
     * Deletes a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html
     */
    toDeleteDiskSnapshot() {
        this.add('lightsail:DeleteDiskSnapshot');
        return this;
    }
    /**
     * Deletes a domain resource and all of its DNS records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.add('lightsail:DeleteDomain');
        return this;
    }
    /**
     * Deletes a DNS record entry for a domain resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html
     */
    toDeleteDomainEntry() {
        this.add('lightsail:DeleteDomainEntry');
        return this;
    }
    /**
     * Deletes an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        this.add('lightsail:DeleteInstance');
        return this;
    }
    /**
     * Deletes an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html
     */
    toDeleteInstanceSnapshot() {
        this.add('lightsail:DeleteInstanceSnapshot');
        return this;
    }
    /**
     * Deletes a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html
     */
    toDeleteKeyPair() {
        this.add('lightsail:DeleteKeyPair');
        return this;
    }
    /**
     * Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html
     */
    toDeleteKnownHostKeys() {
        this.add('lightsail:DeleteKnownHostKeys');
        return this;
    }
    /**
     * Deletes a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html
     */
    toDeleteLoadBalancer() {
        this.add('lightsail:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toDeleteLoadBalancerTlsCertificate() {
        this.add('lightsail:DeleteLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Deletes a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html
     */
    toDeleteRelationalDatabase() {
        this.add('lightsail:DeleteRelationalDatabase');
        return this;
    }
    /**
     * Deletes relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html
     */
    toDeleteRelationalDatabaseSnapshot() {
        this.add('lightsail:DeleteRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Detaches a disk from an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html
     */
    toDetachDisk() {
        this.add('lightsail:DetachDisk');
        return this;
    }
    /**
     * Detaches one or more instances from a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toDetachInstancesFromLoadBalancer() {
        this.add('lightsail:DetachInstancesFromLoadBalancer');
        return this;
    }
    /**
     * Detaches a static IP from an instance to which it is attached.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html
     */
    toDetachStaticIp() {
        this.add('lightsail:DetachStaticIp');
        return this;
    }
    /**
     * Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html
     */
    toDownloadDefaultKeyPair() {
        this.add('lightsail:DownloadDefaultKeyPair');
        return this;
    }
    /**
     * Exports an Amazon Lightsail snapshot to Amazon EC2.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html
     */
    toExportSnapshot() {
        this.add('lightsail:ExportSnapshot');
        return this;
    }
    /**
     * Returns the names of all active (not deleted) resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html
     */
    toGetActiveNames() {
        this.add('lightsail:GetActiveNames');
        return this;
    }
    /**
     * Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html
     */
    toGetBlueprints() {
        this.add('lightsail:GetBlueprints');
        return this;
    }
    /**
     * Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html
     */
    toGetBundles() {
        this.add('lightsail:GetBundles');
        return this;
    }
    /**
     * Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html
     */
    toGetCloudFormationStackRecords() {
        this.add('lightsail:GetCloudFormationStackRecords');
        return this;
    }
    /**
     * Returns information about a disk.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html
     */
    toGetDisk() {
        this.add('lightsail:GetDisk');
        return this;
    }
    /**
     * Returns information about a disk snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html
     */
    toGetDiskSnapshot() {
        this.add('lightsail:GetDiskSnapshot');
        return this;
    }
    /**
     * Returns information about all disk snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html
     */
    toGetDiskSnapshots() {
        this.add('lightsail:GetDiskSnapshots');
        return this;
    }
    /**
     * Returns information about all disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html
     */
    toGetDisks() {
        this.add('lightsail:GetDisks');
        return this;
    }
    /**
     * Returns DNS records for a domain resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html
     */
    toGetDomain() {
        this.add('lightsail:GetDomain');
        return this;
    }
    /**
     * Returns DNS records for all domain resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html
     */
    toGetDomains() {
        this.add('lightsail:GetDomains');
        return this;
    }
    /**
     * Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html
     */
    toGetExportSnapshotRecords() {
        this.add('lightsail:GetExportSnapshotRecords');
        return this;
    }
    /**
     * Returns information about an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html
     */
    toGetInstance() {
        this.add('lightsail:GetInstance');
        return this;
    }
    /**
     * Returns temporary keys you can use to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html
     */
    toGetInstanceAccessDetails() {
        this.add('lightsail:GetInstanceAccessDetails');
        return this;
    }
    /**
     * Returns the data points for the specified metric of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html
     */
    toGetInstanceMetricData() {
        this.add('lightsail:GetInstanceMetricData');
        return this;
    }
    /**
     * Returns the port states of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html
     */
    toGetInstancePortStates() {
        this.add('lightsail:GetInstancePortStates');
        return this;
    }
    /**
     * Returns information about an instance snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html
     */
    toGetInstanceSnapshot() {
        this.add('lightsail:GetInstanceSnapshot');
        return this;
    }
    /**
     * Returns information about all instance snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html
     */
    toGetInstanceSnapshots() {
        this.add('lightsail:GetInstanceSnapshots');
        return this;
    }
    /**
     * Returns the state of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html
     */
    toGetInstanceState() {
        this.add('lightsail:GetInstanceState');
        return this;
    }
    /**
     * Returns information about all instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html
     */
    toGetInstances() {
        this.add('lightsail:GetInstances');
        return this;
    }
    /**
     * Returns information about a key pair.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html
     */
    toGetKeyPair() {
        this.add('lightsail:GetKeyPair');
        return this;
    }
    /**
     * Returns information about all key pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html
     */
    toGetKeyPairs() {
        this.add('lightsail:GetKeyPairs');
        return this;
    }
    /**
     * Returns information about a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html
     */
    toGetLoadBalancer() {
        this.add('lightsail:GetLoadBalancer');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html
     */
    toGetLoadBalancerMetricData() {
        this.add('lightsail:GetLoadBalancerMetricData');
        return this;
    }
    /**
     * Returns information about a load balancer TLS certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toGetLoadBalancerTlsCertificates() {
        this.add('lightsail:GetLoadBalancerTlsCertificates');
        return this;
    }
    /**
     * Returns information about load balancers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html
     */
    toGetLoadBalancers() {
        this.add('lightsail:GetLoadBalancers');
        return this;
    }
    /**
     * Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html
     */
    toGetOperation() {
        this.add('lightsail:GetOperation');
        return this;
    }
    /**
     * Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html
     */
    toGetOperations() {
        this.add('lightsail:GetOperations');
        return this;
    }
    /**
     * Returns operations for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html
     */
    toGetOperationsForResource() {
        this.add('lightsail:GetOperationsForResource');
        return this;
    }
    /**
     * Returns a list of all valid AWS Regions for Amazon Lightsail.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html
     */
    toGetRegions() {
        this.add('lightsail:GetRegions');
        return this;
    }
    /**
     * Returns information about a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html
     */
    toGetRelationalDatabase() {
        this.add('lightsail:GetRelationalDatabase');
        return this;
    }
    /**
     * Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html
     */
    toGetRelationalDatabaseBlueprints() {
        this.add('lightsail:GetRelationalDatabaseBlueprints');
        return this;
    }
    /**
     * Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html
     */
    toGetRelationalDatabaseBundles() {
        this.add('lightsail:GetRelationalDatabaseBundles');
        return this;
    }
    /**
     * Returns events for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html
     */
    toGetRelationalDatabaseEvents() {
        this.add('lightsail:GetRelationalDatabaseEvents');
        return this;
    }
    /**
     * Returns events for the specified log stream of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html
     */
    toGetRelationalDatabaseLogEvents() {
        this.add('lightsail:GetRelationalDatabaseLogEvents');
        return this;
    }
    /**
     * Returns the log streams available for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html
     */
    toGetRelationalDatabaseLogStreams() {
        this.add('lightsail:GetRelationalDatabaseLogStreams');
        return this;
    }
    /**
     * Returns the master user password of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html
     */
    toGetRelationalDatabaseMasterUserPassword() {
        this.add('lightsail:GetRelationalDatabaseMasterUserPassword');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html
     */
    toGetRelationalDatabaseMetricData() {
        this.add('lightsail:GetRelationalDatabaseMetricData');
        return this;
    }
    /**
     * Returns the parameters of a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html
     */
    toGetRelationalDatabaseParameters() {
        this.add('lightsail:GetRelationalDatabaseParameters');
        return this;
    }
    /**
     * Returns information about a relational database snapshot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html
     */
    toGetRelationalDatabaseSnapshot() {
        this.add('lightsail:GetRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Returns information about all relational database snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html
     */
    toGetRelationalDatabaseSnapshots() {
        this.add('lightsail:GetRelationalDatabaseSnapshots');
        return this;
    }
    /**
     * Return information about all relational databases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html
     */
    toGetRelationalDatabases() {
        this.add('lightsail:GetRelationalDatabases');
        return this;
    }
    /**
     * Returns information about a static IP.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html
     */
    toGetStaticIp() {
        this.add('lightsail:GetStaticIp');
        return this;
    }
    /**
     * Returns information about all static IPs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html
     */
    toGetStaticIps() {
        this.add('lightsail:GetStaticIps');
        return this;
    }
    /**
     * Imports a public key from a key pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html
     */
    toImportKeyPair() {
        this.add('lightsail:ImportKeyPair');
        return this;
    }
    /**
     * Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html
     */
    toIsVpcPeered() {
        this.add('lightsail:IsVpcPeered');
        return this;
    }
    /**
     * Adds, or opens a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html
     */
    toOpenInstancePublicPorts() {
        this.add('lightsail:OpenInstancePublicPorts');
        return this;
    }
    /**
     * Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html
     */
    toPeerVpc() {
        this.add('lightsail:PeerVpc');
        return this;
    }
    /**
     * Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html
     */
    toPutInstancePublicPorts() {
        this.add('lightsail:PutInstancePublicPorts');
        return this;
    }
    /**
     * Reboots an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html
     */
    toRebootInstance() {
        this.add('lightsail:RebootInstance');
        return this;
    }
    /**
     * Reboots a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html
     */
    toRebootRelationalDatabase() {
        this.add('lightsail:RebootRelationalDatabase');
        return this;
    }
    /**
     * Deletes a static IP.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html
     */
    toReleaseStaticIp() {
        this.add('lightsail:ReleaseStaticIp');
        return this;
    }
    /**
     * Starts an instance that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html
     */
    toStartInstance() {
        this.add('lightsail:StartInstance');
        return this;
    }
    /**
     * Starts a relational database that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html
     */
    toStartRelationalDatabase() {
        this.add('lightsail:StartRelationalDatabase');
        return this;
    }
    /**
     * Stops an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html
     */
    toStopInstance() {
        this.add('lightsail:StopInstance');
        return this;
    }
    /**
     * Stops a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html
     */
    toStopRelationalDatabase() {
        this.add('lightsail:StopRelationalDatabase');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html
     */
    toTagResource() {
        this.add('lightsail:TagResource');
        return this;
    }
    /**
     * Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html
     */
    toUnpeerVpc() {
        this.add('lightsail:UnpeerVpc');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('lightsail:UntagResource');
        return this;
    }
    /**
     * Updates a domain recordset after it is created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html
     */
    toUpdateDomainEntry() {
        this.add('lightsail:UpdateDomainEntry');
        return this;
    }
    /**
     * Updates a load balancer attribute, such as the health check path and session stickiness.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toUpdateLoadBalancerAttribute() {
        this.add('lightsail:UpdateLoadBalancerAttribute');
        return this;
    }
    /**
     * Updates a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html
     */
    toUpdateRelationalDatabase() {
        this.add('lightsail:UpdateRelationalDatabase');
        return this;
    }
    /**
     * Updates the parameters of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html
     */
    toUpdateRelationalDatabaseParameters() {
        this.add('lightsail:UpdateRelationalDatabaseParameters');
        return this;
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type InstanceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstanceSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type KeyPair to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyPair(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StaticIp to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStaticIp(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Disk to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDisk(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type DiskSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiskSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancer to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoadBalancer(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type PeeredVpc to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPeeredVpc(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancerTlsCertificate to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadBalancerTlsCertificate(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ExportSnapshotRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExportSnapshotRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CloudFormationStackRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCloudFormationStackRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabase to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabase(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabaseSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabaseSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDisk()
     * - .toCreateDiskFromSnapshot()
     * - .toCreateDiskSnapshot()
     * - .toCreateDomain()
     * - .toCreateInstanceSnapshot()
     * - .toCreateInstances()
     * - .toCreateInstancesFromSnapshot()
     * - .toCreateKeyPair()
     * - .toCreateLoadBalancer()
     * - .toCreateRelationalDatabase()
     * - .toCreateRelationalDatabaseFromSnapshot()
     * - .toCreateRelationalDatabaseSnapshot()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Domain
     * - Instance
     * - InstanceSnapshot
     * - KeyPair
     * - StaticIp
     * - Disk
     * - DiskSnapshot
     * - LoadBalancer
     * - RelationalDatabase
     * - RelationalDatabaseSnapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDisk()
     * - .toCreateDiskFromSnapshot()
     * - .toCreateDiskSnapshot()
     * - .toCreateDomain()
     * - .toCreateInstanceSnapshot()
     * - .toCreateInstances()
     * - .toCreateInstancesFromSnapshot()
     * - .toCreateKeyPair()
     * - .toCreateLoadBalancer()
     * - .toCreateRelationalDatabase()
     * - .toCreateRelationalDatabaseFromSnapshot()
     * - .toCreateRelationalDatabaseSnapshot()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Lightsail = Lightsail;
//# sourceMappingURL=data:application/json;base64,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