"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie';
        this.actionList = {
            "AssociateMemberAccount": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateMemberAccount.html",
                "description": "Enables the user to associate a specified AWS account with Amazon Macie as a member account.",
                "accessLevel": "Write"
            },
            "AssociateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateS3Resources.html",
                "description": "Enables the user to associate specified S3 resources with Amazon Macie for monitoring and data classification.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            },
            "DisassociateMemberAccount": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateMemberAccount.html",
                "description": "Enables the user to remove the specified member account from Amazon Macie.",
                "accessLevel": "Write"
            },
            "DisassociateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateS3Resources.html",
                "description": "Enables the user to remove specified S3 resources from being monitored by Amazon Macie.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            },
            "ListMemberAccounts": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListMemberAccounts.html",
                "description": "Enables the user to list all Amazon Macie member accounts for the current Macie master account.",
                "accessLevel": "List"
            },
            "ListS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListS3Resources.html",
                "description": "Enables the user to list all the S3 resources associated with Amazon Macie.",
                "accessLevel": "List"
            },
            "UpdateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_UpdateS3Resources.html",
                "description": "Enables the user to update the classification types for the specified S3 resources.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Enables the user to associate a specified AWS account with Amazon Macie as a member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateMemberAccount.html
     */
    toAssociateMemberAccount() {
        this.add('macie:AssociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to associate specified S3 resources with Amazon Macie for monitoring and data classification.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateS3Resources.html
     */
    toAssociateS3Resources() {
        this.add('macie:AssociateS3Resources');
        return this;
    }
    /**
     * Enables the user to remove the specified member account from Amazon Macie.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateMemberAccount.html
     */
    toDisassociateMemberAccount() {
        this.add('macie:DisassociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to remove specified S3 resources from being monitored by Amazon Macie.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateS3Resources.html
     */
    toDisassociateS3Resources() {
        this.add('macie:DisassociateS3Resources');
        return this;
    }
    /**
     * Enables the user to list all Amazon Macie member accounts for the current Macie master account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListMemberAccounts.html
     */
    toListMemberAccounts() {
        this.add('macie:ListMemberAccounts');
        return this;
    }
    /**
     * Enables the user to list all the S3 resources associated with Amazon Macie.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListS3Resources.html
     */
    toListS3Resources() {
        this.add('macie:ListS3Resources');
        return this;
    }
    /**
     * Enables the user to update the classification types for the specified S3 resources.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_UpdateS3Resources.html
     */
    toUpdateS3Resources() {
        this.add('macie:UpdateS3Resources');
        return this;
    }
    /**
     * Allow access to the specified actions only when the request operates on the specified aws resource
     *
     * Applies to actions:
     * - .toAssociateS3Resources()
     * - .toDisassociateS3Resources()
     * - .toUpdateS3Resources()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAwsSourceArn(value, operator) {
        return this.if(`aws:SourceArn`, value, operator || 'ArnEquals');
    }
}
exports.Macie = Macie;
//# sourceMappingURL=data:application/json;base64,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