"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Managedblockchain = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Managedblockchain extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'managedblockchain';
        this.actionList = {
            "CreateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html",
                "description": "Grants permission to create a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "CreateNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html",
                "description": "Grants permission to create an Amazon Managed Blockchain network.",
                "accessLevel": "Write"
            },
            "CreateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html",
                "description": "Grants permission to create a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "CreateProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html",
                "description": "Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "DeleteMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html",
                "description": "Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "DeleteNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html",
                "description": "Grants permission to delete a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html",
                "description": "Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "GetNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html",
                "description": "Grants permission to return detailed information about an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "GetNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html",
                "description": "Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html",
                "description": "Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "ListNetworks": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html",
                "description": "Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html",
                "description": "Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "ListProposalVotes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html",
                "description": "Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListProposals": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html",
                "description": "Grants permission to list proposals for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "RejectInvitation": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html",
                "description": "Grants permission to reject the invitation to join the blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "invitation": {
                        "required": true
                    }
                }
            },
            "UpdateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html",
                "description": "Grants permission to update a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "UpdateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html",
                "description": "Grants permission to update a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "VoteOnProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html",
                "description": "Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "network": {
                "name": "network",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}",
                "conditionKeys": []
            },
            "member": {
                "name": "member",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}",
                "conditionKeys": []
            },
            "node": {
                "name": "node",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}",
                "conditionKeys": []
            },
            "proposal": {
                "name": "proposal",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}",
                "conditionKeys": []
            },
            "invitation": {
                "name": "invitation",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html
     */
    toCreateMember() {
        this.add('managedblockchain:CreateMember');
        return this;
    }
    /**
     * Grants permission to create an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html
     */
    toCreateNetwork() {
        this.add('managedblockchain:CreateNetwork');
        return this;
    }
    /**
     * Grants permission to create a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html
     */
    toCreateNode() {
        this.add('managedblockchain:CreateNode');
        return this;
    }
    /**
     * Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html
     */
    toCreateProposal() {
        this.add('managedblockchain:CreateProposal');
        return this;
    }
    /**
     * Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html
     */
    toDeleteMember() {
        this.add('managedblockchain:DeleteMember');
        return this;
    }
    /**
     * Grants permission to delete a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html
     */
    toDeleteNode() {
        this.add('managedblockchain:DeleteNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html
     */
    toGetMember() {
        this.add('managedblockchain:GetMember');
        return this;
    }
    /**
     * Grants permission to return detailed information about an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html
     */
    toGetNetwork() {
        this.add('managedblockchain:GetNetwork');
        return this;
    }
    /**
     * Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html
     */
    toGetNode() {
        this.add('managedblockchain:GetNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html
     */
    toGetProposal() {
        this.add('managedblockchain:GetProposal');
        return this;
    }
    /**
     * Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.add('managedblockchain:ListInvitations');
        return this;
    }
    /**
     * Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.add('managedblockchain:ListMembers');
        return this;
    }
    /**
     * Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html
     */
    toListNetworks() {
        this.add('managedblockchain:ListNetworks');
        return this;
    }
    /**
     * Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html
     */
    toListNodes() {
        this.add('managedblockchain:ListNodes');
        return this;
    }
    /**
     * Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html
     */
    toListProposalVotes() {
        this.add('managedblockchain:ListProposalVotes');
        return this;
    }
    /**
     * Grants permission to list proposals for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html
     */
    toListProposals() {
        this.add('managedblockchain:ListProposals');
        return this;
    }
    /**
     * Grants permission to reject the invitation to join the blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html
     */
    toRejectInvitation() {
        this.add('managedblockchain:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to update a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html
     */
    toUpdateMember() {
        this.add('managedblockchain:UpdateMember');
        return this;
    }
    /**
     * Grants permission to update a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html
     */
    toUpdateNode() {
        this.add('managedblockchain:UpdateNode');
        return this;
    }
    /**
     * Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html
     */
    toVoteOnProposal() {
        this.add('managedblockchain:VoteOnProposal');
        return this;
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html
     *
     * @param networkId - Identifier for the networkId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNetwork(networkId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}';
        arn = arn.replace('${NetworkId}', networkId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type member to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html
     *
     * @param memberId - Identifier for the memberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMember(memberId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}';
        arn = arn.replace('${MemberId}', memberId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html
     *
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNode(nodeId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}';
        arn = arn.replace('${NodeId}', nodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proposal to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html
     *
     * @param proposalId - Identifier for the proposalId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProposal(proposalId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}';
        arn = arn.replace('${ProposalId}', proposalId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type invitation to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html
     *
     * @param invitationId - Identifier for the invitationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInvitation(invitationId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}';
        arn = arn.replace('${InvitationId}', invitationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Managedblockchain = Managedblockchain;
//# sourceMappingURL=data:application/json;base64,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