"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kafka = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kafka](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforapachekafka.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kafka extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kafka](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforapachekafka.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kafka';
        this.actionList = {
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster",
                "description": "Grants permission to create a cluster.",
                "accessLevel": "Write",
                "dependentActions": [
                    "ec2:DescribeSecurityGroups",
                    "ec2:DescribeSubnets",
                    "ec2:DescribeVpcs",
                    "iam:AttachRolePolicy",
                    "iam:CreateServiceLinkedRole",
                    "iam:PutRolePolicy",
                    "kms:CreateGrant",
                    "kms:DescribeKey"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to create a configuration.",
                "accessLevel": "Write"
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster",
                "description": "Grants permission to delete a cluster.",
                "accessLevel": "Write"
            },
            "DeleteConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#DeleteConfiguration",
                "description": "Grants permission to delete the specified MSK configuration.",
                "accessLevel": "Write"
            },
            "DescribeCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster",
                "description": "Grants permission to describe a cluster.",
                "accessLevel": "Read"
            },
            "DescribeClusterOperation": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation",
                "description": "Returns a description of the cluster operation specified by the ARN.",
                "accessLevel": "Read"
            },
            "DescribeConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration",
                "description": "Grants permission to describe a configuration.",
                "accessLevel": "Read"
            },
            "DescribeConfigurationRevision": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision",
                "description": "Grants permission to describe a configuration revision.",
                "accessLevel": "Read"
            },
            "GetBootstrapBrokers": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers",
                "description": "Grants permission to get connection details for the broker nodes in a cluster.",
                "accessLevel": "Read"
            },
            "GetCompatibleKafkaVersions": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions",
                "description": "Returns a list of the Apache Kafka versions to which you can update this cluster.",
                "accessLevel": "List"
            },
            "ListClusterOperations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations",
                "description": "Returns a list of all the operations that have been performed on the specified MSK cluster.",
                "accessLevel": "Read"
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters",
                "description": "Grants permission to return a list of all clusters in the current account.",
                "accessLevel": "List"
            },
            "ListConfigurations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to return a list of all configurations in the current account.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes",
                "description": "Grants permission to return a list of nodes in a cluster.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource",
                "description": "Grants permission to list tags of a MSK resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource",
                "description": "Grants permission to tag a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource",
                "description": "Grants permission to remove tags from a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBrokerCount": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount",
                "description": "Updates the number of broker nodes of the cluster.",
                "accessLevel": "Write"
            },
            "UpdateBrokerStorage": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage",
                "description": "Updates the storage size of the broker nodes of the cluster",
                "accessLevel": "Write"
            },
            "UpdateClusterConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration",
                "description": "Update Kafka configuration running on a cluster.",
                "accessLevel": "Write"
            },
            "UpdateClusterKafkaVersion": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion",
                "description": "Updates the cluster to the specified Apache Kafka version.",
                "accessLevel": "Write"
            },
            "UpdateConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#updateconfiguration",
                "description": "Grants permission to create a new revision of the configuration.",
                "accessLevel": "Write"
            },
            "UpdateMonitoring": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring",
                "description": "Updates the monitoring settings for the cluster.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "",
                "arn": "arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a cluster.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster
     */
    toCreateCluster() {
        this.add('kafka:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create a configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    toCreateConfiguration() {
        this.add('kafka:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster
     */
    toDeleteCluster() {
        this.add('kafka:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to delete the specified MSK configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#DeleteConfiguration
     */
    toDeleteConfiguration() {
        this.add('kafka:DeleteConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster
     */
    toDescribeCluster() {
        this.add('kafka:DescribeCluster');
        return this;
    }
    /**
     * Returns a description of the cluster operation specified by the ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation
     */
    toDescribeClusterOperation() {
        this.add('kafka:DescribeClusterOperation');
        return this;
    }
    /**
     * Grants permission to describe a configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration
     */
    toDescribeConfiguration() {
        this.add('kafka:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a configuration revision.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision
     */
    toDescribeConfigurationRevision() {
        this.add('kafka:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to get connection details for the broker nodes in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers
     */
    toGetBootstrapBrokers() {
        this.add('kafka:GetBootstrapBrokers');
        return this;
    }
    /**
     * Returns a list of the Apache Kafka versions to which you can update this cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions
     */
    toGetCompatibleKafkaVersions() {
        this.add('kafka:GetCompatibleKafkaVersions');
        return this;
    }
    /**
     * Returns a list of all the operations that have been performed on the specified MSK cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations
     */
    toListClusterOperations() {
        this.add('kafka:ListClusterOperations');
        return this;
    }
    /**
     * Grants permission to return a list of all clusters in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters
     */
    toListClusters() {
        this.add('kafka:ListClusters');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    toListConfigurations() {
        this.add('kafka:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of nodes in a cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes
     */
    toListNodes() {
        this.add('kafka:ListNodes');
        return this;
    }
    /**
     * Grants permission to list tags of a MSK resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource
     */
    toListTagsForResource() {
        this.add('kafka:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to tag a MSK resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource
     */
    toTagResource() {
        this.add('kafka:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MSK resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource
     */
    toUntagResource() {
        this.add('kafka:UntagResource');
        return this;
    }
    /**
     * Updates the number of broker nodes of the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount
     */
    toUpdateBrokerCount() {
        this.add('kafka:UpdateBrokerCount');
        return this;
    }
    /**
     * Updates the storage size of the broker nodes of the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage
     */
    toUpdateBrokerStorage() {
        this.add('kafka:UpdateBrokerStorage');
        return this;
    }
    /**
     * Update Kafka configuration running on a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration
     */
    toUpdateClusterConfiguration() {
        this.add('kafka:UpdateClusterConfiguration');
        return this;
    }
    /**
     * Updates the cluster to the specified Apache Kafka version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion
     */
    toUpdateClusterKafkaVersion() {
        this.add('kafka:UpdateClusterKafkaVersion');
        return this;
    }
    /**
     * Grants permission to create a new revision of the configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#updateconfiguration
     */
    toUpdateConfiguration() {
        this.add('kafka:UpdateConfiguration');
        return this;
    }
    /**
     * Updates the monitoring settings for the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring
     */
    toUpdateMonitoring() {
        this.add('kafka:UpdateMonitoring');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filter requests based on the allowed set of values for each of the tags
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filter actions based on tag-value associated with a MSK resource.
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filter requests based on the presence of mandatory tag keys in the request
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Kafka = Kafka;
//# sourceMappingURL=data:application/json;base64,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