"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobiletargeting = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpoint.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobiletargeting extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpoint.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mobiletargeting';
        this.actionList = {
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Create an app.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods",
                "description": "Create a campaign for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Create an email template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateExportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Create an export job that exports endpoint definitions to Amazon S3.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateImportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Import endpoint definitions from to create a segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Create a Journey for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreatePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Create a push notification template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration",
                "description": "Create an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write"
            },
            "CreateSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods",
                "description": "Create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Create an sms message template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Create a voice message template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Delete the ADM channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Delete the APNs channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Delete the APNs sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Delete the APNs VoIP channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Delete the APNs VoIP sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Delete a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Delete the Baidu channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Delete a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "DeleteEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Delete the email channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Delete an email template or an email template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Delete an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Delete the event stream for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Delete the GCM channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Delete a specific journey.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                }
            },
            "DeletePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Delete a push notification template or a push notification template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration",
                "description": "Delete an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "DeleteSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Delete a specific segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "DeleteSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Delete the SMS channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Delete an sms message template or an sms message template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteUserEndpoints": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods",
                "description": "Delete all of the endpoints that are associated with a user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Delete the Voice channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Delete a voice message template or a voice message template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Retrieve information about the Amazon Device Messaging (ADM) channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Retrieve information about the APNs channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Retrieve information about the APNs sandbox channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Retrieve information about the APNs VoIP channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Retrieve information about the APNs VoIP sandbox channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Retrieve information about a specific app in your Amazon Pinpoint account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApplicationSettings": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods",
                "description": "Retrieve the default settings for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApps": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apps.html#rest-api-apps-methods",
                "description": "Retrieve a list of apps in your Amazon Pinpoint account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Retrieve information about the Baidu channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Retrieve information about a specific campaign.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignActivities": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-activities.html#rest-api-campaign-activities-methods",
                "description": "Retrieve information about the activities performed by a campaign.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-version.html#rest-api-campaign-version-methods",
                "description": "Retrieve information about a specific campaign version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-versions.html#rest-api-campaign-versions-methods",
                "description": "Retrieve information about the current and prior versions of a campaign.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaigns": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods",
                "description": "Retrieve information about all campaigns for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetChannels": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-channels.html#rest-api-channels-methods",
                "description": "Get all channels information for your app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Obtain information about the email channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Retrieve information about a specific or the active version of an email template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Retrieve information about a specific endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Retrieve information about the event stream for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetExportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Obtain information about a specific export job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetExportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Retrieve a list of all of the export jobs for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Retrieve information about the GCM channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetImportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-job.html#rest-api-import-job-methods",
                "description": "Retrieve information about a specific import job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetImportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Retrieve information about all import jobs for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Retrieve information about a specific journey.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                }
            },
            "GetPushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Retrieve information about a specific or the active version of an push notification template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration",
                "description": "Retrieve information about an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "GetRecommenderConfigurations": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations",
                "description": "Retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account.",
                "accessLevel": "List"
            },
            "GetSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Retrieve information about a specific segment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentExportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Retrieve information about jobs that export endpoint definitions from segments to Amazon S3.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentImportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Retrieve information about jobs that create segments by importing endpoint definitions from .",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-version.html#rest-api-segment-version-methods",
                "description": "Retrieve information about a specific segment version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-versions.html#rest-api-segment-versions-methods",
                "description": "Retrieve information about the current and prior versions of a segment.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegments": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods",
                "description": "Retrieve information about the segments for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Obtain information about the SMS channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Retrieve information about a specific or the active version of an sms message template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetUserEndpoints": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods",
                "description": "Retrieve information about the endpoints that are associated with a user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Obtain information about the Voice channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Retrieve information about a specific or the active version of a voice message template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "ListJourneys": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#apps-application-id-journeys-http-methods",
                "description": "Retrieve information about all journeys for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-get",
                "description": "List tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                }
            },
            "ListTemplateVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods",
                "description": "Retrieve all versions about a specific template.",
                "accessLevel": "List",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#templates-http-methods",
                "description": "Retrieve metadata about the queried templates.",
                "accessLevel": "List",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "PhoneNumberValidate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-phone-number-validate.html#rest-api-phone-number-validate-methods",
                "description": "Obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "PutEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Create or update an event stream for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "PutEvents": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-events.html#rest-api-events-methods",
                "description": "Create or update events for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "RemoveAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Used to remove the attributes for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "SendMessages": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-messages.html#rest-api-messages-methods",
                "description": "Send an SMS message or push notification to specific endpoints.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "SendUsersMessages": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-users-messages.html#rest-api-users-messages-methods",
                "description": "Send an SMS message or push notification to all endpoints that are associated with a specific user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-post",
                "description": "Adds tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-delete",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Update the Amazon Device Messaging (ADM) channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) VoIP channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) VoIP sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApplicationSettings": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods",
                "description": "Update the default settings for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Update the Baidu channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Update a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Update the email channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Update a specific email template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Create an endpoint or update the information for an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointsBatch": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoints.html#rest-api-endpoints-methods",
                "description": "Create or update endpoints as a batch operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Update a specific journey.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateJourneyState": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#apps-application-id-journeys-journey-id-state-http-methods",
                "description": "Update a specific journey state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdatePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Update a specific push notification template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration",
                "description": "Update an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "UpdateSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Update a specific segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Update the SMS channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Update a specific sms message template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateTemplateActiveVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods",
                "description": "Upate the active version parameter of a specific template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "UpdateVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Update the Voice channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Update a specific voice message template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "apps": {
                "name": "apps",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/developerguide/gettingstarted.html#gettingstarted-addapp",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "campaigns": {
                "name": "campaigns",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/campaigns/${CampaignId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "journeys": {
                "name": "journeys",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/journeys/${JourneyId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "segments": {
                "name": "segments",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/segments/${SegmentId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "templates": {
                "name": "templates",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:templates/${TemplateName}/${ChannelType}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "recommenders": {
                "name": "recommenders",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:recommenders/${RecommenderId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toCreateApp() {
        this.add('mobiletargeting:CreateApp');
        return this;
    }
    /**
     * Create a campaign for an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    toCreateCampaign() {
        this.add('mobiletargeting:CreateCampaign');
        return this;
    }
    /**
     * Create an email template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toCreateEmailTemplate() {
        this.add('mobiletargeting:CreateEmailTemplate');
        return this;
    }
    /**
     * Create an export job that exports endpoint definitions to Amazon S3.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toCreateExportJob() {
        this.add('mobiletargeting:CreateExportJob');
        return this;
    }
    /**
     * Import endpoint definitions from to create a segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toCreateImportJob() {
        this.add('mobiletargeting:CreateImportJob');
        return this;
    }
    /**
     * Create a Journey for an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toCreateJourney() {
        this.add('mobiletargeting:CreateJourney');
        return this;
    }
    /**
     * Create a push notification template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toCreatePushTemplate() {
        this.add('mobiletargeting:CreatePushTemplate');
        return this;
    }
    /**
     * Create an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration
     */
    toCreateRecommenderConfiguration() {
        this.add('mobiletargeting:CreateRecommenderConfiguration');
        return this;
    }
    /**
     * Create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    toCreateSegment() {
        this.add('mobiletargeting:CreateSegment');
        return this;
    }
    /**
     * Create an sms message template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toCreateSmsTemplate() {
        this.add('mobiletargeting:CreateSmsTemplate');
        return this;
    }
    /**
     * Create a voice message template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toCreateVoiceTemplate() {
        this.add('mobiletargeting:CreateVoiceTemplate');
        return this;
    }
    /**
     * Delete the ADM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toDeleteAdmChannel() {
        this.add('mobiletargeting:DeleteAdmChannel');
        return this;
    }
    /**
     * Delete the APNs channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toDeleteApnsChannel() {
        this.add('mobiletargeting:DeleteApnsChannel');
        return this;
    }
    /**
     * Delete the APNs sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toDeleteApnsSandboxChannel() {
        this.add('mobiletargeting:DeleteApnsSandboxChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toDeleteApnsVoipChannel() {
        this.add('mobiletargeting:DeleteApnsVoipChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toDeleteApnsVoipSandboxChannel() {
        this.add('mobiletargeting:DeleteApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toDeleteApp() {
        this.add('mobiletargeting:DeleteApp');
        return this;
    }
    /**
     * Delete the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toDeleteBaiduChannel() {
        this.add('mobiletargeting:DeleteBaiduChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toDeleteCampaign() {
        this.add('mobiletargeting:DeleteCampaign');
        return this;
    }
    /**
     * Delete the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toDeleteEmailChannel() {
        this.add('mobiletargeting:DeleteEmailChannel');
        return this;
    }
    /**
     * Delete an email template or an email template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toDeleteEmailTemplate() {
        this.add('mobiletargeting:DeleteEmailTemplate');
        return this;
    }
    /**
     * Delete an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toDeleteEndpoint() {
        this.add('mobiletargeting:DeleteEndpoint');
        return this;
    }
    /**
     * Delete the event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toDeleteEventStream() {
        this.add('mobiletargeting:DeleteEventStream');
        return this;
    }
    /**
     * Delete the GCM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toDeleteGcmChannel() {
        this.add('mobiletargeting:DeleteGcmChannel');
        return this;
    }
    /**
     * Delete a specific journey.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toDeleteJourney() {
        this.add('mobiletargeting:DeleteJourney');
        return this;
    }
    /**
     * Delete a push notification template or a push notification template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toDeletePushTemplate() {
        this.add('mobiletargeting:DeletePushTemplate');
        return this;
    }
    /**
     * Delete an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration
     */
    toDeleteRecommenderConfiguration() {
        this.add('mobiletargeting:DeleteRecommenderConfiguration');
        return this;
    }
    /**
     * Delete a specific segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toDeleteSegment() {
        this.add('mobiletargeting:DeleteSegment');
        return this;
    }
    /**
     * Delete the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toDeleteSmsChannel() {
        this.add('mobiletargeting:DeleteSmsChannel');
        return this;
    }
    /**
     * Delete an sms message template or an sms message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toDeleteSmsTemplate() {
        this.add('mobiletargeting:DeleteSmsTemplate');
        return this;
    }
    /**
     * Delete all of the endpoints that are associated with a user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    toDeleteUserEndpoints() {
        this.add('mobiletargeting:DeleteUserEndpoints');
        return this;
    }
    /**
     * Delete the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toDeleteVoiceChannel() {
        this.add('mobiletargeting:DeleteVoiceChannel');
        return this;
    }
    /**
     * Delete a voice message template or a voice message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toDeleteVoiceTemplate() {
        this.add('mobiletargeting:DeleteVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toGetAdmChannel() {
        this.add('mobiletargeting:GetAdmChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toGetApnsChannel() {
        this.add('mobiletargeting:GetApnsChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toGetApnsSandboxChannel() {
        this.add('mobiletargeting:GetApnsSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toGetApnsVoipChannel() {
        this.add('mobiletargeting:GetApnsVoipChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toGetApnsVoipSandboxChannel() {
        this.add('mobiletargeting:GetApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about a specific app in your Amazon Pinpoint account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toGetApp() {
        this.add('mobiletargeting:GetApp');
        return this;
    }
    /**
     * Retrieve the default settings for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    toGetApplicationSettings() {
        this.add('mobiletargeting:GetApplicationSettings');
        return this;
    }
    /**
     * Retrieve a list of apps in your Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apps.html#rest-api-apps-methods
     */
    toGetApps() {
        this.add('mobiletargeting:GetApps');
        return this;
    }
    /**
     * Retrieve information about the Baidu channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toGetBaiduChannel() {
        this.add('mobiletargeting:GetBaiduChannel');
        return this;
    }
    /**
     * Retrieve information about a specific campaign.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toGetCampaign() {
        this.add('mobiletargeting:GetCampaign');
        return this;
    }
    /**
     * Retrieve information about the activities performed by a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-activities.html#rest-api-campaign-activities-methods
     */
    toGetCampaignActivities() {
        this.add('mobiletargeting:GetCampaignActivities');
        return this;
    }
    /**
     * Retrieve information about a specific campaign version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-version.html#rest-api-campaign-version-methods
     */
    toGetCampaignVersion() {
        this.add('mobiletargeting:GetCampaignVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-versions.html#rest-api-campaign-versions-methods
     */
    toGetCampaignVersions() {
        this.add('mobiletargeting:GetCampaignVersions');
        return this;
    }
    /**
     * Retrieve information about all campaigns for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    toGetCampaigns() {
        this.add('mobiletargeting:GetCampaigns');
        return this;
    }
    /**
     * Get all channels information for your app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-channels.html#rest-api-channels-methods
     */
    toGetChannels() {
        this.add('mobiletargeting:GetChannels');
        return this;
    }
    /**
     * Obtain information about the email channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toGetEmailChannel() {
        this.add('mobiletargeting:GetEmailChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an email template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toGetEmailTemplate() {
        this.add('mobiletargeting:GetEmailTemplate');
        return this;
    }
    /**
     * Retrieve information about a specific endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toGetEndpoint() {
        this.add('mobiletargeting:GetEndpoint');
        return this;
    }
    /**
     * Retrieve information about the event stream for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toGetEventStream() {
        this.add('mobiletargeting:GetEventStream');
        return this;
    }
    /**
     * Obtain information about a specific export job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetExportJob() {
        this.add('mobiletargeting:GetExportJob');
        return this;
    }
    /**
     * Retrieve a list of all of the export jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetExportJobs() {
        this.add('mobiletargeting:GetExportJobs');
        return this;
    }
    /**
     * Retrieve information about the GCM channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toGetGcmChannel() {
        this.add('mobiletargeting:GetGcmChannel');
        return this;
    }
    /**
     * Retrieve information about a specific import job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-job.html#rest-api-import-job-methods
     */
    toGetImportJob() {
        this.add('mobiletargeting:GetImportJob');
        return this;
    }
    /**
     * Retrieve information about all import jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toGetImportJobs() {
        this.add('mobiletargeting:GetImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific journey.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toGetJourney() {
        this.add('mobiletargeting:GetJourney');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an push notification template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toGetPushTemplate() {
        this.add('mobiletargeting:GetPushTemplate');
        return this;
    }
    /**
     * Retrieve information about an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration
     */
    toGetRecommenderConfiguration() {
        this.add('mobiletargeting:GetRecommenderConfiguration');
        return this;
    }
    /**
     * Retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations
     */
    toGetRecommenderConfigurations() {
        this.add('mobiletargeting:GetRecommenderConfigurations');
        return this;
    }
    /**
     * Retrieve information about a specific segment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toGetSegment() {
        this.add('mobiletargeting:GetSegment');
        return this;
    }
    /**
     * Retrieve information about jobs that export endpoint definitions from segments to Amazon S3.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetSegmentExportJobs() {
        this.add('mobiletargeting:GetSegmentExportJobs');
        return this;
    }
    /**
     * Retrieve information about jobs that create segments by importing endpoint definitions from .
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toGetSegmentImportJobs() {
        this.add('mobiletargeting:GetSegmentImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific segment version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-version.html#rest-api-segment-version-methods
     */
    toGetSegmentVersion() {
        this.add('mobiletargeting:GetSegmentVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a segment.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-versions.html#rest-api-segment-versions-methods
     */
    toGetSegmentVersions() {
        this.add('mobiletargeting:GetSegmentVersions');
        return this;
    }
    /**
     * Retrieve information about the segments for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    toGetSegments() {
        this.add('mobiletargeting:GetSegments');
        return this;
    }
    /**
     * Obtain information about the SMS channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toGetSmsChannel() {
        this.add('mobiletargeting:GetSmsChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an sms message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toGetSmsTemplate() {
        this.add('mobiletargeting:GetSmsTemplate');
        return this;
    }
    /**
     * Retrieve information about the endpoints that are associated with a user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    toGetUserEndpoints() {
        this.add('mobiletargeting:GetUserEndpoints');
        return this;
    }
    /**
     * Obtain information about the Voice channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toGetVoiceChannel() {
        this.add('mobiletargeting:GetVoiceChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of a voice message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toGetVoiceTemplate() {
        this.add('mobiletargeting:GetVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about all journeys for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#apps-application-id-journeys-http-methods
     */
    toListJourneys() {
        this.add('mobiletargeting:ListJourneys');
        return this;
    }
    /**
     * List tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTagsForResource() {
        this.add('mobiletargeting:ListTagsForResource');
        return this;
    }
    /**
     * Retrieve all versions about a specific template.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    toListTemplateVersions() {
        this.add('mobiletargeting:ListTemplateVersions');
        return this;
    }
    /**
     * Retrieve metadata about the queried templates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#templates-http-methods
     */
    toListTemplates() {
        this.add('mobiletargeting:ListTemplates');
        return this;
    }
    /**
     * Obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-phone-number-validate.html#rest-api-phone-number-validate-methods
     */
    toPhoneNumberValidate() {
        this.add('mobiletargeting:PhoneNumberValidate');
        return this;
    }
    /**
     * Create or update an event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toPutEventStream() {
        this.add('mobiletargeting:PutEventStream');
        return this;
    }
    /**
     * Create or update events for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-events.html#rest-api-events-methods
     */
    toPutEvents() {
        this.add('mobiletargeting:PutEvents');
        return this;
    }
    /**
     * Used to remove the attributes for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toRemoveAttributes() {
        this.add('mobiletargeting:RemoveAttributes');
        return this;
    }
    /**
     * Send an SMS message or push notification to specific endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-messages.html#rest-api-messages-methods
     */
    toSendMessages() {
        this.add('mobiletargeting:SendMessages');
        return this;
    }
    /**
     * Send an SMS message or push notification to all endpoints that are associated with a specific user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-users-messages.html#rest-api-users-messages-methods
     */
    toSendUsersMessages() {
        this.add('mobiletargeting:SendUsersMessages');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toTagResource() {
        this.add('mobiletargeting:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toUntagResource() {
        this.add('mobiletargeting:UntagResource');
        return this;
    }
    /**
     * Update the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toUpdateAdmChannel() {
        this.add('mobiletargeting:UpdateAdmChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toUpdateApnsChannel() {
        this.add('mobiletargeting:UpdateApnsChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toUpdateApnsSandboxChannel() {
        this.add('mobiletargeting:UpdateApnsSandboxChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toUpdateApnsVoipChannel() {
        this.add('mobiletargeting:UpdateApnsVoipChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toUpdateApnsVoipSandboxChannel() {
        this.add('mobiletargeting:UpdateApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Update the default settings for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    toUpdateApplicationSettings() {
        this.add('mobiletargeting:UpdateApplicationSettings');
        return this;
    }
    /**
     * Update the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toUpdateBaiduChannel() {
        this.add('mobiletargeting:UpdateBaiduChannel');
        return this;
    }
    /**
     * Update a specific campaign.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toUpdateCampaign() {
        this.add('mobiletargeting:UpdateCampaign');
        return this;
    }
    /**
     * Update the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toUpdateEmailChannel() {
        this.add('mobiletargeting:UpdateEmailChannel');
        return this;
    }
    /**
     * Update a specific email template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toUpdateEmailTemplate() {
        this.add('mobiletargeting:UpdateEmailTemplate');
        return this;
    }
    /**
     * Create an endpoint or update the information for an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toUpdateEndpoint() {
        this.add('mobiletargeting:UpdateEndpoint');
        return this;
    }
    /**
     * Create or update endpoints as a batch operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoints.html#rest-api-endpoints-methods
     */
    toUpdateEndpointsBatch() {
        this.add('mobiletargeting:UpdateEndpointsBatch');
        return this;
    }
    /**
     * Update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toUpdateGcmChannel() {
        this.add('mobiletargeting:UpdateGcmChannel');
        return this;
    }
    /**
     * Update a specific journey.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toUpdateJourney() {
        this.add('mobiletargeting:UpdateJourney');
        return this;
    }
    /**
     * Update a specific journey state.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#apps-application-id-journeys-journey-id-state-http-methods
     */
    toUpdateJourneyState() {
        this.add('mobiletargeting:UpdateJourneyState');
        return this;
    }
    /**
     * Update a specific push notification template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toUpdatePushTemplate() {
        this.add('mobiletargeting:UpdatePushTemplate');
        return this;
    }
    /**
     * Update an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration
     */
    toUpdateRecommenderConfiguration() {
        this.add('mobiletargeting:UpdateRecommenderConfiguration');
        return this;
    }
    /**
     * Update a specific segment.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toUpdateSegment() {
        this.add('mobiletargeting:UpdateSegment');
        return this;
    }
    /**
     * Update the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toUpdateSmsChannel() {
        this.add('mobiletargeting:UpdateSmsChannel');
        return this;
    }
    /**
     * Update a specific sms message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toUpdateSmsTemplate() {
        this.add('mobiletargeting:UpdateSmsTemplate');
        return this;
    }
    /**
     * Upate the active version parameter of a specific template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    toUpdateTemplateActiveVersion() {
        this.add('mobiletargeting:UpdateTemplateActiveVersion');
        return this;
    }
    /**
     * Update the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toUpdateVoiceChannel() {
        this.add('mobiletargeting:UpdateVoiceChannel');
        return this;
    }
    /**
     * Update a specific voice message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toUpdateVoiceTemplate() {
        this.add('mobiletargeting:UpdateVoiceTemplate');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/developerguide/gettingstarted.html#gettingstarted-addapp
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type campaigns to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html
     *
     * @param appId - Identifier for the appId.
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaigns(appId, campaignId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/campaigns/${CampaignId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${CampaignId}', campaignId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type journeys to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJourneys(appId, journeyId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/journeys/${JourneyId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${JourneyId}', journeyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type segments to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html
     *
     * @param appId - Identifier for the appId.
     * @param segmentId - Identifier for the segmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSegments(appId, segmentId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/segments/${SegmentId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${SegmentId}', segmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type templates to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html
     *
     * @param templateName - Identifier for the templateName.
     * @param channelType - Identifier for the channelType.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplates(templateName, channelType, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:templates/${TemplateName}/${ChannelType}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${ChannelType}', channelType);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recommenders to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html
     *
     * @param recommenderId - Identifier for the recommenderId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRecommenders(recommenderId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:recommenders/${RecommenderId}';
        arn = arn.replace('${RecommenderId}', recommenderId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the pinpoint service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCampaign()
     * - .toUpdateEmailTemplate()
     * - .toUpdateJourney()
     * - .toUpdateJourneyState()
     * - .toUpdatePushTemplate()
     * - .toUpdateSegment()
     * - .toUpdateSmsTemplate()
     * - .toUpdateVoiceTemplate()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     *
     * Applies to resource types:
     * - apps
     * - campaigns
     * - journeys
     * - segments
     * - templates
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the pinpoint service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCampaign()
     * - .toUpdateEmailTemplate()
     * - .toUpdateJourney()
     * - .toUpdateJourneyState()
     * - .toUpdatePushTemplate()
     * - .toUpdateSegment()
     * - .toUpdateSmsTemplate()
     * - .toUpdateVoiceTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Mobiletargeting = Mobiletargeting;
//# sourceMappingURL=data:application/json;base64,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