"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Polly = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Polly extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'polly';
        this.actionList = {
            "DeleteLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html",
                "description": "Deletes the specified pronunciation lexicon stored in an AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "DescribeVoices": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html",
                "description": "Returns the list of voices that are available for use when requesting speech synthesis.",
                "accessLevel": "List"
            },
            "GetLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html",
                "description": "Returns the content of the specified pronunciation lexicon stored in an AWS Region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "GetSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html",
                "description": "Enables the user to get information about specific speech synthesis task.",
                "accessLevel": "Read"
            },
            "ListLexicons": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html",
                "description": "Returns a list of pronunciation lexicons stored in an AWS Region.",
                "accessLevel": "List"
            },
            "ListSpeechSynthesisTasks": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html",
                "description": "Enables the user to list requested speech synthesis tasks.",
                "accessLevel": "List"
            },
            "PutLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html",
                "description": "Stores a pronunciation lexicon in an AWS Region.",
                "accessLevel": "Write"
            },
            "StartSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html",
                "description": "Enables the user to synthesize long inputs to provided S3 location.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:PutObject"
                ],
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            },
            "SynthesizeSpeech": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html",
                "description": "Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "lexicon": {
                "name": "lexicon",
                "url": "https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html",
                "arn": "arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html
     */
    toDeleteLexicon() {
        this.add('polly:DeleteLexicon');
        return this;
    }
    /**
     * Returns the list of voices that are available for use when requesting speech synthesis.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html
     */
    toDescribeVoices() {
        this.add('polly:DescribeVoices');
        return this;
    }
    /**
     * Returns the content of the specified pronunciation lexicon stored in an AWS Region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html
     */
    toGetLexicon() {
        this.add('polly:GetLexicon');
        return this;
    }
    /**
     * Enables the user to get information about specific speech synthesis task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html
     */
    toGetSpeechSynthesisTask() {
        this.add('polly:GetSpeechSynthesisTask');
        return this;
    }
    /**
     * Returns a list of pronunciation lexicons stored in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html
     */
    toListLexicons() {
        this.add('polly:ListLexicons');
        return this;
    }
    /**
     * Enables the user to list requested speech synthesis tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html
     */
    toListSpeechSynthesisTasks() {
        this.add('polly:ListSpeechSynthesisTasks');
        return this;
    }
    /**
     * Stores a pronunciation lexicon in an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html
     */
    toPutLexicon() {
        this.add('polly:PutLexicon');
        return this;
    }
    /**
     * Enables the user to synthesize long inputs to provided S3 location.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html
     */
    toStartSpeechSynthesisTask() {
        this.add('polly:StartSpeechSynthesisTask');
        return this;
    }
    /**
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html
     */
    toSynthesizeSpeech() {
        this.add('polly:SynthesizeSpeech');
        return this;
    }
    /**
     * Adds a resource of type lexicon to the statement
     *
     * https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html
     *
     * @param lexiconName - Identifier for the lexiconName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLexicon(lexiconName, account, region, partition) {
        var arn = 'arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}';
        arn = arn.replace('${LexiconName}', lexiconName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Polly = Polly;
//# sourceMappingURL=data:application/json;base64,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