"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rekognition = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rekognition';
        this.actionList = {
            "CompareFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html",
                "description": "Compares a face in source input image with each face detected in the target input image.",
                "accessLevel": "Read"
            },
            "CreateCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html",
                "description": "Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html",
                "description": "Creates a new Amazon Rekognition Custom Labels project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html",
                "description": "Creates a new version of a model and begins training.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    },
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "CreateStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html",
                "description": "Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    },
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DeleteCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html",
                "description": "Deletes the specified collection. Note that this operation removes all faces in the collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html",
                "description": "Deletes faces from a collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html",
                "description": "Deletes a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html",
                "description": "Deletes a model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DeleteStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html",
                "description": "Deletes the stream processor identified by Name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DescribeCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html",
                "description": "Describes the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DescribeProjectVersions": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html",
                "description": "Lists and describes the model versions in an Amazon Rekognition Custom Labels project.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeProjects": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html",
                "description": "Lists and gets information about your Amazon Rekognition Custom Labels projects.",
                "accessLevel": "Read"
            },
            "DescribeStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html",
                "description": "Provides information about a stream processor created by CreateStreamProcessor.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DetectCustomLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html",
                "description": "Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DetectFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html",
                "description": "Detects human faces within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html",
                "description": "Detects instances of real-world labels within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectModerationLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html",
                "description": "Detects moderation labels within input image.",
                "accessLevel": "Read"
            },
            "DetectText": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html",
                "description": "Detects text in the input image and converts it into machine-readable text.",
                "accessLevel": "Read"
            },
            "GetCelebrityInfo": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html",
                "description": "Gets the name and additional information about a celebrity based on his or her Rekognition ID.",
                "accessLevel": "Read"
            },
            "GetCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html",
                "description": "Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.",
                "accessLevel": "Read"
            },
            "GetContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html",
                "description": "Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.",
                "accessLevel": "Read"
            },
            "GetFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html",
                "description": "Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.",
                "accessLevel": "Read"
            },
            "GetFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html",
                "description": "Gets the face search results for Rekognition Video face search started by StartFaceSearch.",
                "accessLevel": "Read"
            },
            "GetLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html",
                "description": "Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.",
                "accessLevel": "Read"
            },
            "GetPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html",
                "description": "Gets information about people detected within a video.",
                "accessLevel": "Read"
            },
            "GetSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html",
                "description": "Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.",
                "accessLevel": "Read"
            },
            "GetTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html",
                "description": "Gets text detection results for a Rekognition Video analysis started by StartTextDetection.",
                "accessLevel": "Read"
            },
            "IndexFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html",
                "description": "Detects faces in the input image and adds them to the specified collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListCollections": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html",
                "description": "Returns a list of collection IDs in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html",
                "description": "Returns metadata for faces in the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListStreamProcessors": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html",
                "description": "Gets a list of stream processors that you have created with CreateStreamProcessor.",
                "accessLevel": "List",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "RecognizeCelebrities": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html",
                "description": "Returns an array of celebrities recognized in the input image.",
                "accessLevel": "Read"
            },
            "SearchFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html",
                "description": "For a given input face ID, searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "SearchFacesByImage": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html",
                "description": "For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html",
                "description": "Starts asynchronous recognition of celebrities in a video.",
                "accessLevel": "Write"
            },
            "StartContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html",
                "description": "Starts asynchronous detection of explicit or suggestive adult content in a video.",
                "accessLevel": "Write"
            },
            "StartFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html",
                "description": "Starts asynchronous detection of faces in a video.",
                "accessLevel": "Write"
            },
            "StartFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html",
                "description": "Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html",
                "description": "Starts asynchronous detection of labels in a video.",
                "accessLevel": "Write"
            },
            "StartPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html",
                "description": "Starts the asynchronous tracking of persons in a video.",
                "accessLevel": "Write"
            },
            "StartProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html",
                "description": "Starts the deployment of a model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StartSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html",
                "description": "Starts asynchronous detection of segments in a video.",
                "accessLevel": "Write"
            },
            "StartStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html",
                "description": "Starts processing a stream processor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "StartTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html",
                "description": "Starts asynchronous detection of text in a video.",
                "accessLevel": "Write"
            },
            "StopProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html",
                "description": "Stops a deployed model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StopStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html",
                "description": "Stops a running stream processor that was created by CreateStreamProcessor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "collection": {
                "name": "collection",
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}",
                "conditionKeys": []
            },
            "streamprocessor": {
                "name": "streamprocessor",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}",
                "conditionKeys": []
            },
            "projectversion": {
                "name": "projectversion",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Compares a face in source input image with each face detected in the target input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html
     */
    toCompareFaces() {
        this.add('rekognition:CompareFaces');
        return this;
    }
    /**
     * Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html
     */
    toCreateCollection() {
        this.add('rekognition:CreateCollection');
        return this;
    }
    /**
     * Creates a new Amazon Rekognition Custom Labels project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html
     */
    toCreateProject() {
        this.add('rekognition:CreateProject');
        return this;
    }
    /**
     * Creates a new version of a model and begins training.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html
     */
    toCreateProjectVersion() {
        this.add('rekognition:CreateProjectVersion');
        return this;
    }
    /**
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html
     */
    toCreateStreamProcessor() {
        this.add('rekognition:CreateStreamProcessor');
        return this;
    }
    /**
     * Deletes the specified collection. Note that this operation removes all faces in the collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html
     */
    toDeleteCollection() {
        this.add('rekognition:DeleteCollection');
        return this;
    }
    /**
     * Deletes faces from a collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html
     */
    toDeleteFaces() {
        this.add('rekognition:DeleteFaces');
        return this;
    }
    /**
     * Deletes a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html
     */
    toDeleteProject() {
        this.add('rekognition:DeleteProject');
        return this;
    }
    /**
     * Deletes a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html
     */
    toDeleteProjectVersion() {
        this.add('rekognition:DeleteProjectVersion');
        return this;
    }
    /**
     * Deletes the stream processor identified by Name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html
     */
    toDeleteStreamProcessor() {
        this.add('rekognition:DeleteStreamProcessor');
        return this;
    }
    /**
     * Describes the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html
     */
    toDescribeCollection() {
        this.add('rekognition:DescribeCollection');
        return this;
    }
    /**
     * Lists and describes the model versions in an Amazon Rekognition Custom Labels project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html
     */
    toDescribeProjectVersions() {
        this.add('rekognition:DescribeProjectVersions');
        return this;
    }
    /**
     * Lists and gets information about your Amazon Rekognition Custom Labels projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html
     */
    toDescribeProjects() {
        this.add('rekognition:DescribeProjects');
        return this;
    }
    /**
     * Provides information about a stream processor created by CreateStreamProcessor.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html
     */
    toDescribeStreamProcessor() {
        this.add('rekognition:DescribeStreamProcessor');
        return this;
    }
    /**
     * Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html
     */
    toDetectCustomLabels() {
        this.add('rekognition:DetectCustomLabels');
        return this;
    }
    /**
     * Detects human faces within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html
     */
    toDetectFaces() {
        this.add('rekognition:DetectFaces');
        return this;
    }
    /**
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html
     */
    toDetectLabels() {
        this.add('rekognition:DetectLabels');
        return this;
    }
    /**
     * Detects moderation labels within input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html
     */
    toDetectModerationLabels() {
        this.add('rekognition:DetectModerationLabels');
        return this;
    }
    /**
     * Detects text in the input image and converts it into machine-readable text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html
     */
    toDetectText() {
        this.add('rekognition:DetectText');
        return this;
    }
    /**
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html
     */
    toGetCelebrityInfo() {
        this.add('rekognition:GetCelebrityInfo');
        return this;
    }
    /**
     * Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html
     */
    toGetCelebrityRecognition() {
        this.add('rekognition:GetCelebrityRecognition');
        return this;
    }
    /**
     * Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html
     */
    toGetContentModeration() {
        this.add('rekognition:GetContentModeration');
        return this;
    }
    /**
     * Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html
     */
    toGetFaceDetection() {
        this.add('rekognition:GetFaceDetection');
        return this;
    }
    /**
     * Gets the face search results for Rekognition Video face search started by StartFaceSearch.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html
     */
    toGetFaceSearch() {
        this.add('rekognition:GetFaceSearch');
        return this;
    }
    /**
     * Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html
     */
    toGetLabelDetection() {
        this.add('rekognition:GetLabelDetection');
        return this;
    }
    /**
     * Gets information about people detected within a video.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html
     */
    toGetPersonTracking() {
        this.add('rekognition:GetPersonTracking');
        return this;
    }
    /**
     * Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html
     */
    toGetSegmentDetection() {
        this.add('rekognition:GetSegmentDetection');
        return this;
    }
    /**
     * Gets text detection results for a Rekognition Video analysis started by StartTextDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html
     */
    toGetTextDetection() {
        this.add('rekognition:GetTextDetection');
        return this;
    }
    /**
     * Detects faces in the input image and adds them to the specified collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html
     */
    toIndexFaces() {
        this.add('rekognition:IndexFaces');
        return this;
    }
    /**
     * Returns a list of collection IDs in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html
     */
    toListCollections() {
        this.add('rekognition:ListCollections');
        return this;
    }
    /**
     * Returns metadata for faces in the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html
     */
    toListFaces() {
        this.add('rekognition:ListFaces');
        return this;
    }
    /**
     * Gets a list of stream processors that you have created with CreateStreamProcessor.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html
     */
    toListStreamProcessors() {
        this.add('rekognition:ListStreamProcessors');
        return this;
    }
    /**
     * Returns an array of celebrities recognized in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html
     */
    toRecognizeCelebrities() {
        this.add('rekognition:RecognizeCelebrities');
        return this;
    }
    /**
     * For a given input face ID, searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html
     */
    toSearchFaces() {
        this.add('rekognition:SearchFaces');
        return this;
    }
    /**
     * For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html
     */
    toSearchFacesByImage() {
        this.add('rekognition:SearchFacesByImage');
        return this;
    }
    /**
     * Starts asynchronous recognition of celebrities in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html
     */
    toStartCelebrityRecognition() {
        this.add('rekognition:StartCelebrityRecognition');
        return this;
    }
    /**
     * Starts asynchronous detection of explicit or suggestive adult content in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html
     */
    toStartContentModeration() {
        this.add('rekognition:StartContentModeration');
        return this;
    }
    /**
     * Starts asynchronous detection of faces in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html
     */
    toStartFaceDetection() {
        this.add('rekognition:StartFaceDetection');
        return this;
    }
    /**
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html
     */
    toStartFaceSearch() {
        this.add('rekognition:StartFaceSearch');
        return this;
    }
    /**
     * Starts asynchronous detection of labels in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html
     */
    toStartLabelDetection() {
        this.add('rekognition:StartLabelDetection');
        return this;
    }
    /**
     * Starts the asynchronous tracking of persons in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html
     */
    toStartPersonTracking() {
        this.add('rekognition:StartPersonTracking');
        return this;
    }
    /**
     * Starts the deployment of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html
     */
    toStartProjectVersion() {
        this.add('rekognition:StartProjectVersion');
        return this;
    }
    /**
     * Starts asynchronous detection of segments in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html
     */
    toStartSegmentDetection() {
        this.add('rekognition:StartSegmentDetection');
        return this;
    }
    /**
     * Starts processing a stream processor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html
     */
    toStartStreamProcessor() {
        this.add('rekognition:StartStreamProcessor');
        return this;
    }
    /**
     * Starts asynchronous detection of text in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html
     */
    toStartTextDetection() {
        this.add('rekognition:StartTextDetection');
        return this;
    }
    /**
     * Stops a deployed model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html
     */
    toStopProjectVersion() {
        this.add('rekognition:StopProjectVersion');
        return this;
    }
    /**
     * Stops a running stream processor that was created by CreateStreamProcessor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html
     */
    toStopStreamProcessor() {
        this.add('rekognition:StopStreamProcessor');
        return this;
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCollection(collectionId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}';
        arn = arn.replace('${CollectionId}', collectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}';
        arn = arn.replace('${StreamprocessorId}', streamprocessorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${VersionName}', versionName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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