"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53resolver = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [route53resolver](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53resolver.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53resolver extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [route53resolver](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53resolver.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'route53resolver';
        this.actionList = {
            "AssociateResolverEndpointIpAddress": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverEndpointIpAddress.html",
                "description": "Grants permission to associate a specified IP address with a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "AssociateResolverQueryLogConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverQueryLogConfig.html",
                "description": "Grants permission to associate an Amazon VPC with a specified query logging configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "AssociateResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverRule.html",
                "description": "Grants permission to associate a specified Resolver rule with a specified VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "CreateResolverEndpoint": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html",
                "description": "Grants permission to create a Resolver endpoint. There are two types of Resolver endpoints, inbound and outbound",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "CreateResolverQueryLogConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverQueryLogConfig.html",
                "description": "Grants permission to create a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that originate in your VPCs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "CreateResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverRule.html",
                "description": "For DNS queries that originate in your VPC, grants permission to define how to route the queries out of the VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "DeleteResolverEndpoint": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverEndpoint.html",
                "description": "Grants permission to delete a Resolver endpoint. The effect of deleting a Resolver endpoint depends on whether it's an inbound or an outbound endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteResolverQueryLogConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverQueryLogConfig.html",
                "description": "Grants permission to delete a Resolver query logging configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "DeleteResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverRule.html",
                "description": "Grants permission to delete a Resolver rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "DisassociateResolverEndpointIpAddress": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverEndpointIpAddress.html",
                "description": "Grants permission to remove a specified IP address from a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "DisassociateResolverQueryLogConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html",
                "description": "Grants permission to remove the association between a specified Resolver query logging configuration and a specified VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "DisassociateResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverRule.html",
                "description": "Grants permission to remove the association between a specified Resolver rule and a specified VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "GetResolverEndpoint": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html",
                "description": "Grants permission to get information about a specified Resolver endpoint, such as whether it's an inbound or an outbound endpoint, and the IP addresses in your VPC that DNS queries are forwarded to on the way into or out of your VPC",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "GetResolverQueryLogConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfig.html",
                "description": "Grants permission to get information about a specified Resolver query logging configuration, such as the number of VPCs that the configuration is logging queries for and the location that logs are sent to",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "GetResolverQueryLogConfigAssociation": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigAssociation.html",
                "description": "Grants permission to get information about a specified association between a Resolver query logging configuration and an Amazon VPC. When you associate a VPC with a query logging configuration, Resolver logs DNS queries that originate in that VPC",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "GetResolverQueryLogConfigPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigPolicy.html",
                "description": "Grants permission to get information about a specified Resolver query logging policy, which specifies the Resolver query logging operations and resources that you want to allow another AWS account to use",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "GetResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRule.html",
                "description": "Grants permission to get information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for and the IP address that queries are forwarded to.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "GetResolverRuleAssociation": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRuleAssociation.html",
                "description": "Grants permission to get information about an association between a specified Resolver rule and a VPC",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "GetResolverRulePolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRulePolicy.html",
                "description": "Grants permission to get information about a Resolver rule policy, which specifies the Resolver operations and resources that you want to allow another AWS account to use",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "ListResolverEndpointIpAddresses": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpointIpAddresses.html",
                "description": "For a specified Resolver endpoint, grants permission to list the IP addresses that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)",
                "accessLevel": "List",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "ListResolverEndpoints": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html",
                "description": "Grants permission to list all the Resolver endpoints that were created using the current AWS account",
                "accessLevel": "List"
            },
            "ListResolverQueryLogConfigAssociations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html",
                "description": "Grants permission to list information about associations between Amazon VPCs and query logging configurations",
                "accessLevel": "List",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "ListResolverQueryLogConfigs": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html",
                "description": "Grants permission to list information about the specified query logging configurations, which define where you want Resolver to save DNS query logs and specify the VPCs that you want to log queries for",
                "accessLevel": "List",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "ListResolverRuleAssociations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html",
                "description": "Grants permission to list the associations that were created between Resolver rules and VPCs using the current AWS account",
                "accessLevel": "List"
            },
            "ListResolverRules": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html",
                "description": "Grants permission to list the Resolver rules that were created using the current AWS account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListTagsForResource.html",
                "description": "Grants permission to list the tags that you associated with the specified resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": false
                    },
                    "resolver-rule": {
                        "required": false
                    }
                }
            },
            "PutResolverQueryLogConfigPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverQueryLogConfigPolicy.html",
                "description": "Grants permission to specify an AWS account that you want to share a query logging configuration with, the query logging configuration that you want to share, and the operations that you want the account to be able to perform on the configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-query-log-config": {
                        "required": true
                    }
                }
            },
            "PutResolverRulePolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverRulePolicy.html",
                "description": "Grants permission to specify an AWS account that you want to share rules with, the Resolver rules that you want to share, and the operations that you want the account to be able to perform on those rules",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_TagResource.html",
                "description": "Grants permission to add one or more tags to a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": false
                    },
                    "resolver-rule": {
                        "required": false
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UntagResource.html",
                "description": "Grants permission to remove one or more tags from a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": false
                    },
                    "resolver-rule": {
                        "required": false
                    }
                }
            },
            "UpdateResolverEndpoint": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverEndpoint.html",
                "description": "Grants permission to update selected settings for an inbound or an outbound Resolver endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-endpoint": {
                        "required": true
                    }
                }
            },
            "UpdateResolverRule": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverRule.html",
                "description": "Grants permission to update settings for a specified Resolver rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resolver-rule": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "resolver-query-log-config": {
                "name": "resolver-query-log-config",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:route53resolver:${Region}:${Account}:resolver-query-log-config/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "resolver-rule": {
                "name": "resolver-rule",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:route53resolver:${Region}:${Account}:resolver-rule/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "resolver-endpoint": {
                "name": "resolver-endpoint",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:route53resolver:${Region}:${Account}:resolver-endpoint/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate a specified IP address with a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverEndpointIpAddress.html
     */
    toAssociateResolverEndpointIpAddress() {
        this.add('route53resolver:AssociateResolverEndpointIpAddress');
        return this;
    }
    /**
     * Grants permission to associate an Amazon VPC with a specified query logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverQueryLogConfig.html
     */
    toAssociateResolverQueryLogConfig() {
        this.add('route53resolver:AssociateResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to associate a specified Resolver rule with a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverRule.html
     */
    toAssociateResolverRule() {
        this.add('route53resolver:AssociateResolverRule');
        return this;
    }
    /**
     * Grants permission to create a Resolver endpoint. There are two types of Resolver endpoints, inbound and outbound
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html
     */
    toCreateResolverEndpoint() {
        this.add('route53resolver:CreateResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to create a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that originate in your VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverQueryLogConfig.html
     */
    toCreateResolverQueryLogConfig() {
        this.add('route53resolver:CreateResolverQueryLogConfig');
        return this;
    }
    /**
     * For DNS queries that originate in your VPC, grants permission to define how to route the queries out of the VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverRule.html
     */
    toCreateResolverRule() {
        this.add('route53resolver:CreateResolverRule');
        return this;
    }
    /**
     * Grants permission to delete a Resolver endpoint. The effect of deleting a Resolver endpoint depends on whether it's an inbound or an outbound endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverEndpoint.html
     */
    toDeleteResolverEndpoint() {
        this.add('route53resolver:DeleteResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a Resolver query logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverQueryLogConfig.html
     */
    toDeleteResolverQueryLogConfig() {
        this.add('route53resolver:DeleteResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to delete a Resolver rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverRule.html
     */
    toDeleteResolverRule() {
        this.add('route53resolver:DeleteResolverRule');
        return this;
    }
    /**
     * Grants permission to remove a specified IP address from a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverEndpointIpAddress.html
     */
    toDisassociateResolverEndpointIpAddress() {
        this.add('route53resolver:DisassociateResolverEndpointIpAddress');
        return this;
    }
    /**
     * Grants permission to remove the association between a specified Resolver query logging configuration and a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html
     */
    toDisassociateResolverQueryLogConfig() {
        this.add('route53resolver:DisassociateResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to remove the association between a specified Resolver rule and a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverRule.html
     */
    toDisassociateResolverRule() {
        this.add('route53resolver:DisassociateResolverRule');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver endpoint, such as whether it's an inbound or an outbound endpoint, and the IP addresses in your VPC that DNS queries are forwarded to on the way into or out of your VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html
     */
    toGetResolverEndpoint() {
        this.add('route53resolver:GetResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver query logging configuration, such as the number of VPCs that the configuration is logging queries for and the location that logs are sent to
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfig.html
     */
    toGetResolverQueryLogConfig() {
        this.add('route53resolver:GetResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to get information about a specified association between a Resolver query logging configuration and an Amazon VPC. When you associate a VPC with a query logging configuration, Resolver logs DNS queries that originate in that VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigAssociation.html
     */
    toGetResolverQueryLogConfigAssociation() {
        this.add('route53resolver:GetResolverQueryLogConfigAssociation');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver query logging policy, which specifies the Resolver query logging operations and resources that you want to allow another AWS account to use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigPolicy.html
     */
    toGetResolverQueryLogConfigPolicy() {
        this.add('route53resolver:GetResolverQueryLogConfigPolicy');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for and the IP address that queries are forwarded to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRule.html
     */
    toGetResolverRule() {
        this.add('route53resolver:GetResolverRule');
        return this;
    }
    /**
     * Grants permission to get information about an association between a specified Resolver rule and a VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRuleAssociation.html
     */
    toGetResolverRuleAssociation() {
        this.add('route53resolver:GetResolverRuleAssociation');
        return this;
    }
    /**
     * Grants permission to get information about a Resolver rule policy, which specifies the Resolver operations and resources that you want to allow another AWS account to use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRulePolicy.html
     */
    toGetResolverRulePolicy() {
        this.add('route53resolver:GetResolverRulePolicy');
        return this;
    }
    /**
     * For a specified Resolver endpoint, grants permission to list the IP addresses that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpointIpAddresses.html
     */
    toListResolverEndpointIpAddresses() {
        this.add('route53resolver:ListResolverEndpointIpAddresses');
        return this;
    }
    /**
     * Grants permission to list all the Resolver endpoints that were created using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html
     */
    toListResolverEndpoints() {
        this.add('route53resolver:ListResolverEndpoints');
        return this;
    }
    /**
     * Grants permission to list information about associations between Amazon VPCs and query logging configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html
     */
    toListResolverQueryLogConfigAssociations() {
        this.add('route53resolver:ListResolverQueryLogConfigAssociations');
        return this;
    }
    /**
     * Grants permission to list information about the specified query logging configurations, which define where you want Resolver to save DNS query logs and specify the VPCs that you want to log queries for
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html
     */
    toListResolverQueryLogConfigs() {
        this.add('route53resolver:ListResolverQueryLogConfigs');
        return this;
    }
    /**
     * Grants permission to list the associations that were created between Resolver rules and VPCs using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html
     */
    toListResolverRuleAssociations() {
        this.add('route53resolver:ListResolverRuleAssociations');
        return this;
    }
    /**
     * Grants permission to list the Resolver rules that were created using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html
     */
    toListResolverRules() {
        this.add('route53resolver:ListResolverRules');
        return this;
    }
    /**
     * Grants permission to list the tags that you associated with the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('route53resolver:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to specify an AWS account that you want to share a query logging configuration with, the query logging configuration that you want to share, and the operations that you want the account to be able to perform on the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverQueryLogConfigPolicy.html
     */
    toPutResolverQueryLogConfigPolicy() {
        this.add('route53resolver:PutResolverQueryLogConfigPolicy');
        return this;
    }
    /**
     * Grants permission to specify an AWS account that you want to share rules with, the Resolver rules that you want to share, and the operations that you want the account to be able to perform on those rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverRulePolicy.html
     */
    toPutResolverRulePolicy() {
        this.add('route53resolver:PutResolverRulePolicy');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_TagResource.html
     */
    toTagResource() {
        this.add('route53resolver:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UntagResource.html
     */
    toUntagResource() {
        this.add('route53resolver:UntagResource');
        return this;
    }
    /**
     * Grants permission to update selected settings for an inbound or an outbound Resolver endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverEndpoint.html
     */
    toUpdateResolverEndpoint() {
        this.add('route53resolver:UpdateResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to update settings for a specified Resolver rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverRule.html
     */
    toUpdateResolverRule() {
        this.add('route53resolver:UpdateResolverRule');
        return this;
    }
    /**
     * Adds a resource of type resolver-query-log-config to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverQueryLogConfig(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-query-log-config/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resolver-rule to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverRule(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-rule/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resolver-endpoint to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverEndpoint(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-endpoint/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - resolver-query-log-config
     * - resolver-rule
     * - resolver-endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Route53resolver = Route53resolver;
//# sourceMappingURL=data:application/json;base64,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