"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 's3';
        this.actionList = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html",
                "description": "Grants permission to abort a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "BypassGovernanceRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass",
                "description": "Grants permission to allow circumvention of governance-mode object retention settings",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html",
                "description": "Grants permission to create a new access point",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256"
                ]
            },
            "CreateBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html",
                "description": "Grants permission to create a new bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html",
                "description": "Grants permission to create a new Amazon S3 Batch Operations job",
                "accessLevel": "Write",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:RequestJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html",
                "description": "Grants permission to delete the access point named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html",
                "description": "Grants permission to delete the policy on a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html",
                "description": "Grants permission to delete the bucket named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html",
                "description": "Grants permission to delete the policy on a specified bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html",
                "description": "Grants permission to remove the website configuration for a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html",
                "description": "Grants permission to remove tags from an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to use the tagging subresource to remove the entire tag set from the specified object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove a specific version of an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to remove the entire tag set for a specific version of the object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html",
                "description": "Grants permission to retrieve the configuration parameters and status for a batch operations job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html",
                "description": "Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html",
                "description": "Grants permission to return configuration information about the specified access point",
                "accessLevel": "Read",
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html",
                "description": "Grants permission to returns the access point policy associated with the specified access point",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html",
                "description": "Grants permission to return the policy status for a specific access point policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html",
                "description": "Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html",
                "description": "Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html",
                "description": "Grants permission to return the CORS configuration information set for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketLocation": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html",
                "description": "Grants permission to return the Region that an Amazon S3 bucket resides in",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                }
            },
            "GetBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html",
                "description": "Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html",
                "description": "Grants permission to get the notification configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html",
                "description": "Grants permission to get the Object Lock configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "GetBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html",
                "description": "Grants permission to return the policy of the specified bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html",
                "description": "Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html",
                "description": "Grants permission to return the request payment configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html",
                "description": "Grants permission to return the tag set associated with an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html",
                "description": "Grants permission to return the versioning state of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html",
                "description": "Grants permission to return the website configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html",
                "description": "Grants permission to return the default encryption configuration an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html",
                "description": "Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html",
                "description": "Grants permission to return the tag set of an existing Amazon S3 Batch Operations job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html",
                "description": "Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html",
                "description": "Grants permission to get a metrics configuration from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve objects from Amazon S3",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html",
                "description": "Grants permission to get an object's current Legal Hold status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html",
                "description": "Grants permission to retrieve the retention settings for an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html",
                "description": "Grants permission to return the tag set of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to return torrent files from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve a specific version of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of a specific object version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionForReplication": {
                "url": "",
                "description": "Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTagging": {
                "url": "",
                "description": "Grants permission to return the tag set for a specific version of the object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to get Torrent files about a different version using the versionId subresource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html",
                "description": "Grants permission to get the replication configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAccessPoints": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html",
                "description": "Grants permission to list access points",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAllMyBuckets": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html",
                "description": "Grants permission to list all buckets owned by the authenticated sender of the request",
                "accessLevel": "List",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html",
                "description": "Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)",
                "accessLevel": "List",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketMultipartUploads": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html",
                "description": "Grants permission to list in-progress multipart uploads",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketVersions": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html",
                "description": "Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html",
                "description": "Grants permission to list current jobs and jobs that have ended recently",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListMultipartUploadParts": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html",
                "description": "Grants permission to list the parts that have been uploaded for a specific multipart upload",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ObjectOwnerOverrideToBucketOwner": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission",
                "description": "Grants permission to change replica ownership",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html",
                "description": "Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html",
                "description": "Grants permission to associate an access policy with a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Permissions management",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html",
                "description": "Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html",
                "description": "Grants permission to set the permissions on an existing bucket using access control lists (ACLs)",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "PutBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html",
                "description": "Grants permission to set the CORS configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html",
                "description": "Grants permission to set the logging parameters for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html",
                "description": "Grants permission to receive notifications when certain events happen in an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html",
                "description": "Grants permission to put Object Lock configuration on a specific bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "PutBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html",
                "description": "Grants permission to add or replace a bucket policy on a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html",
                "description": "Grants permission to set the request payment configuration of a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html",
                "description": "Grants permission to add a set of tags to an existing Amazon S3 bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html",
                "description": "Grants permission to set the versioning state of an existing Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html",
                "description": "Grants permission to set the configuration of the website that is specified in the website subresource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html",
                "description": "Grants permission to set the encryption configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html",
                "description": "Grants permission to add an inventory configuration to the bucket, identified by the inventory ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html",
                "description": "Grants permission to replace tags on an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html",
                "description": "Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html",
                "description": "Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html",
                "description": "Grants permission to add an object to a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html",
                "description": "Grants permission to apply a Legal Hold configuration to the specified object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html",
                "description": "Grants permission to place an Object Retention configuration on an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days"
                ]
            },
            "PutObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set to an object that already exists in a bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set for a specific version of an object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html",
                "description": "Grants permission to create a new replication configuration or replace an existing one",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateDelete": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate delete markers to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate objects and object tags to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id"
                ]
            },
            "ReplicateTags": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate object tags to the destination bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "RestoreObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html",
                "description": "Grants permission to restore an archived copy of an object back into Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "UpdateJobPriority": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html",
                "description": "Grants permission to update the priority of an existing job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "UpdateJobStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html",
                "description": "Grants permission to update the status for the specified job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "s3:JobSuspendedCause"
                ]
            }
        };
        this.resourceTypes = {
            "accesspoint": {
                "name": "accesspoint",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}",
                "conditionKeys": []
            },
            "bucket": {
                "name": "bucket",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html",
                "arn": "arn:${Partition}:s3:::${BucketName}",
                "conditionKeys": []
            },
            "object": {
                "name": "object",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html",
                "arn": "arn:${Partition}:s3:::${BucketName}/${ObjectName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:job/${JobId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload() {
        this.add('s3:AbortMultipartUpload');
        return this;
    }
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass
     */
    toBypassGovernanceRetention() {
        this.add('s3:BypassGovernanceRetention');
        return this;
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        this.add('s3:CreateAccessPoint');
        return this;
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket() {
        this.add('s3:CreateBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    toCreateJob() {
        this.add('s3:CreateJob');
        return this;
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        this.add('s3:DeleteAccessPoint');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        this.add('s3:DeleteAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket() {
        this.add('s3:DeleteBucket');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        this.add('s3:DeleteBucketPolicy');
        return this;
    }
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    toDeleteBucketWebsite() {
        this.add('s3:DeleteBucketWebsite');
        return this;
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    toDeleteJobTagging() {
        this.add('s3:DeleteJobTagging');
        return this;
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject() {
        this.add('s3:DeleteObject');
        return this;
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging() {
        this.add('s3:DeleteObjectTagging');
        return this;
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObjectVersion() {
        this.add('s3:DeleteObjectVersion');
        return this;
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectVersionTagging() {
        this.add('s3:DeleteObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    toDescribeJob() {
        this.add('s3:DescribeJob');
        return this;
    }
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    toGetAccelerateConfiguration() {
        this.add('s3:GetAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        this.add('s3:GetAccessPoint');
        return this;
    }
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        this.add('s3:GetAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    toGetAccessPointPolicyStatus() {
        this.add('s3:GetAccessPointPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    toGetAccountPublicAccessBlock() {
        this.add('s3:GetAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    toGetAnalyticsConfiguration() {
        this.add('s3:GetAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    toGetBucketAcl() {
        this.add('s3:GetBucketAcl');
        return this;
    }
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    toGetBucketCORS() {
        this.add('s3:GetBucketCORS');
        return this;
    }
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    toGetBucketLocation() {
        this.add('s3:GetBucketLocation');
        return this;
    }
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    toGetBucketLogging() {
        this.add('s3:GetBucketLogging');
        return this;
    }
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    toGetBucketNotification() {
        this.add('s3:GetBucketNotification');
        return this;
    }
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    toGetBucketObjectLockConfiguration() {
        this.add('s3:GetBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        this.add('s3:GetBucketPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    toGetBucketPolicyStatus() {
        this.add('s3:GetBucketPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    toGetBucketPublicAccessBlock() {
        this.add('s3:GetBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    toGetBucketRequestPayment() {
        this.add('s3:GetBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    toGetBucketTagging() {
        this.add('s3:GetBucketTagging');
        return this;
    }
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    toGetBucketVersioning() {
        this.add('s3:GetBucketVersioning');
        return this;
    }
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    toGetBucketWebsite() {
        this.add('s3:GetBucketWebsite');
        return this;
    }
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration() {
        this.add('s3:GetEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    toGetInventoryConfiguration() {
        this.add('s3:GetInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    toGetJobTagging() {
        this.add('s3:GetJobTagging');
        return this;
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        this.add('s3:GetLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    toGetMetricsConfiguration() {
        this.add('s3:GetMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject() {
        this.add('s3:GetObject');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectAcl() {
        this.add('s3:GetObjectAcl');
        return this;
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    toGetObjectLegalHold() {
        this.add('s3:GetObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    toGetObjectRetention() {
        this.add('s3:GetObjectRetention');
        return this;
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging() {
        this.add('s3:GetObjectTagging');
        return this;
    }
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectTorrent() {
        this.add('s3:GetObjectTorrent');
        return this;
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObjectVersion() {
        this.add('s3:GetObjectVersion');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectVersionAcl() {
        this.add('s3:GetObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     */
    toGetObjectVersionForReplication() {
        this.add('s3:GetObjectVersionForReplication');
        return this;
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     */
    toGetObjectVersionTagging() {
        this.add('s3:GetObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectVersionTorrent() {
        this.add('s3:GetObjectVersionTorrent');
        return this;
    }
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    toGetReplicationConfiguration() {
        this.add('s3:GetReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    toListAccessPoints() {
        this.add('s3:ListAccessPoints');
        return this;
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    toListAllMyBuckets() {
        this.add('s3:ListAllMyBuckets');
        return this;
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket() {
        this.add('s3:ListBucket');
        return this;
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads() {
        this.add('s3:ListBucketMultipartUploads');
        return this;
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    toListBucketVersions() {
        this.add('s3:ListBucketVersions');
        return this;
    }
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    toListJobs() {
        this.add('s3:ListJobs');
        return this;
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts() {
        this.add('s3:ListMultipartUploadParts');
        return this;
    }
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission
     */
    toObjectOwnerOverrideToBucketOwner() {
        this.add('s3:ObjectOwnerOverrideToBucketOwner');
        return this;
    }
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    toPutAccelerateConfiguration() {
        this.add('s3:PutAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        this.add('s3:PutAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    toPutAccountPublicAccessBlock() {
        this.add('s3:PutAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    toPutAnalyticsConfiguration() {
        this.add('s3:PutAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    toPutBucketAcl() {
        this.add('s3:PutBucketAcl');
        return this;
    }
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    toPutBucketCORS() {
        this.add('s3:PutBucketCORS');
        return this;
    }
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    toPutBucketLogging() {
        this.add('s3:PutBucketLogging');
        return this;
    }
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    toPutBucketNotification() {
        this.add('s3:PutBucketNotification');
        return this;
    }
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    toPutBucketObjectLockConfiguration() {
        this.add('s3:PutBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        this.add('s3:PutBucketPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    toPutBucketPublicAccessBlock() {
        this.add('s3:PutBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    toPutBucketRequestPayment() {
        this.add('s3:PutBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    toPutBucketTagging() {
        this.add('s3:PutBucketTagging');
        return this;
    }
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    toPutBucketVersioning() {
        this.add('s3:PutBucketVersioning');
        return this;
    }
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    toPutBucketWebsite() {
        this.add('s3:PutBucketWebsite');
        return this;
    }
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration() {
        this.add('s3:PutEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    toPutInventoryConfiguration() {
        this.add('s3:PutInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    toPutJobTagging() {
        this.add('s3:PutJobTagging');
        return this;
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        this.add('s3:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    toPutMetricsConfiguration() {
        this.add('s3:PutMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject() {
        this.add('s3:PutObject');
        return this;
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl() {
        this.add('s3:PutObjectAcl');
        return this;
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    toPutObjectLegalHold() {
        this.add('s3:PutObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    toPutObjectRetention() {
        this.add('s3:PutObjectRetention');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging() {
        this.add('s3:PutObjectTagging');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectVersionAcl() {
        this.add('s3:PutObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectVersionTagging() {
        this.add('s3:PutObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    toPutReplicationConfiguration() {
        this.add('s3:PutReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateDelete() {
        this.add('s3:ReplicateDelete');
        return this;
    }
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateObject() {
        this.add('s3:ReplicateObject');
        return this;
    }
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateTags() {
        this.add('s3:ReplicateTags');
        return this;
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    toRestoreObject() {
        this.add('s3:RestoreObject');
        return this;
    }
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    toUpdateJobPriority() {
        this.add('s3:UpdateJobPriority');
        return this;
    }
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    toUpdateJobStatus() {
        this.add('s3:UpdateJobStatus');
        return this;
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}';
        arn = arn.replace('${AccessPointName}', accessPointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(bucketName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(bucketName, objectName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}/${ObjectName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${ObjectName}', objectName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toPutJobTagging()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toPutJobTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`s3:AccessPointNetworkOrigin`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`s3:DataAccessPointAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`s3:DataAccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access to updating the job priority by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value, operator) {
        return this.if(`s3:ExistingJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling existing jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value, operator) {
        return this.if(`s3:ExistingJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Requires that an existing object tag has a specific tag key and value.
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`s3:ExistingObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling suspended jobs by a specific job suspended cause (for example, AWAITING_CONFIRMATION)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value, operator) {
        return this.if(`s3:JobSuspendedCause`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationConstraint(value, operator) {
        return this.if(`s3:LocationConstraint`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating jobs by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value, operator) {
        return this.if(`s3:RequestJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating new jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toUpdateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value, operator) {
        return this.if(`s3:RequestJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Restricts the tag keys and values allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`s3:RequestObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Restricts the tag keys allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`s3:RequestObjectTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`s3:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Restricts incoming requests to a specific authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`s3:authType`, value, operator || 'StringLike');
    }
    /**
     * Requires the delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`s3:delimiter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value, operator) {
        return this.if(`s3:locationconstraint`, value, operator || 'StringLike');
    }
    /**
     * Limits the maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`s3:max-keys`, value, operator || 'NumericEquals');
    }
    /**
     * Enables enforcement of the specified object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value, operator) {
        return this.if(`s3:object-lock-legal-hold`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of the specified object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value, operator) {
        return this.if(`s3:object-lock-mode`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of an object relative to the remaining retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRemainingRetentionDays(value, operator) {
        return this.if(`s3:object-lock-remaining-retention-days`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of a specific retain-until-date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRetainUntilDate(value, operator) {
        return this.if(`s3:object-lock-retain-until-date`, value, operator || 'StringLike');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`s3:prefix`, value, operator || 'StringLike');
    }
    /**
     * Identifies the length of time, in milliseconds, that a signature is valid in an authenticated request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`s3:signatureAge`, value, operator || 'NumericEquals');
    }
    /**
     * Identifies the version of AWS Signature that is supported for authenticated requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`s3:signatureversion`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`s3:versionid`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-acl header with a specific canned ACL in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`s3:x-amz-acl`, value, operator || 'StringLike');
    }
    /**
     * Disallows unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`s3:x-amz-content-sha256`, value, operator || 'StringLike');
    }
    /**
     * Restricts the copy source to a specific bucket, prefix, or object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`s3:x-amz-copy-source`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-full-control (full control) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value, operator) {
        return this.if(`s3:x-amz-grant-full-control`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read (read access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value, operator) {
        return this.if(`s3:x-amz-grant-read`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read-acp (read permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value, operator) {
        return this.if(`s3:x-amz-grant-read-acp`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write (write access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value, operator) {
        return this.if(`s3:x-amz-grant-write`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write-acp (write permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value, operator) {
        return this.if(`s3:x-amz-grant-write-acp`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`s3:x-amz-metadata-directive`, value, operator || 'StringLike');
    }
    /**
     * Requires server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption`, value, operator || 'StringLike');
    }
    /**
     * Requires a specific AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption-aws-kms-key-id`, value, operator || 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`s3:x-amz-storage-class`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value, operator) {
        return this.if(`s3:x-amz-website-redirect-location`, value, operator || 'StringLike');
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uczMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25zMy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBa0c7QUFFbEc7Ozs7R0FJRztBQUNILE1BQWEsRUFBRyxTQUFRLHdCQUFlO0lBNHFEckM7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFqckROLGtCQUFhLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLGVBQVUsR0FBWTtZQUM5QixzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWix1QkFBdUI7b0JBQ3ZCLDJCQUEyQjtvQkFDM0IsNkJBQTZCO29CQUM3QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLHVGQUF1RjtnQkFDdEcsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLDJCQUEyQjtvQkFDM0IseUJBQXlCO29CQUN6QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtvQkFDekIsc0JBQXNCO29CQUN0Qiw2QkFBNkI7b0JBQzdCLHFCQUFxQjtvQkFDckIseUJBQXlCO29CQUN6QixzQkFBc0I7b0JBQ3RCLDBCQUEwQjtvQkFDMUIsNkJBQTZCO29CQUM3QixpQ0FBaUM7b0JBQ2pDLGdEQUFnRDtvQkFDaEQsd0JBQXdCO29CQUN4QixvQ0FBb0M7b0JBQ3BDLHFCQUFxQjtvQkFDckIsa0NBQWtDO29CQUNsQyx5Q0FBeUM7b0JBQ3pDLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUsZ0RBQWdEO2dCQUMvRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsdUJBQXVCO29CQUN2QixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIsY0FBYztvQkFDZCx5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHVFQUF1RTtnQkFDOUUsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLHVCQUF1QjtvQkFDdkIsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLGNBQWM7b0JBQ2QseUJBQXlCO29CQUN6Qiw2QkFBNkI7b0JBQzdCLHFCQUFxQjtvQkFDckIseUJBQXlCO29CQUN6QixzQkFBc0I7b0JBQ3RCLDBCQUEwQjtpQkFDM0I7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNEVBQTRFO2dCQUNuRixhQUFhLEVBQUUsa0VBQWtFO2dCQUNqRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtvQkFDekIsdUJBQXVCO29CQUN2Qix3QkFBd0I7b0JBQ3hCLGFBQWE7b0JBQ2IsMEJBQTBCO2lCQUMzQjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSwrREFBK0Q7Z0JBQzlFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osdUJBQXVCO29CQUN2QiwyQkFBMkI7b0JBQzNCLDZCQUE2QjtvQkFDN0IsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QseUJBQXlCLEVBQUU7Z0JBQ3pCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSxvRUFBb0U7Z0JBQ25GLGFBQWEsRUFBRSx3QkFBd0I7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWix1QkFBdUI7b0JBQ3ZCLDJCQUEyQjtvQkFDM0IsNkJBQTZCO29CQUM3QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHVFQUF1RTtnQkFDOUUsYUFBYSxFQUFFLHlEQUF5RDtnQkFDeEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsb0VBQW9FO2dCQUNuRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUsa0ZBQWtGO2dCQUNqRyxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLEtBQUssRUFBRTt3QkFDTCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtvQkFDekIsd0JBQXdCO29CQUN4Qix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHVFQUF1RTtnQkFDOUUsYUFBYSxFQUFFLHVJQUF1STtnQkFDdEosYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHlHQUF5RztnQkFDeEgsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3Qiw0QkFBNEI7b0JBQzVCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsNkRBQTZEO2dCQUM1RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLGNBQWM7b0JBQ2QseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSxxRkFBcUY7Z0JBQ3BHLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsNEJBQTRCO29CQUM1QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsbUdBQW1HO2dCQUNsSCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLEtBQUssRUFBRTt3QkFDTCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsa0pBQWtKO2dCQUNqSyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsd0ZBQXdGO2dCQUN2RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHNCQUFzQixFQUFFO2dCQUN0QixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsaUdBQWlHO2dCQUNoSCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsNkZBQTZGO2dCQUNwRyxhQUFhLEVBQUUsa0ZBQWtGO2dCQUNqRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsc0ZBQXNGO2dCQUNyRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUsNEhBQTRIO2dCQUMzSSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsNkdBQTZHO2dCQUM1SCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsd0VBQXdFO2dCQUMvRSxhQUFhLEVBQUUsNEZBQTRGO2dCQUMzRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUsNEVBQTRFO2dCQUNuRixhQUFhLEVBQUUsNEVBQTRFO2dCQUMzRixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixhQUFhLEVBQUUsb0lBQW9JO2dCQUNuSixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixhQUFhLEVBQUUsZ0ZBQWdGO2dCQUMvRixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGtDQUFrQyxFQUFFO2dCQUNsQyxLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsK0VBQStFO2dCQUM5RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7aUJBQ3RCO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLDBFQUEwRTtnQkFDakYsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLCtIQUErSDtnQkFDOUksYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDJGQUEyRjtnQkFDMUcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHVGQUF1RjtnQkFDdEcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLDZFQUE2RTtnQkFDNUYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLCtFQUErRTtnQkFDOUYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHNGQUFzRjtnQkFDckcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLCtIQUErSDtnQkFDOUksYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLHVGQUF1RjtnQkFDdEcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGdHQUFnRztnQkFDL0csYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDJFQUEyRTtnQkFDMUYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLG9FQUFvRTtnQkFDM0UsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3Qiw0QkFBNEI7b0JBQzVCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsd0VBQXdFO2dCQUN2RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLDRCQUE0QjtvQkFDNUIsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSxnRUFBZ0U7Z0JBQy9FLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSxvRUFBb0U7Z0JBQ25GLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSxzREFBc0Q7Z0JBQ3JFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsNEJBQTRCO29CQUM1QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLG9FQUFvRTtnQkFDbkYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9FQUFvRTtnQkFDM0UsYUFBYSxFQUFFLCtEQUErRDtnQkFDOUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3Qiw0QkFBNEI7b0JBQzVCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLGNBQWM7b0JBQ2QseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx1RUFBdUU7Z0JBQzlFLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsNEJBQTRCO29CQUM1QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGdDQUFnQyxFQUFFO2dCQUNoQyxLQUFLLEVBQUUsRUFBRTtnQkFDVCxhQUFhLEVBQUUsc0dBQXNHO2dCQUNySCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsRUFBRTtnQkFDVCxhQUFhLEVBQUUsOEVBQThFO2dCQUM3RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLDRCQUE0QjtvQkFDNUIsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIsY0FBYztvQkFDZCx5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLGtHQUFrRztnQkFDakgsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsK0ZBQStGO2dCQUM5RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsc0VBQXNFO2dCQUM3RSxhQUFhLEVBQUUsd0ZBQXdGO2dCQUN2RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsd0VBQXdFO2dCQUMvRSxhQUFhLEVBQUUsMEZBQTBGO2dCQUN6RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsY0FBYztvQkFDZCxhQUFhO29CQUNiLFdBQVc7b0JBQ1gsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUseURBQXlEO2dCQUN4RSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUsNkZBQTZGO2dCQUM1RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsY0FBYztvQkFDZCxhQUFhO29CQUNiLFdBQVc7b0JBQ1gsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELFVBQVUsRUFBRTtnQkFDVixLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixhQUFhLEVBQUUsMEVBQTBFO2dCQUN6RixhQUFhLEVBQUUsTUFBTTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDBCQUEwQixFQUFFO2dCQUMxQixLQUFLLEVBQUUsb0VBQW9FO2dCQUMzRSxhQUFhLEVBQUUsNkZBQTZGO2dCQUM1RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGtDQUFrQyxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsa0hBQWtIO2dCQUN6SCxhQUFhLEVBQUUsK0NBQStDO2dCQUM5RCxhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxxSEFBcUg7Z0JBQ3BJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSwrRUFBK0U7Z0JBQzlGLGFBQWEsRUFBRSx3QkFBd0I7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCw2QkFBNkIsRUFBRTtnQkFDN0IsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLDhGQUE4RjtnQkFDN0csYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUsaUhBQWlIO2dCQUNoSSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsa0dBQWtHO2dCQUNqSCxhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIsY0FBYztvQkFDZCx5QkFBeUI7b0JBQ3pCLDZCQUE2QjtvQkFDN0IscUJBQXFCO29CQUNyQix5QkFBeUI7b0JBQ3pCLHNCQUFzQjtvQkFDdEIsMEJBQTBCO2lCQUMzQjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSw4RkFBOEY7Z0JBQzdHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0Qsa0NBQWtDLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtpQkFDdEI7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixhQUFhLEVBQUUsaUVBQWlFO2dCQUNoRixhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSwrRUFBK0U7Z0JBQ3RGLGFBQWEsRUFBRSwyR0FBMkc7Z0JBQzFILGFBQWEsRUFBRSx3QkFBd0I7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHdFQUF3RTtnQkFDdkYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLHdFQUF3RTtnQkFDdkYsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLCtFQUErRTtnQkFDOUYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLHdHQUF3RztnQkFDdkgsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLCtFQUErRTtnQkFDOUYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLG1HQUFtRztnQkFDbEgsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGlGQUFpRjtnQkFDaEcsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7b0JBQ3pCLHdCQUF3QjtvQkFDeEIseUJBQXlCO29CQUN6QixhQUFhO29CQUNiLDBCQUEwQjtpQkFDM0I7YUFDRjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUseUhBQXlIO2dCQUN4SSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUsd0hBQXdIO2dCQUN2SSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsb0VBQW9FO2dCQUMzRSxhQUFhLEVBQUUsZ0RBQWdEO2dCQUMvRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLDJCQUEyQjtvQkFDM0IseUJBQXlCO29CQUN6QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtvQkFDekIsc0JBQXNCO29CQUN0Qiw2QkFBNkI7b0JBQzdCLHFCQUFxQjtvQkFDckIseUJBQXlCO29CQUN6QixzQkFBc0I7b0JBQ3RCLDBCQUEwQjtvQkFDMUIsNkJBQTZCO29CQUM3QixpQ0FBaUM7b0JBQ2pDLGdEQUFnRDtvQkFDaEQsd0JBQXdCO29CQUN4QixvQ0FBb0M7b0JBQ3BDLHFCQUFxQjtvQkFDckIsa0NBQWtDO29CQUNsQyx5Q0FBeUM7b0JBQ3pDLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsa0hBQWtIO2dCQUNqSSxhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsNEJBQTRCO29CQUM1QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtvQkFDekIsNkJBQTZCO29CQUM3QixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtvQkFDekIsc0JBQXNCO29CQUN0QiwwQkFBMEI7b0JBQzFCLHdCQUF3QjtpQkFDekI7YUFDRjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUsK0VBQStFO2dCQUM5RixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDJCQUEyQjtvQkFDM0IsdUJBQXVCO29CQUN2Qiw2QkFBNkI7b0JBQzdCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtvQkFDekIsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSwyRUFBMkU7Z0JBQzFGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO29CQUN6QixxQkFBcUI7b0JBQ3JCLGtDQUFrQztvQkFDbEMseUNBQXlDO2lCQUMxQzthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMkJBQTJCO29CQUMzQix1QkFBdUI7b0JBQ3ZCLDZCQUE2QjtvQkFDN0IsNEJBQTRCO29CQUM1QiwyQkFBMkI7b0JBQzNCLHlCQUF5QjtvQkFDekIsYUFBYTtvQkFDYixpQkFBaUI7b0JBQ2pCLHFCQUFxQjtvQkFDckIseUJBQXlCO2lCQUMxQjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx1RUFBdUU7Z0JBQzlFLGFBQWEsRUFBRSw2SUFBNkk7Z0JBQzVKLGFBQWEsRUFBRSx3QkFBd0I7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3Qiw0QkFBNEI7b0JBQzVCLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLGNBQWM7b0JBQ2QsY0FBYztvQkFDZCx5QkFBeUI7b0JBQ3pCLDZCQUE2QjtvQkFDN0IscUJBQXFCO29CQUNyQix5QkFBeUI7b0JBQ3pCLHNCQUFzQjtvQkFDdEIsMEJBQTBCO29CQUMxQix3QkFBd0I7aUJBQ3pCO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLG1GQUFtRjtnQkFDbEcsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3Qiw0QkFBNEI7b0JBQzVCLDJCQUEyQjtvQkFDM0IseUJBQXlCO29CQUN6QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQixjQUFjO29CQUNkLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsd0ZBQXdGO2dCQUN2RyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUseUVBQXlFO2dCQUN4RixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtpQkFDMUI7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUsa0ZBQWtGO2dCQUNqRyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsaUJBQWlCO29CQUNqQixxQkFBcUI7b0JBQ3JCLHlCQUF5QjtvQkFDekIsaUNBQWlDO29CQUNqQyxnREFBZ0Q7aUJBQ2pEO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLGdGQUFnRjtnQkFDL0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkI7b0JBQzNCLHVCQUF1QjtvQkFDdkIsNkJBQTZCO29CQUM3QixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7b0JBQ3pCLHVCQUF1QjtvQkFDdkIsd0JBQXdCO29CQUN4Qix5QkFBeUI7aUJBQzFCO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGlCQUFpQjtvQkFDakIscUJBQXFCO29CQUNyQix5QkFBeUI7b0JBQ3pCLHdCQUF3QjtvQkFDeEIseUJBQXlCO29CQUN6QixzQkFBc0I7aUJBQ3ZCO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsYUFBYSxFQUFFO2dCQUNiLE1BQU0sRUFBRSxhQUFhO2dCQUNyQixLQUFLLEVBQUUsb0VBQW9FO2dCQUMzRSxLQUFLLEVBQUUseUVBQXlFO2dCQUNoRixlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELFFBQVEsRUFBRTtnQkFDUixNQUFNLEVBQUUsUUFBUTtnQkFDaEIsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsS0FBSyxFQUFFLHFDQUFxQztnQkFDNUMsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLFFBQVE7Z0JBQ2hCLEtBQUssRUFBRSxtRUFBbUU7Z0JBQzFFLEtBQUssRUFBRSxtREFBbUQ7Z0JBQzFELGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsS0FBSyxFQUFFO2dCQUNMLE1BQU0sRUFBRSxLQUFLO2dCQUNiLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLEtBQUssRUFBRSx1REFBdUQ7Z0JBQzlELGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQ0c7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FtQkc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN6QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN6QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdDQUFnQztRQUNyQyxJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzFCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQWtCRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDeEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQWtCRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGtDQUFrQztRQUN2QyxJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQ0c7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDekIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BdUJHO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F3Qkc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BbUJHO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhLENBQUMsZUFBdUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNqRyxJQUFJLEdBQUcsR0FBRyx5RUFBeUUsQ0FBQztRQUNwRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUN6RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxRQUFRLENBQUMsVUFBa0IsRUFBRSxTQUFrQjtRQUNwRCxJQUFJLEdBQUcsR0FBRyxxQ0FBcUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksUUFBUSxDQUFDLFVBQWtCLEVBQUUsVUFBa0IsRUFBRSxTQUFrQjtRQUN4RSxJQUFJLEdBQUcsR0FBRyxtREFBbUQsQ0FBQztRQUM5RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxLQUFLLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLElBQUksR0FBRyxHQUFHLHVEQUF1RCxDQUFDO1FBQ2xFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUNoRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBaUI7UUFDakYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzdELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTBDRztJQUNJLDBCQUEwQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDM0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0EwQ0c7SUFDSSx3QkFBd0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQy9FLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXdDRztJQUNJLG9CQUFvQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDckUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3ZFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0kscUJBQXFCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN0RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxlQUFlLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FzQkc7SUFDSSxtQkFBbUIsQ0FBQyxHQUFXLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUNqRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXlCLEdBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxtQkFBbUIsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3BFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksb0JBQW9CLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUNyRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMzRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHFCQUFxQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDdEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksb0JBQW9CLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUNyRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxlQUFlLENBQUMsQ0FBQztJQUM5RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxrQkFBa0IsQ0FBQyxHQUFXLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUNoRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXdCLEdBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3ZFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksV0FBVyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDNUQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0dHO0lBQ0ksVUFBVSxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDM0QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFdBQVcsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzVELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxvQkFBb0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzNFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFNBQVMsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzFELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxlQUFlLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kscUJBQXFCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN0RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdCQUFnQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDakUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDekUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtDQUFrQyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDN0YsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLDJCQUEyQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDNUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDdEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksUUFBUSxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDekQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0dHO0lBQ0ksY0FBYyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDL0QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksZUFBZSxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FzR0c7SUFDSSxrQkFBa0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ25FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxXQUFXLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUM1RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0ksU0FBUyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDMUQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW9HRztJQUNJLG1CQUFtQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUNqRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksc0JBQXNCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN2RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksZUFBZSxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDaEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDekUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGtCQUFrQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDbkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGdCQUFnQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDakUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLG1CQUFtQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksdUJBQXVCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksMEJBQTBCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUMzRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kscUNBQXFDLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUN0RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0RBQWdELEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNwRyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGtCQUFrQixDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDbkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksNkJBQTZCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUM5RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RixDQUFDO0NBQ0Y7QUFwdUpELGdCQW91SkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24sIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtzM10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uczMuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBTMyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3MzJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJBYm9ydE11bHRpcGFydFVwbG9hZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9BYm9ydE11bHRpcGFydFVwbG9hZC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYWJvcnQgYSBtdWx0aXBhcnQgdXBsb2FkXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQnlwYXNzR292ZXJuYW5jZVJldGVudGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L29iamVjdC1sb2NrLW1hbmFnaW5nLmh0bWwjb2JqZWN0LWxvY2stbWFuYWdpbmctYnlwYXNzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYWxsb3cgY2lyY3VtdmVudGlvbiBvZiBnb3Zlcm5hbmNlLW1vZGUgb2JqZWN0IHJldGVudGlvbiBzZXR0aW5nc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOlJlcXVlc3RPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzpSZXF1ZXN0T2JqZWN0VGFnS2V5c1wiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWFjbFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCIsXG4gICAgICAgIFwiczM6eC1hbXotY29weS1zb3VyY2VcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1mdWxsLWNvbnRyb2xcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1yZWFkXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtcmVhZC1hY3BcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC13cml0ZVwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXdyaXRlLWFjcFwiLFxuICAgICAgICBcInMzOngtYW16LW1ldGFkYXRhLWRpcmVjdGl2ZVwiLFxuICAgICAgICBcInMzOngtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1zZXJ2ZXItc2lkZS1lbmNyeXB0aW9uLWF3cy1rbXMta2V5LWlkXCIsXG4gICAgICAgIFwiczM6eC1hbXotc3RvcmFnZS1jbGFzc1wiLFxuICAgICAgICBcInMzOngtYW16LXdlYnNpdGUtcmVkaXJlY3QtbG9jYXRpb25cIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1tb2RlXCIsXG4gICAgICAgIFwiczM6b2JqZWN0LWxvY2stcmV0YWluLXVudGlsLWRhdGVcIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1yZW1haW5pbmctcmV0ZW50aW9uLWRheXNcIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1sZWdhbC1ob2xkXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlQWNjZXNzUG9pbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9DcmVhdGVBY2Nlc3NQb2ludC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGFjY2VzcyBwb2ludFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImFjY2Vzc3BvaW50XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6bG9jYXRpb25jb25zdHJhaW50XCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWFjbFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlQnVja2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0NyZWF0ZUJ1Y2tldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6bG9jYXRpb25jb25zdHJhaW50XCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWFjbFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtZnVsbC1jb250cm9sXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtcmVhZFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXJlYWQtYWNwXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtd3JpdGVcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC13cml0ZS1hY3BcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVKb2JcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9DcmVhdGVKb2IuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBBbWF6b24gUzMgQmF0Y2ggT3BlcmF0aW9ucyBqb2JcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiLFxuICAgICAgICBcInMzOlJlcXVlc3RKb2JQcmlvcml0eVwiLFxuICAgICAgICBcInMzOlJlcXVlc3RKb2JPcGVyYXRpb25cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiLFxuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZUFjY2Vzc1BvaW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVsZXRlQWNjZXNzUG9pbnQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgYWNjZXNzIHBvaW50IG5hbWVkIGluIHRoZSBVUklcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJhY2Nlc3Nwb2ludFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZUFjY2Vzc1BvaW50UG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVsZXRlQWNjZXNzUG9pbnRQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgcG9saWN5IG9uIGEgc3BlY2lmaWVkIGFjY2VzcyBwb2ludFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYWNjZXNzcG9pbnRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZWxldGVCdWNrZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlQnVja2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGJ1Y2tldCBuYW1lZCBpbiB0aGUgVVJJXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZWxldGVCdWNrZXRQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlQnVja2V0UG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHBvbGljeSBvbiBhIHNwZWNpZmllZCBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVsZXRlQnVja2V0V2Vic2l0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9EZWxldGVCdWNrZXRXZWJzaXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIHdlYnNpdGUgY29uZmlndXJhdGlvbiBmb3IgYSBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZUpvYlRhZ2dpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9EZWxldGVKb2JUYWdnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBCYXRjaCBPcGVyYXRpb25zIGpvYlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiam9iXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ0pvYlByaW9yaXR5XCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdKb2JPcGVyYXRpb25cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZWxldGVPYmplY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlT2JqZWN0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIG51bGwgdmVyc2lvbiBvZiBhbiBvYmplY3QgYW5kIGluc2VydCBhIGRlbGV0ZSBtYXJrZXIsIHdoaWNoIGJlY29tZXMgdGhlIGN1cnJlbnQgdmVyc2lvbiBvZiB0aGUgb2JqZWN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVsZXRlT2JqZWN0VGFnZ2luZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9EZWxldGVPYmplY3RUYWdnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1c2UgdGhlIHRhZ2dpbmcgc3VicmVzb3VyY2UgdG8gcmVtb3ZlIHRoZSBlbnRpcmUgdGFnIHNldCBmcm9tIHRoZSBzcGVjaWZpZWQgb2JqZWN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZU9iamVjdFZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlT2JqZWN0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgYSBzcGVjaWZpYyB2ZXJzaW9uIG9mIGFuIG9iamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp2ZXJzaW9uaWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZU9iamVjdFZlcnNpb25UYWdnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0RlbGV0ZU9iamVjdFRhZ2dpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgZW50aXJlIHRhZyBzZXQgZm9yIGEgc3BlY2lmaWMgdmVyc2lvbiBvZiB0aGUgb2JqZWN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp2ZXJzaW9uaWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlSm9iXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVzY3JpYmVKb2IuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBjb25maWd1cmF0aW9uIHBhcmFtZXRlcnMgYW5kIHN0YXR1cyBmb3IgYSBiYXRjaCBvcGVyYXRpb25zIGpvYi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImpvYlwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QWNjZWxlcmF0ZUNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0QWNjZWxlcmF0ZUNvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZXMgdGhlIGFjY2VsZXJhdGUgc3VicmVzb3VyY2UgdG8gcmV0dXJuIHRoZSBUcmFuc2ZlciBBY2NlbGVyYXRpb24gc3RhdGUgb2YgYSBidWNrZXQsIHdoaWNoIGlzIGVpdGhlciBFbmFibGVkIG9yIFN1c3BlbmRlZFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRBY2Nlc3NQb2ludFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0dldEFjY2Vzc1BvaW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3BlY2lmaWVkIGFjY2VzcyBwb2ludFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QWNjZXNzUG9pbnRQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9HZXRBY2Nlc3NQb2ludFBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJucyB0aGUgYWNjZXNzIHBvaW50IHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBhY2Nlc3MgcG9pbnRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImFjY2Vzc3BvaW50XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QWNjZXNzUG9pbnRQb2xpY3lTdGF0dXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9HZXRBY2Nlc3NQb2ludFBvbGljeVN0YXR1cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBwb2xpY3kgc3RhdHVzIGZvciBhIHNwZWNpZmljIGFjY2VzcyBwb2ludCBwb2xpY3lcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImFjY2Vzc3BvaW50XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QWNjb3VudFB1YmxpY0FjY2Vzc0Jsb2NrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfR2V0UHVibGljQWNjZXNzQmxvY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBQdWJsaWNBY2Nlc3NCbG9jayBjb25maWd1cmF0aW9uIGZvciBhbiBBV1MgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRBbmFseXRpY3NDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhbiBhbmFseXRpY3MgY29uZmlndXJhdGlvbiBmcm9tIGFuIEFtYXpvbiBTMyBidWNrZXQsIGlkZW50aWZpZWQgYnkgdGhlIGFuYWx5dGljcyBjb25maWd1cmF0aW9uIElEXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldEFjbFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRBY2wuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSB0aGUgYWNsIHN1YnJlc291cmNlIHRvIHJldHVybiB0aGUgYWNjZXNzIGNvbnRyb2wgbGlzdCAoQUNMKSBvZiBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldENPUlNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0Q29ycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBDT1JTIGNvbmZpZ3VyYXRpb24gaW5mb3JtYXRpb24gc2V0IGZvciBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldExvY2F0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldExvY2F0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIFJlZ2lvbiB0aGF0IGFuIEFtYXpvbiBTMyBidWNrZXQgcmVzaWRlcyBpblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRCdWNrZXRMb2dnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldExvZ2dpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgbG9nZ2luZyBzdGF0dXMgb2YgYW4gQW1hem9uIFMzIGJ1Y2tldCBhbmQgdGhlIHBlcm1pc3Npb25zIHVzZXJzIGhhdmUgdG8gdmlldyBvciBtb2RpZnkgdGhhdCBzdGF0dXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QnVja2V0Tm90aWZpY2F0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldE5vdGlmaWNhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBub3RpZmljYXRpb24gY29uZmlndXJhdGlvbiBvZiBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldE9iamVjdExvY2tDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdExvY2tDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIE9iamVjdCBMb2NrIGNvbmZpZ3VyYXRpb24gb2YgYW4gQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QnVja2V0UG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldFBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBwb2xpY3kgb2YgdGhlIHNwZWNpZmllZCBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QnVja2V0UG9saWN5U3RhdHVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldFBvbGljeVN0YXR1cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIHBvbGljeSBzdGF0dXMgZm9yIGEgc3BlY2lmaWMgQW1hem9uIFMzIGJ1Y2tldCwgd2hpY2ggaW5kaWNhdGVzIHdoZXRoZXIgdGhlIGJ1Y2tldCBpcyBwdWJsaWNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QnVja2V0UHVibGljQWNjZXNzQmxvY2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0UHVibGljQWNjZXNzQmxvY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBQdWJsaWNBY2Nlc3NCbG9jayBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldFJlcXVlc3RQYXltZW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldFJlcXVlc3RQYXltZW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHJlcXVlc3QgcGF5bWVudCBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldFRhZ2dpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0VGFnZ2luZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB0YWcgc2V0IGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldEJ1Y2tldFZlcnNpb25pbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0VmVyc2lvbmluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB2ZXJzaW9uaW5nIHN0YXRlIG9mIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0QnVja2V0V2Vic2l0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRXZWJzaXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHdlYnNpdGUgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRFbmNyeXB0aW9uQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRFbmNyeXB0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGRlZmF1bHQgZW5jcnlwdGlvbiBjb25maWd1cmF0aW9uIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0SW52ZW50b3J5Q29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRJbnZlbnRvcnlDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYW4gaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24gZnJvbSBhbiBBbWF6b24gUzMgYnVja2V0LCBpZGVudGlmaWVkIGJ5IHRoZSBpbnZlbnRvcnkgY29uZmlndXJhdGlvbiBJRFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRKb2JUYWdnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfR2V0Sm9iVGFnZ2luZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB0YWcgc2V0IG9mIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBCYXRjaCBPcGVyYXRpb25zIGpvYlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiam9iXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRMaWZlY3ljbGVDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgbGlmZWN5Y2xlIGNvbmZpZ3VyYXRpb24gaW5mb3JtYXRpb24gc2V0IG9uIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0TWV0cmljc0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0TWV0cmljc0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIG1ldHJpY3MgY29uZmlndXJhdGlvbiBmcm9tIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0T2JqZWN0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgb2JqZWN0cyBmcm9tIEFtYXpvbiBTM1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nT2JqZWN0VGFnLzxrZXk+XCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RBY2xcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0QWNsLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGFjY2VzcyBjb250cm9sIGxpc3QgKEFDTCkgb2YgYW4gb2JqZWN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldE9iamVjdExlZ2FsSG9sZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3RMZWdhbEhvbGQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhbiBvYmplY3QncyBjdXJyZW50IExlZ2FsIEhvbGQgc3RhdHVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RSZXRlbnRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0UmV0ZW50aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgcmV0ZW50aW9uIHNldHRpbmdzIGZvciBhbiBvYmplY3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldE9iamVjdFRhZ2dpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0VGFnZ2luZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB0YWcgc2V0IG9mIGFuIG9iamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nT2JqZWN0VGFnLzxrZXk+XCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RUb3JyZW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdFRvcnJlbnQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0b3JyZW50IGZpbGVzIGZyb20gYW4gQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBzcGVjaWZpYyB2ZXJzaW9uIG9mIGFuIG9iamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nT2JqZWN0VGFnLzxrZXk+XCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6dmVyc2lvbmlkXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RWZXJzaW9uQWNsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdEFjbC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBhY2Nlc3MgY29udHJvbCBsaXN0IChBQ0wpIG9mIGEgc3BlY2lmaWMgb2JqZWN0IHZlcnNpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ09iamVjdFRhZy88a2V5PlwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOnZlcnNpb25pZFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0T2JqZWN0VmVyc2lvbkZvclJlcGxpY2F0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVwbGljYXRlIGJvdGggdW5lbmNyeXB0ZWQgb2JqZWN0cyBhbmQgb2JqZWN0cyBlbmNyeXB0ZWQgd2l0aCBTU0UtUzMgb3IgU1NFLUtNU1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJHZXRPYmplY3RWZXJzaW9uVGFnZ2luZ1wiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgdGFnIHNldCBmb3IgYSBzcGVjaWZpYyB2ZXJzaW9uIG9mIHRoZSBvYmplY3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ09iamVjdFRhZy88a2V5PlwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOnZlcnNpb25pZFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiR2V0T2JqZWN0VmVyc2lvblRvcnJlbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0VG9ycmVudC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IFRvcnJlbnQgZmlsZXMgYWJvdXQgYSBkaWZmZXJlbnQgdmVyc2lvbiB1c2luZyB0aGUgdmVyc2lvbklkIHN1YnJlc291cmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp2ZXJzaW9uaWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkdldFJlcGxpY2F0aW9uQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRSZXBsaWNhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSByZXBsaWNhdGlvbiBjb25maWd1cmF0aW9uIGluZm9ybWF0aW9uIHNldCBvbiBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkxpc3RBY2Nlc3NQb2ludHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9MaXN0QWNjZXNzUG9pbnRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFjY2VzcyBwb2ludHNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiTGlzdEFsbE15QnVja2V0c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9MaXN0QnVja2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgYnVja2V0cyBvd25lZCBieSB0aGUgYXV0aGVudGljYXRlZCBzZW5kZXIgb2YgdGhlIHJlcXVlc3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiTGlzdEJ1Y2tldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9MaXN0T2JqZWN0c1YyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHNvbWUgb3IgYWxsIG9mIHRoZSBvYmplY3RzIGluIGFuIEFtYXpvbiBTMyBidWNrZXQgKHVwIHRvIDEwMDApXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpkZWxpbWl0ZXJcIixcbiAgICAgICAgXCJzMzptYXgta2V5c1wiLFxuICAgICAgICBcInMzOnByZWZpeFwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkxpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0xpc3RNdWx0aXBhcnRVcGxvYWRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluLXByb2dyZXNzIG11bHRpcGFydCB1cGxvYWRzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJMaXN0QnVja2V0VmVyc2lvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfTGlzdE9iamVjdFZlcnNpb25zLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IG1ldGFkYXRhIGFib3V0IGFsbCB0aGUgdmVyc2lvbnMgb2Ygb2JqZWN0cyBpbiBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpkZWxpbWl0ZXJcIixcbiAgICAgICAgXCJzMzptYXgta2V5c1wiLFxuICAgICAgICBcInMzOnByZWZpeFwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkxpc3RKb2JzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfTGlzdEpvYnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY3VycmVudCBqb2JzIGFuZCBqb2JzIHRoYXQgaGF2ZSBlbmRlZCByZWNlbnRseVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJMaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfTGlzdFBhcnRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBwYXJ0cyB0aGF0IGhhdmUgYmVlbiB1cGxvYWRlZCBmb3IgYSBzcGVjaWZpYyBtdWx0aXBhcnQgdXBsb2FkXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJPYmplY3RPd25lck92ZXJyaWRlVG9CdWNrZXRPd25lclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L3JlcGxpY2F0aW9uLWNoYW5nZS1vd25lci5odG1sI3JlcGwtb3duZXJzaGlwLWFkZC1yb2xlLXBlcm1pc3Npb25cIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjaGFuZ2UgcmVwbGljYSBvd25lcnNoaXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QWNjZWxlcmF0ZUNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0QWNjZWxlcmF0ZUNvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSB0aGUgYWNjZWxlcmF0ZSBzdWJyZXNvdXJjZSB0byBzZXQgdGhlIFRyYW5zZmVyIEFjY2VsZXJhdGlvbiBzdGF0ZSBvZiBhbiBleGlzdGluZyBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEFjY2Vzc1BvaW50UG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfUHV0QWNjZXNzUG9pbnRQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhbiBhY2Nlc3MgcG9saWN5IHdpdGggYSBzcGVjaWZpZWQgYWNjZXNzIHBvaW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJhY2Nlc3Nwb2ludFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEFjY291bnRQdWJsaWNBY2Nlc3NCbG9ja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX1B1dFB1YmxpY0FjY2Vzc0Jsb2NrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgb3IgbW9kaWZ5IHRoZSBQdWJsaWNBY2Nlc3NCbG9jayBjb25maWd1cmF0aW9uIGZvciBhbiBBV1MgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRBbmFseXRpY3NDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCBhbiBhbmFseXRpY3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIGJ1Y2tldCwgc3BlY2lmaWVkIGJ5IHRoZSBhbmFseXRpY3MgY29uZmlndXJhdGlvbiBJRFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0QWNsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEFjbC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBwZXJtaXNzaW9ucyBvbiBhbiBleGlzdGluZyBidWNrZXQgdXNpbmcgYWNjZXNzIGNvbnRyb2wgbGlzdHMgKEFDTHMpXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1hY2xcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LWZ1bGwtY29udHJvbFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXJlYWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1yZWFkLWFjcFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXdyaXRlXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtd3JpdGUtYWNwXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0Q09SU1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRDb3JzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIENPUlMgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0TG9nZ2luZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRMb2dnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIGxvZ2dpbmcgcGFyYW1ldGVycyBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0Tm90aWZpY2F0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldE5vdGlmaWNhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVjZWl2ZSBub3RpZmljYXRpb25zIHdoZW4gY2VydGFpbiBldmVudHMgaGFwcGVuIGluIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEJ1Y2tldE9iamVjdExvY2tDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdExvY2tDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBwdXQgT2JqZWN0IExvY2sgY29uZmlndXJhdGlvbiBvbiBhIHNwZWNpZmljIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEJ1Y2tldFBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciByZXBsYWNlIGEgYnVja2V0IHBvbGljeSBvbiBhIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRCdWNrZXRQdWJsaWNBY2Nlc3NCbG9ja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRQdWJsaWNBY2Nlc3NCbG9jay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIG1vZGlmeSB0aGUgUHVibGljQWNjZXNzQmxvY2sgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpYyBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEJ1Y2tldFJlcXVlc3RQYXltZW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldFJlcXVlc3RQYXltZW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIHJlcXVlc3QgcGF5bWVudCBjb25maWd1cmF0aW9uIG9mIGEgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRCdWNrZXRUYWdnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldFRhZ2dpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhIHNldCBvZiB0YWdzIHRvIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0VmVyc2lvbmluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRWZXJzaW9uaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIHZlcnNpb25pbmcgc3RhdGUgb2YgYW4gZXhpc3RpbmcgQW1hem9uIFMzIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0QnVja2V0V2Vic2l0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRXZWJzaXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIGNvbmZpZ3VyYXRpb24gb2YgdGhlIHdlYnNpdGUgdGhhdCBpcyBzcGVjaWZpZWQgaW4gdGhlIHdlYnNpdGUgc3VicmVzb3VyY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEVuY3J5cHRpb25Db25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEVuY3J5cHRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgZW5jcnlwdGlvbiBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gUzMgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiYnVja2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRJbnZlbnRvcnlDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEludmVudG9yeUNvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbiBpbnZlbnRvcnkgY29uZmlndXJhdGlvbiB0byB0aGUgYnVja2V0LCBpZGVudGlmaWVkIGJ5IHRoZSBpbnZlbnRvcnkgSURcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEpvYlRhZ2dpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9QdXRKb2JUYWdnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXBsYWNlIHRhZ3Mgb24gYW4gZXhpc3RpbmcgQW1hem9uIFMzIEJhdGNoIE9wZXJhdGlvbnMgam9iXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJqb2JcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nSm9iUHJpb3JpdHlcIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ0pvYk9wZXJhdGlvblwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCIsXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0TGlmZWN5Y2xlQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRMaWZlY3ljbGVDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgbGlmZWN5Y2xlIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBidWNrZXQgb3IgcmVwbGFjZSBhbiBleGlzdGluZyBsaWZlY3ljbGUgY29uZmlndXJhdGlvblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0TWV0cmljc0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0TWV0cmljc0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCBvciB1cGRhdGUgYSBtZXRyaWNzIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBDbG91ZFdhdGNoIHJlcXVlc3QgbWV0cmljcyBmcm9tIGFuIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJidWNrZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dE9iamVjdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRPYmplY3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbiBvYmplY3QgdG8gYSBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6UmVxdWVzdE9iamVjdFRhZy88a2V5PlwiLFxuICAgICAgICBcInMzOlJlcXVlc3RPYmplY3RUYWdLZXlzXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotYWNsXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb3B5LXNvdXJjZVwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LWZ1bGwtY29udHJvbFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXJlYWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1yZWFkLWFjcFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXdyaXRlXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtd3JpdGUtYWNwXCIsXG4gICAgICAgIFwiczM6eC1hbXotbWV0YWRhdGEtZGlyZWN0aXZlXCIsXG4gICAgICAgIFwiczM6eC1hbXotc2VydmVyLXNpZGUtZW5jcnlwdGlvblwiLFxuICAgICAgICBcInMzOngtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb24tYXdzLWttcy1rZXktaWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1zdG9yYWdlLWNsYXNzXCIsXG4gICAgICAgIFwiczM6eC1hbXotd2Vic2l0ZS1yZWRpcmVjdC1sb2NhdGlvblwiLFxuICAgICAgICBcInMzOm9iamVjdC1sb2NrLW1vZGVcIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1yZXRhaW4tdW50aWwtZGF0ZVwiLFxuICAgICAgICBcInMzOm9iamVjdC1sb2NrLXJlbWFpbmluZy1yZXRlbnRpb24tZGF5c1wiLFxuICAgICAgICBcInMzOm9iamVjdC1sb2NrLWxlZ2FsLWhvbGRcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRPYmplY3RBY2xcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0T2JqZWN0QWNsLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIGFjY2VzcyBjb250cm9sIGxpc3QgKEFDTCkgcGVybWlzc2lvbnMgZm9yIGFuIG9iamVjdCB0aGF0IGFscmVhZHkgZXhpc3RzIGluIGEgYnVja2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1hY2xcIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LWZ1bGwtY29udHJvbFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXJlYWRcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1yZWFkLWFjcFwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXdyaXRlXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtd3JpdGUtYWNwXCIsXG4gICAgICAgIFwiczM6eC1hbXotc3RvcmFnZS1jbGFzc1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dE9iamVjdExlZ2FsSG9sZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRPYmplY3RMZWdhbEhvbGQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFwcGx5IGEgTGVnYWwgSG9sZCBjb25maWd1cmF0aW9uIHRvIHRoZSBzcGVjaWZpZWQgb2JqZWN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCIsXG4gICAgICAgIFwiczM6b2JqZWN0LWxvY2stbGVnYWwtaG9sZFwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dE9iamVjdFJldGVudGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRPYmplY3RSZXRlbnRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHBsYWNlIGFuIE9iamVjdCBSZXRlbnRpb24gY29uZmlndXJhdGlvbiBvbiBhbiBvYmplY3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJvYmplY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBY2NvdW50XCIsXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QXJuXCIsXG4gICAgICAgIFwiczM6QWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1tb2RlXCIsXG4gICAgICAgIFwiczM6b2JqZWN0LWxvY2stcmV0YWluLXVudGlsLWRhdGVcIixcbiAgICAgICAgXCJzMzpvYmplY3QtbG9jay1yZW1haW5pbmctcmV0ZW50aW9uLWRheXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRPYmplY3RUYWdnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdFRhZ2dpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgc3VwcGxpZWQgdGFnLXNldCB0byBhbiBvYmplY3QgdGhhdCBhbHJlYWR5IGV4aXN0cyBpbiBhIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nT2JqZWN0VGFnLzxrZXk+XCIsXG4gICAgICAgIFwiczM6UmVxdWVzdE9iamVjdFRhZy88a2V5PlwiLFxuICAgICAgICBcInMzOlJlcXVlc3RPYmplY3RUYWdLZXlzXCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRPYmplY3RWZXJzaW9uQWNsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdEFjbC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXNlIHRoZSBhY2wgc3VicmVzb3VyY2UgdG8gc2V0IHRoZSBhY2Nlc3MgY29udHJvbCBsaXN0IChBQ0wpIHBlcm1pc3Npb25zIGZvciBhbiBvYmplY3QgdGhhdCBhbHJlYWR5IGV4aXN0cyBpbiBhIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudFwiLFxuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFyblwiLFxuICAgICAgICBcInMzOkFjY2Vzc1BvaW50TmV0d29ya09yaWdpblwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nT2JqZWN0VGFnLzxrZXk+XCIsXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6dmVyc2lvbmlkXCIsXG4gICAgICAgIFwiczM6eC1hbXotYWNsXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1mdWxsLWNvbnRyb2xcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC1yZWFkXCIsXG4gICAgICAgIFwiczM6eC1hbXotZ3JhbnQtcmVhZC1hY3BcIixcbiAgICAgICAgXCJzMzp4LWFtei1ncmFudC13cml0ZVwiLFxuICAgICAgICBcInMzOngtYW16LWdyYW50LXdyaXRlLWFjcFwiLFxuICAgICAgICBcInMzOngtYW16LXN0b3JhZ2UtY2xhc3NcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRPYmplY3RWZXJzaW9uVGFnZ2luZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRPYmplY3RUYWdnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIHN1cHBsaWVkIHRhZy1zZXQgZm9yIGEgc3BlY2lmaWMgdmVyc2lvbiBvZiBhbiBvYmplY3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ09iamVjdFRhZy88a2V5PlwiLFxuICAgICAgICBcInMzOlJlcXVlc3RPYmplY3RUYWcvPGtleT5cIixcbiAgICAgICAgXCJzMzpSZXF1ZXN0T2JqZWN0VGFnS2V5c1wiLFxuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOnZlcnNpb25pZFwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHV0UmVwbGljYXRpb25Db25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldFJlcGxpY2F0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgcmVwbGljYXRpb24gY29uZmlndXJhdGlvbiBvciByZXBsYWNlIGFuIGV4aXN0aW5nIG9uZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ1Y2tldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUmVwbGljYXRlRGVsZXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvc2V0dGluZy1yZXBsLWNvbmZpZy1wZXJtLW92ZXJ2aWV3Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgZGVsZXRlIG1hcmtlcnMgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUmVwbGljYXRlT2JqZWN0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvc2V0dGluZy1yZXBsLWNvbmZpZy1wZXJtLW92ZXJ2aWV3Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgb2JqZWN0cyBhbmQgb2JqZWN0IHRhZ3MgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCIsXG4gICAgICAgIFwiczM6eC1hbXotc2VydmVyLXNpZGUtZW5jcnlwdGlvblwiLFxuICAgICAgICBcInMzOngtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb24tYXdzLWttcy1rZXktaWRcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJSZXBsaWNhdGVUYWdzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvc2V0dGluZy1yZXBsLWNvbmZpZy1wZXJtLW92ZXJ2aWV3Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgb2JqZWN0IHRhZ3MgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib2JqZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJSZXN0b3JlT2JqZWN0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1Jlc3RvcmVPYmplY3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJlc3RvcmUgYW4gYXJjaGl2ZWQgY29weSBvZiBhbiBvYmplY3QgYmFjayBpbnRvIEFtYXpvbiBTM1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm9iamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOkRhdGFBY2Nlc3NQb2ludEFjY291bnRcIixcbiAgICAgICAgXCJzMzpEYXRhQWNjZXNzUG9pbnRBcm5cIixcbiAgICAgICAgXCJzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5cIixcbiAgICAgICAgXCJzMzphdXRoVHlwZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZUFnZVwiLFxuICAgICAgICBcInMzOnNpZ25hdHVyZXZlcnNpb25cIixcbiAgICAgICAgXCJzMzp4LWFtei1jb250ZW50LXNoYTI1NlwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZUpvYlByaW9yaXR5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfVXBkYXRlSm9iUHJpb3JpdHkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgcHJpb3JpdHkgb2YgYW4gZXhpc3Rpbmcgam9iXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiam9iXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6YXV0aFR5cGVcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmVBZ2VcIixcbiAgICAgICAgXCJzMzpzaWduYXR1cmV2ZXJzaW9uXCIsXG4gICAgICAgIFwiczM6eC1hbXotY29udGVudC1zaGEyNTZcIixcbiAgICAgICAgXCJzMzpSZXF1ZXN0Sm9iUHJpb3JpdHlcIixcbiAgICAgICAgXCJzMzpFeGlzdGluZ0pvYlByaW9yaXR5XCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdKb2JPcGVyYXRpb25cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVcGRhdGVKb2JTdGF0dXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9VcGRhdGVKb2JTdGF0dXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgc3RhdHVzIGZvciB0aGUgc3BlY2lmaWVkIGpvYlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImpvYlwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcInMzOmF1dGhUeXBlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJlQWdlXCIsXG4gICAgICAgIFwiczM6c2lnbmF0dXJldmVyc2lvblwiLFxuICAgICAgICBcInMzOngtYW16LWNvbnRlbnQtc2hhMjU2XCIsXG4gICAgICAgIFwiczM6RXhpc3RpbmdKb2JQcmlvcml0eVwiLFxuICAgICAgICBcInMzOkV4aXN0aW5nSm9iT3BlcmF0aW9uXCIsXG4gICAgICAgIFwiczM6Sm9iU3VzcGVuZGVkQ2F1c2VcIlxuICAgICAgXVxuICAgIH1cbiAgfTtcbiAgcHJvdGVjdGVkIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJhY2Nlc3Nwb2ludFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJhY2Nlc3Nwb2ludFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hY2Nlc3MtcG9pbnRzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpzMzoke1JlZ2lvbn06JHtBY2NvdW50fTphY2Nlc3Nwb2ludC8ke0FjY2Vzc1BvaW50TmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJidWNrZXRcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiYnVja2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nQnVja2V0Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpzMzo6OiR7QnVja2V0TmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJvYmplY3RcIjoge1xuICAgICAgXCJuYW1lXCI6IFwib2JqZWN0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nT2JqZWN0cy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06czM6Ojoke0J1Y2tldE5hbWV9LyR7T2JqZWN0TmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJqb2JcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiam9iXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1tYW5hZ2luZy1qb2JzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpzMzoke1JlZ2lvbn06JHtBY2NvdW50fTpqb2IvJHtKb2JJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH1cbiAgfTtcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtzM10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uczMuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWJvcnQgYSBtdWx0aXBhcnQgdXBsb2FkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfQWJvcnRNdWx0aXBhcnRVcGxvYWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWJvcnRNdWx0aXBhcnRVcGxvYWQoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkFib3J0TXVsdGlwYXJ0VXBsb2FkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWxsb3cgY2lyY3VtdmVudGlvbiBvZiBnb3Zlcm5hbmNlLW1vZGUgb2JqZWN0IHJldGVudGlvbiBzZXR0aW5nc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZygpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZ0tleXMoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpBY2woKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZYQW16Q29weVNvdXJjZSgpXG4gICAqIC0gLmlmWEFtekdyYW50RnVsbENvbnRyb2woKVxuICAgKiAtIC5pZlhBbXpHcmFudFJlYWQoKVxuICAgKiAtIC5pZlhBbXpHcmFudFJlYWRBY3AoKVxuICAgKiAtIC5pZlhBbXpHcmFudFdyaXRlKClcbiAgICogLSAuaWZYQW16R3JhbnRXcml0ZUFjcCgpXG4gICAqIC0gLmlmWEFtek1ldGFkYXRhRGlyZWN0aXZlKClcbiAgICogLSAuaWZYQW16U2VydmVyU2lkZUVuY3J5cHRpb24oKVxuICAgKiAtIC5pZlhBbXpTZXJ2ZXJTaWRlRW5jcnlwdGlvbkF3c0ttc0tleUlkKClcbiAgICogLSAuaWZYQW16U3RvcmFnZUNsYXNzKClcbiAgICogLSAuaWZYQW16V2Vic2l0ZVJlZGlyZWN0TG9jYXRpb24oKVxuICAgKiAtIC5pZk9iamVjdExvY2tNb2RlKClcbiAgICogLSAuaWZPYmplY3RMb2NrUmV0YWluVW50aWxEYXRlKClcbiAgICogLSAuaWZPYmplY3RMb2NrUmVtYWluaW5nUmV0ZW50aW9uRGF5cygpXG4gICAqIC0gLmlmT2JqZWN0TG9ja0xlZ2FsSG9sZCgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L29iamVjdC1sb2NrLW1hbmFnaW5nLmh0bWwjb2JqZWN0LWxvY2stbWFuYWdpbmctYnlwYXNzXG4gICAqL1xuICBwdWJsaWMgdG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpCeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGFjY2VzcyBwb2ludFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZMb2NhdGlvbmNvbnN0cmFpbnQoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekFjbCgpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9DcmVhdGVBY2Nlc3NQb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBY2Nlc3NQb2ludCgpIHtcbiAgICB0aGlzLmFkZCgnczM6Q3JlYXRlQWNjZXNzUG9pbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmTG9jYXRpb25jb25zdHJhaW50KClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpBY2woKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZYQW16R3JhbnRGdWxsQ29udHJvbCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZEFjcCgpXG4gICAqIC0gLmlmWEFtekdyYW50V3JpdGUoKVxuICAgKiAtIC5pZlhBbXpHcmFudFdyaXRlQWNwKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0NyZWF0ZUJ1Y2tldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCdWNrZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkNyZWF0ZUJ1Y2tldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBBbWF6b24gUzMgQmF0Y2ggT3BlcmF0aW9ucyBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZSZXF1ZXN0Sm9iUHJpb3JpdHkoKVxuICAgKiAtIC5pZlJlcXVlc3RKb2JPcGVyYXRpb24oKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9DcmVhdGVKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSm9iKCkge1xuICAgIHRoaXMuYWRkKCdzMzpDcmVhdGVKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGFjY2VzcyBwb2ludCBuYW1lZCBpbiB0aGUgVVJJXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9EZWxldGVBY2Nlc3NQb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBY2Nlc3NQb2ludCgpIHtcbiAgICB0aGlzLmFkZCgnczM6RGVsZXRlQWNjZXNzUG9pbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHBvbGljeSBvbiBhIHNwZWNpZmllZCBhY2Nlc3MgcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVsZXRlQWNjZXNzUG9pbnRQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQWNjZXNzUG9pbnRQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlbGV0ZUFjY2Vzc1BvaW50UG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBidWNrZXQgbmFtZWQgaW4gdGhlIFVSSVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlQnVja2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUJ1Y2tldCgpIHtcbiAgICB0aGlzLmFkZCgnczM6RGVsZXRlQnVja2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBwb2xpY3kgb24gYSBzcGVjaWZpZWQgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0RlbGV0ZUJ1Y2tldFBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVCdWNrZXRQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlbGV0ZUJ1Y2tldFBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgd2Vic2l0ZSBjb25maWd1cmF0aW9uIGZvciBhIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlQnVja2V0V2Vic2l0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVCdWNrZXRXZWJzaXRlKCkge1xuICAgIHRoaXMuYWRkKCdzMzpEZWxldGVCdWNrZXRXZWJzaXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhZ3MgZnJvbSBhbiBleGlzdGluZyBBbWF6b24gUzMgQmF0Y2ggT3BlcmF0aW9ucyBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKiAtIC5pZkV4aXN0aW5nSm9iUHJpb3JpdHkoKVxuICAgKiAtIC5pZkV4aXN0aW5nSm9iT3BlcmF0aW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVsZXRlSm9iVGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVKb2JUYWdnaW5nKCkge1xuICAgIHRoaXMuYWRkKCdzMzpEZWxldGVKb2JUYWdnaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRoZSBudWxsIHZlcnNpb24gb2YgYW4gb2JqZWN0IGFuZCBpbnNlcnQgYSBkZWxldGUgbWFya2VyLCB3aGljaCBiZWNvbWVzIHRoZSBjdXJyZW50IHZlcnNpb24gb2YgdGhlIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0RlbGV0ZU9iamVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVPYmplY3QoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlbGV0ZU9iamVjdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSB0aGUgdGFnZ2luZyBzdWJyZXNvdXJjZSB0byByZW1vdmUgdGhlIGVudGlyZSB0YWcgc2V0IGZyb20gdGhlIHNwZWNpZmllZCBvYmplY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkV4aXN0aW5nT2JqZWN0VGFnKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9EZWxldGVPYmplY3RUYWdnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU9iamVjdFRhZ2dpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlbGV0ZU9iamVjdFRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgYSBzcGVjaWZpYyB2ZXJzaW9uIG9mIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlZlcnNpb25pZCgpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlT2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU9iamVjdFZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlbGV0ZU9iamVjdFZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIGVudGlyZSB0YWcgc2V0IGZvciBhIHNwZWNpZmljIHZlcnNpb24gb2YgdGhlIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmRXhpc3RpbmdPYmplY3RUYWcoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlZlcnNpb25pZCgpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfRGVsZXRlT2JqZWN0VGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6RGVsZXRlT2JqZWN0VmVyc2lvblRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgY29uZmlndXJhdGlvbiBwYXJhbWV0ZXJzIGFuZCBzdGF0dXMgZm9yIGEgYmF0Y2ggb3BlcmF0aW9ucyBqb2IuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfRGVzY3JpYmVKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVKb2IoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkRlc2NyaWJlSm9iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXNlcyB0aGUgYWNjZWxlcmF0ZSBzdWJyZXNvdXJjZSB0byByZXR1cm4gdGhlIFRyYW5zZmVyIEFjY2VsZXJhdGlvbiBzdGF0ZSBvZiBhIGJ1Y2tldCwgd2hpY2ggaXMgZWl0aGVyIEVuYWJsZWQgb3IgU3VzcGVuZGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBjb25maWd1cmF0aW9uIGluZm9ybWF0aW9uIGFib3V0IHRoZSBzcGVjaWZpZWQgYWNjZXNzIHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0dldEFjY2Vzc1BvaW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY2Vzc1BvaW50KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRBY2Nlc3NQb2ludCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybnMgdGhlIGFjY2VzcyBwb2ludCBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgYWNjZXNzIHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0dldEFjY2Vzc1BvaW50UG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY2Vzc1BvaW50UG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRBY2Nlc3NQb2ludFBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgcG9saWN5IHN0YXR1cyBmb3IgYSBzcGVjaWZpYyBhY2Nlc3MgcG9pbnQgcG9saWN5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRBY2Nlc3NQb2ludFBvbGljeVN0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBQdWJsaWNBY2Nlc3NCbG9jayBjb25maWd1cmF0aW9uIGZvciBhbiBBV1MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0dldFB1YmxpY0FjY2Vzc0Jsb2NrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY291bnRQdWJsaWNBY2Nlc3NCbG9jaygpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QWNjb3VudFB1YmxpY0FjY2Vzc0Jsb2NrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGFuYWx5dGljcyBjb25maWd1cmF0aW9uIGZyb20gYW4gQW1hem9uIFMzIGJ1Y2tldCwgaWRlbnRpZmllZCBieSB0aGUgYW5hbHl0aWNzIGNvbmZpZ3VyYXRpb24gSURcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0QW5hbHl0aWNzQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRBbmFseXRpY3NDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRBbmFseXRpY3NDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXNlIHRoZSBhY2wgc3VicmVzb3VyY2UgdG8gcmV0dXJuIHRoZSBhY2Nlc3MgY29udHJvbCBsaXN0IChBQ0wpIG9mIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0QWNsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldEFjbCgpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QnVja2V0QWNsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBDT1JTIGNvbmZpZ3VyYXRpb24gaW5mb3JtYXRpb24gc2V0IGZvciBhbiBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldENvcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVja2V0Q09SUygpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QnVja2V0Q09SUycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgUmVnaW9uIHRoYXQgYW4gQW1hem9uIFMzIGJ1Y2tldCByZXNpZGVzIGluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0TG9jYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVja2V0TG9jYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEJ1Y2tldExvY2F0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBsb2dnaW5nIHN0YXR1cyBvZiBhbiBBbWF6b24gUzMgYnVja2V0IGFuZCB0aGUgcGVybWlzc2lvbnMgdXNlcnMgaGF2ZSB0byB2aWV3IG9yIG1vZGlmeSB0aGF0IHN0YXR1c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRMb2dnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldExvZ2dpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEJ1Y2tldExvZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIG5vdGlmaWNhdGlvbiBjb25maWd1cmF0aW9uIG9mIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0Tm90aWZpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldE5vdGlmaWNhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QnVja2V0Tm90aWZpY2F0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBPYmplY3QgTG9jayBjb25maWd1cmF0aW9uIG9mIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCdWNrZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QnVja2V0T2JqZWN0TG9ja0NvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHBvbGljeSBvZiB0aGUgc3BlY2lmaWVkIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVja2V0UG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRCdWNrZXRQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgcG9saWN5IHN0YXR1cyBmb3IgYSBzcGVjaWZpYyBBbWF6b24gUzMgYnVja2V0LCB3aGljaCBpbmRpY2F0ZXMgd2hldGhlciB0aGUgYnVja2V0IGlzIHB1YmxpY1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRQb2xpY3lTdGF0dXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVja2V0UG9saWN5U3RhdHVzKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRCdWNrZXRQb2xpY3lTdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgUHVibGljQWNjZXNzQmxvY2sgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRQdWJsaWNBY2Nlc3NCbG9jay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCdWNrZXRQdWJsaWNBY2Nlc3NCbG9jaygpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0QnVja2V0UHVibGljQWNjZXNzQmxvY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHJlcXVlc3QgcGF5bWVudCBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldFJlcXVlc3RQYXltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldFJlcXVlc3RQYXltZW50KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRCdWNrZXRSZXF1ZXN0UGF5bWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgdGFnIHNldCBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRUYWdnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldFRhZ2dpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEJ1Y2tldFRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHZlcnNpb25pbmcgc3RhdGUgb2YgYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRWZXJzaW9uaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldFZlcnNpb25pbmcoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEJ1Y2tldFZlcnNpb25pbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHdlYnNpdGUgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRXZWJzaXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJ1Y2tldFdlYnNpdGUoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEJ1Y2tldFdlYnNpdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGRlZmF1bHQgZW5jcnlwdGlvbiBjb25maWd1cmF0aW9uIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0QnVja2V0RW5jcnlwdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRFbmNyeXB0aW9uQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYW4gaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24gZnJvbSBhbiBBbWF6b24gUzMgYnVja2V0LCBpZGVudGlmaWVkIGJ5IHRoZSBpbnZlbnRvcnkgY29uZmlndXJhdGlvbiBJRFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRJbnZlbnRvcnlDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEludmVudG9yeUNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldEludmVudG9yeUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHRhZyBzZXQgb2YgYW4gZXhpc3RpbmcgQW1hem9uIFMzIEJhdGNoIE9wZXJhdGlvbnMgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfR2V0Sm9iVGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRKb2JUYWdnaW5nKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRKb2JUYWdnaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBsaWZlY3ljbGUgY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiBzZXQgb24gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRMaWZlY3ljbGVDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBtZXRyaWNzIGNvbmZpZ3VyYXRpb24gZnJvbSBhbiBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldEJ1Y2tldE1ldHJpY3NDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1ldHJpY3NDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRNZXRyaWNzQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIG9iamVjdHMgZnJvbSBBbWF6b24gUzNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkV4aXN0aW5nT2JqZWN0VGFnKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0T2JqZWN0KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRPYmplY3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGFjY2VzcyBjb250cm9sIGxpc3QgKEFDTCkgb2YgYW4gb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0QWNsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9iamVjdEFjbCgpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0QWNsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIG9iamVjdCdzIGN1cnJlbnQgTGVnYWwgSG9sZCBzdGF0dXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdExlZ2FsSG9sZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRPYmplY3RMZWdhbEhvbGQoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldE9iamVjdExlZ2FsSG9sZCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSByZXRlbnRpb24gc2V0dGluZ3MgZm9yIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfR2V0T2JqZWN0UmV0ZW50aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9iamVjdFJldGVudGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0UmV0ZW50aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB0YWcgc2V0IG9mIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmRXhpc3RpbmdPYmplY3RUYWcoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0dldE9iamVjdFRhZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0T2JqZWN0VGFnZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0VGFnZ2luZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0b3JyZW50IGZpbGVzIGZyb20gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3RUb3JyZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9iamVjdFRvcnJlbnQoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldE9iamVjdFRvcnJlbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIHNwZWNpZmljIHZlcnNpb24gb2YgYW4gb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmVmVyc2lvbmlkKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0T2JqZWN0VmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0VmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgYWNjZXNzIGNvbnRyb2wgbGlzdCAoQUNMKSBvZiBhIHNwZWNpZmljIG9iamVjdCB2ZXJzaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmVmVyc2lvbmlkKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3RBY2wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0T2JqZWN0VmVyc2lvbkFjbCgpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0VmVyc2lvbkFjbCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlcGxpY2F0ZSBib3RoIHVuZW5jcnlwdGVkIG9iamVjdHMgYW5kIG9iamVjdHMgZW5jcnlwdGVkIHdpdGggU1NFLVMzIG9yIFNTRS1LTVNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKi9cbiAgcHVibGljIHRvR2V0T2JqZWN0VmVyc2lvbkZvclJlcGxpY2F0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRPYmplY3RWZXJzaW9uRm9yUmVwbGljYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIHRhZyBzZXQgZm9yIGEgc3BlY2lmaWMgdmVyc2lvbiBvZiB0aGUgb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmVmVyc2lvbmlkKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqL1xuICBwdWJsaWMgdG9HZXRPYmplY3RWZXJzaW9uVGFnZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6R2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgVG9ycmVudCBmaWxlcyBhYm91dCBhIGRpZmZlcmVudCB2ZXJzaW9uIHVzaW5nIHRoZSB2ZXJzaW9uSWQgc3VicmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmVmVyc2lvbmlkKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRPYmplY3RUb3JyZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KCkge1xuICAgIHRoaXMuYWRkKCdzMzpHZXRPYmplY3RWZXJzaW9uVG9ycmVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcmVwbGljYXRpb24gY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbiBzZXQgb24gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9HZXRCdWNrZXRSZXBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkdldFJlcGxpY2F0aW9uQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWNjZXNzIHBvaW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0xpc3RBY2Nlc3NQb2ludHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFjY2Vzc1BvaW50cygpIHtcbiAgICB0aGlzLmFkZCgnczM6TGlzdEFjY2Vzc1BvaW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIGJ1Y2tldHMgb3duZWQgYnkgdGhlIGF1dGhlbnRpY2F0ZWQgc2VuZGVyIG9mIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0xpc3RCdWNrZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBbGxNeUJ1Y2tldHMoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkxpc3RBbGxNeUJ1Y2tldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHNvbWUgb3IgYWxsIG9mIHRoZSBvYmplY3RzIGluIGFuIEFtYXpvbiBTMyBidWNrZXQgKHVwIHRvIDEwMDApXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmRGVsaW1pdGVyKClcbiAgICogLSAuaWZNYXhLZXlzKClcbiAgICogLSAuaWZQcmVmaXgoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfTGlzdE9iamVjdHNWMi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QnVja2V0KCkge1xuICAgIHRoaXMuYWRkKCdzMzpMaXN0QnVja2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbi1wcm9ncmVzcyBtdWx0aXBhcnQgdXBsb2Fkc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfTGlzdE11bHRpcGFydFVwbG9hZHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJ1Y2tldE11bHRpcGFydFVwbG9hZHMoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkxpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBtZXRhZGF0YSBhYm91dCBhbGwgdGhlIHZlcnNpb25zIG9mIG9iamVjdHMgaW4gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZkRlbGltaXRlcigpXG4gICAqIC0gLmlmTWF4S2V5cygpXG4gICAqIC0gLmlmUHJlZml4KClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0xpc3RPYmplY3RWZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QnVja2V0VmVyc2lvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkxpc3RCdWNrZXRWZXJzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY3VycmVudCBqb2JzIGFuZCBqb2JzIHRoYXQgaGF2ZSBlbmRlZCByZWNlbnRseVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX0xpc3RKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RKb2JzKCkge1xuICAgIHRoaXMuYWRkKCdzMzpMaXN0Sm9icycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHBhcnRzIHRoYXQgaGF2ZSBiZWVuIHVwbG9hZGVkIGZvciBhIHNwZWNpZmljIG11bHRpcGFydCB1cGxvYWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX0xpc3RQYXJ0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOkxpc3RNdWx0aXBhcnRVcGxvYWRQYXJ0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNoYW5nZSByZXBsaWNhIG93bmVyc2hpcFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L3JlcGxpY2F0aW9uLWNoYW5nZS1vd25lci5odG1sI3JlcGwtb3duZXJzaGlwLWFkZC1yb2xlLXBlcm1pc3Npb25cbiAgICovXG4gIHB1YmxpYyB0b09iamVjdE93bmVyT3ZlcnJpZGVUb0J1Y2tldE93bmVyKCkge1xuICAgIHRoaXMuYWRkKCdzMzpPYmplY3RPd25lck92ZXJyaWRlVG9CdWNrZXRPd25lcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSB0aGUgYWNjZWxlcmF0ZSBzdWJyZXNvdXJjZSB0byBzZXQgdGhlIFRyYW5zZmVyIEFjY2VsZXJhdGlvbiBzdGF0ZSBvZiBhbiBleGlzdGluZyBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEFjY2VsZXJhdGVDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhbiBhY2Nlc3MgcG9saWN5IHdpdGggYSBzcGVjaWZpZWQgYWNjZXNzIHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9jb250cm9sX1B1dEFjY2Vzc1BvaW50UG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEFjY2Vzc1BvaW50UG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRBY2Nlc3NQb2ludFBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBvciBtb2RpZnkgdGhlIFB1YmxpY0FjY2Vzc0Jsb2NrIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfUHV0UHVibGljQWNjZXNzQmxvY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QWNjb3VudFB1YmxpY0FjY2Vzc0Jsb2NrKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRBY2NvdW50UHVibGljQWNjZXNzQmxvY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgYW4gYW5hbHl0aWNzIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBidWNrZXQsIHNwZWNpZmllZCBieSB0aGUgYW5hbHl0aWNzIGNvbmZpZ3VyYXRpb24gSURcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgcGVybWlzc2lvbnMgb24gYW4gZXhpc3RpbmcgYnVja2V0IHVzaW5nIGFjY2VzcyBjb250cm9sIGxpc3RzIChBQ0xzKVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16QWNsKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqIC0gLmlmWEFtekdyYW50RnVsbENvbnRyb2woKVxuICAgKiAtIC5pZlhBbXpHcmFudFJlYWQoKVxuICAgKiAtIC5pZlhBbXpHcmFudFJlYWRBY3AoKVxuICAgKiAtIC5pZlhBbXpHcmFudFdyaXRlKClcbiAgICogLSAuaWZYQW16R3JhbnRXcml0ZUFjcCgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXRBY2wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QnVja2V0QWNsKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRBY2wnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdGhlIENPUlMgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0Q29ycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCdWNrZXRDT1JTKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRDT1JTJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBsb2dnaW5nIHBhcmFtZXRlcnMgZm9yIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldExvZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QnVja2V0TG9nZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0QnVja2V0TG9nZ2luZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlY2VpdmUgbm90aWZpY2F0aW9ucyB3aGVuIGNlcnRhaW4gZXZlbnRzIGhhcHBlbiBpbiBhbiBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRCdWNrZXROb3RpZmljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QnVja2V0Tm90aWZpY2F0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXROb3RpZmljYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBwdXQgT2JqZWN0IExvY2sgY29uZmlndXJhdGlvbiBvbiBhIHNwZWNpZmljIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9QdXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCdWNrZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0QnVja2V0T2JqZWN0TG9ja0NvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgb3IgcmVwbGFjZSBhIGJ1Y2tldCBwb2xpY3kgb24gYSBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0UG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEJ1Y2tldFBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0QnVja2V0UG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIG1vZGlmeSB0aGUgUHVibGljQWNjZXNzQmxvY2sgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpYyBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dFB1YmxpY0FjY2Vzc0Jsb2NrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRQdWJsaWNBY2Nlc3NCbG9jaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgcmVxdWVzdCBwYXltZW50IGNvbmZpZ3VyYXRpb24gb2YgYSBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldFJlcXVlc3RQYXltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEJ1Y2tldFJlcXVlc3RQYXltZW50KCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRSZXF1ZXN0UGF5bWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhIHNldCBvZiB0YWdzIHRvIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0VGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCdWNrZXRUYWdnaW5nKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRUYWdnaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSB2ZXJzaW9uaW5nIHN0YXRlIG9mIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldFZlcnNpb25pbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QnVja2V0VmVyc2lvbmluZygpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0QnVja2V0VmVyc2lvbmluZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgY29uZmlndXJhdGlvbiBvZiB0aGUgd2Vic2l0ZSB0aGF0IGlzIHNwZWNpZmllZCBpbiB0aGUgd2Vic2l0ZSBzdWJyZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0V2Vic2l0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCdWNrZXRXZWJzaXRlKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRCdWNrZXRXZWJzaXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBlbmNyeXB0aW9uIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dEJ1Y2tldEVuY3J5cHRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOlB1dEVuY3J5cHRpb25Db25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGFuIGludmVudG9yeSBjb25maWd1cmF0aW9uIHRvIHRoZSBidWNrZXQsIGlkZW50aWZpZWQgYnkgdGhlIGludmVudG9yeSBJRFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0SW52ZW50b3J5Q29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRJbnZlbnRvcnlDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRJbnZlbnRvcnlDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVwbGFjZSB0YWdzIG9uIGFuIGV4aXN0aW5nIEFtYXpvbiBTMyBCYXRjaCBPcGVyYXRpb25zIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqIC0gLmlmRXhpc3RpbmdKb2JQcmlvcml0eSgpXG4gICAqIC0gLmlmRXhpc3RpbmdKb2JPcGVyYXRpb24oKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfY29udHJvbF9QdXRKb2JUYWdnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEpvYlRhZ2dpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOlB1dEpvYlRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgbGlmZWN5Y2xlIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBidWNrZXQgb3IgcmVwbGFjZSBhbiBleGlzdGluZyBsaWZlY3ljbGUgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0TGlmZWN5Y2xlQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRMaWZlY3ljbGVDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRMaWZlY3ljbGVDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IG9yIHVwZGF0ZSBhIG1ldHJpY3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIENsb3VkV2F0Y2ggcmVxdWVzdCBtZXRyaWNzIGZyb20gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0TWV0cmljc0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0TWV0cmljc0NvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3MzOlB1dE1ldHJpY3NDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGFuIG9iamVjdCB0byBhIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZlJlcXVlc3RPYmplY3RUYWcoKVxuICAgKiAtIC5pZlJlcXVlc3RPYmplY3RUYWdLZXlzKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16QWNsKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqIC0gLmlmWEFtekNvcHlTb3VyY2UoKVxuICAgKiAtIC5pZlhBbXpHcmFudEZ1bGxDb250cm9sKClcbiAgICogLSAuaWZYQW16R3JhbnRSZWFkKClcbiAgICogLSAuaWZYQW16R3JhbnRSZWFkQWNwKClcbiAgICogLSAuaWZYQW16R3JhbnRXcml0ZSgpXG4gICAqIC0gLmlmWEFtekdyYW50V3JpdGVBY3AoKVxuICAgKiAtIC5pZlhBbXpNZXRhZGF0YURpcmVjdGl2ZSgpXG4gICAqIC0gLmlmWEFtelNlcnZlclNpZGVFbmNyeXB0aW9uKClcbiAgICogLSAuaWZYQW16U2VydmVyU2lkZUVuY3J5cHRpb25Bd3NLbXNLZXlJZCgpXG4gICAqIC0gLmlmWEFtelN0b3JhZ2VDbGFzcygpXG4gICAqIC0gLmlmWEFteldlYnNpdGVSZWRpcmVjdExvY2F0aW9uKClcbiAgICogLSAuaWZPYmplY3RMb2NrTW9kZSgpXG4gICAqIC0gLmlmT2JqZWN0TG9ja1JldGFpblVudGlsRGF0ZSgpXG4gICAqIC0gLmlmT2JqZWN0TG9ja1JlbWFpbmluZ1JldGVudGlvbkRheXMoKVxuICAgKiAtIC5pZk9iamVjdExvY2tMZWdhbEhvbGQoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0T2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dE9iamVjdCgpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0T2JqZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBhY2Nlc3MgY29udHJvbCBsaXN0IChBQ0wpIHBlcm1pc3Npb25zIGZvciBhbiBvYmplY3QgdGhhdCBhbHJlYWR5IGV4aXN0cyBpbiBhIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmRXhpc3RpbmdPYmplY3RUYWcoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpBY2woKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZYQW16R3JhbnRGdWxsQ29udHJvbCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZEFjcCgpXG4gICAqIC0gLmlmWEFtekdyYW50V3JpdGUoKVxuICAgKiAtIC5pZlhBbXpHcmFudFdyaXRlQWNwKClcbiAgICogLSAuaWZYQW16U3RvcmFnZUNsYXNzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdEFjbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRPYmplY3RBY2woKSB7XG4gICAgdGhpcy5hZGQoJ3MzOlB1dE9iamVjdEFjbCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFwcGx5IGEgTGVnYWwgSG9sZCBjb25maWd1cmF0aW9uIHRvIHRoZSBzcGVjaWZpZWQgb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKiAtIC5pZk9iamVjdExvY2tMZWdhbEhvbGQoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0T2JqZWN0TGVnYWxIb2xkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dE9iamVjdExlZ2FsSG9sZCgpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0T2JqZWN0TGVnYWxIb2xkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcGxhY2UgYW4gT2JqZWN0IFJldGVudGlvbiBjb25maWd1cmF0aW9uIG9uIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZPYmplY3RMb2NrTW9kZSgpXG4gICAqIC0gLmlmT2JqZWN0TG9ja1JldGFpblVudGlsRGF0ZSgpXG4gICAqIC0gLmlmT2JqZWN0TG9ja1JlbWFpbmluZ1JldGVudGlvbkRheXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0T2JqZWN0UmV0ZW50aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dE9iamVjdFJldGVudGlvbigpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0T2JqZWN0UmV0ZW50aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBzdXBwbGllZCB0YWctc2V0IHRvIGFuIG9iamVjdCB0aGF0IGFscmVhZHkgZXhpc3RzIGluIGEgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZygpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZ0tleXMoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdFRhZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0T2JqZWN0VGFnZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0T2JqZWN0VGFnZ2luZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVzZSB0aGUgYWNsIHN1YnJlc291cmNlIHRvIHNldCB0aGUgYWNjZXNzIGNvbnRyb2wgbGlzdCAoQUNMKSBwZXJtaXNzaW9ucyBmb3IgYW4gb2JqZWN0IHRoYXQgYWxyZWFkeSBleGlzdHMgaW4gYSBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QWNjb3VudCgpXG4gICAqIC0gLmlmRGF0YUFjY2Vzc1BvaW50QXJuKClcbiAgICogLSAuaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4oKVxuICAgKiAtIC5pZkV4aXN0aW5nT2JqZWN0VGFnKClcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZWZXJzaW9uaWQoKVxuICAgKiAtIC5pZlhBbXpBY2woKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZYQW16R3JhbnRGdWxsQ29udHJvbCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZCgpXG4gICAqIC0gLmlmWEFtekdyYW50UmVhZEFjcCgpXG4gICAqIC0gLmlmWEFtekdyYW50V3JpdGUoKVxuICAgKiAtIC5pZlhBbXpHcmFudFdyaXRlQWNwKClcbiAgICogLSAuaWZYQW16U3RvcmFnZUNsYXNzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX1B1dE9iamVjdEFjbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRPYmplY3RWZXJzaW9uQWNsKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRPYmplY3RWZXJzaW9uQWNsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBzdXBwbGllZCB0YWctc2V0IGZvciBhIHNwZWNpZmljIHZlcnNpb24gb2YgYW4gb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFjY291bnQoKVxuICAgKiAtIC5pZkRhdGFBY2Nlc3NQb2ludEFybigpXG4gICAqIC0gLmlmQWNjZXNzUG9pbnROZXR3b3JrT3JpZ2luKClcbiAgICogLSAuaWZFeGlzdGluZ09iamVjdFRhZygpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZygpXG4gICAqIC0gLmlmUmVxdWVzdE9iamVjdFRhZ0tleXMoKVxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlZlcnNpb25pZCgpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0T2JqZWN0VGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRPYmplY3RWZXJzaW9uVGFnZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnczM6UHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgcmVwbGljYXRpb24gY29uZmlndXJhdGlvbiBvciByZXBsYWNlIGFuIGV4aXN0aW5nIG9uZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUHV0QnVja2V0UmVwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0UmVwbGljYXRpb25Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdzMzpQdXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgZGVsZXRlIG1hcmtlcnMgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9zZXR0aW5nLXJlcGwtY29uZmlnLXBlcm0tb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVwbGljYXRlRGVsZXRlKCkge1xuICAgIHRoaXMuYWRkKCdzMzpSZXBsaWNhdGVEZWxldGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgb2JqZWN0cyBhbmQgb2JqZWN0IHRhZ3MgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKiAtIC5pZlhBbXpTZXJ2ZXJTaWRlRW5jcnlwdGlvbigpXG4gICAqIC0gLmlmWEFtelNlcnZlclNpZGVFbmNyeXB0aW9uQXdzS21zS2V5SWQoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9zZXR0aW5nLXJlcGwtY29uZmlnLXBlcm0tb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVwbGljYXRlT2JqZWN0KCkge1xuICAgIHRoaXMuYWRkKCdzMzpSZXBsaWNhdGVPYmplY3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXBsaWNhdGUgb2JqZWN0IHRhZ3MgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L3NldHRpbmctcmVwbC1jb25maWctcGVybS1vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXBsaWNhdGVUYWdzKCkge1xuICAgIHRoaXMuYWRkKCdzMzpSZXBsaWNhdGVUYWdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzdG9yZSBhbiBhcmNoaXZlZCBjb3B5IG9mIGFuIG9iamVjdCBiYWNrIGludG8gQW1hem9uIFMzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KClcbiAgICogLSAuaWZEYXRhQWNjZXNzUG9pbnRBcm4oKVxuICAgKiAtIC5pZkFjY2Vzc1BvaW50TmV0d29ya09yaWdpbigpXG4gICAqIC0gLmlmQXV0aFR5cGUoKVxuICAgKiAtIC5pZlNpZ25hdHVyZUFnZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJldmVyc2lvbigpXG4gICAqIC0gLmlmWEFtekNvbnRlbnRTaGEyNTYoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9BUElfUmVzdG9yZU9iamVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXN0b3JlT2JqZWN0KCkge1xuICAgIHRoaXMuYWRkKCdzMzpSZXN0b3JlT2JqZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBwcmlvcml0eSBvZiBhbiBleGlzdGluZyBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF1dGhUeXBlKClcbiAgICogLSAuaWZTaWduYXR1cmVBZ2UoKVxuICAgKiAtIC5pZlNpZ25hdHVyZXZlcnNpb24oKVxuICAgKiAtIC5pZlhBbXpDb250ZW50U2hhMjU2KClcbiAgICogLSAuaWZSZXF1ZXN0Sm9iUHJpb3JpdHkoKVxuICAgKiAtIC5pZkV4aXN0aW5nSm9iUHJpb3JpdHkoKVxuICAgKiAtIC5pZkV4aXN0aW5nSm9iT3BlcmF0aW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfVXBkYXRlSm9iUHJpb3JpdHkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSm9iUHJpb3JpdHkoKSB7XG4gICAgdGhpcy5hZGQoJ3MzOlVwZGF0ZUpvYlByaW9yaXR5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBzdGF0dXMgZm9yIHRoZSBzcGVjaWZpZWQgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBdXRoVHlwZSgpXG4gICAqIC0gLmlmU2lnbmF0dXJlQWdlKClcbiAgICogLSAuaWZTaWduYXR1cmV2ZXJzaW9uKClcbiAgICogLSAuaWZYQW16Q29udGVudFNoYTI1NigpXG4gICAqIC0gLmlmRXhpc3RpbmdKb2JQcmlvcml0eSgpXG4gICAqIC0gLmlmRXhpc3RpbmdKb2JPcGVyYXRpb24oKVxuICAgKiAtIC5pZkpvYlN1c3BlbmRlZENhdXNlKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvQVBJX2NvbnRyb2xfVXBkYXRlSm9iU3RhdHVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUpvYlN0YXR1cygpIHtcbiAgICB0aGlzLmFkZCgnczM6VXBkYXRlSm9iU3RhdHVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYWNjZXNzcG9pbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hY2Nlc3MtcG9pbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFjY2Vzc1BvaW50TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhY2Nlc3NQb2ludE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25BY2Nlc3Nwb2ludChhY2Nlc3NQb2ludE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06czM6JHtSZWdpb259OiR7QWNjb3VudH06YWNjZXNzcG9pbnQvJHtBY2Nlc3NQb2ludE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2Nlc3NQb2ludE5hbWV9JywgYWNjZXNzUG9pbnROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBidWNrZXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ0J1Y2tldC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBidWNrZXROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGJ1Y2tldE5hbWUuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQnVja2V0KGJ1Y2tldE5hbWU6IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnMzOjo6JHtCdWNrZXROYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QnVja2V0TmFtZX0nLCBidWNrZXROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG9iamVjdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nT2JqZWN0cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBidWNrZXROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGJ1Y2tldE5hbWUuXG4gICAqIEBwYXJhbSBvYmplY3ROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG9iamVjdE5hbWUuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uT2JqZWN0KGJ1Y2tldE5hbWU6IHN0cmluZywgb2JqZWN0TmFtZTogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06czM6Ojoke0J1Y2tldE5hbWV9LyR7T2JqZWN0TmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0J1Y2tldE5hbWV9JywgYnVja2V0TmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7T2JqZWN0TmFtZX0nLCBvYmplY3ROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1tYW5hZ2luZy1qb2JzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uSm9iKGpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnMzOiR7UmVnaW9ufToke0FjY291bnR9OmpvYi8ke0pvYklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Sm9iSWR9Jywgam9iSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUpvYigpXG4gICAqIC0gLnRvUHV0Sm9iVGFnZ2luZygpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSm9iKClcbiAgICogLSAudG9QdXRKb2JUYWdnaW5nKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgbmV0d29yayBvcmlnaW4gKEludGVybmV0IG9yIFZQQylcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvY3JlYXRpbmctYWNjZXNzLXBvaW50cy5odG1sI2FjY2Vzcy1wb2ludHMtcG9saWNpZXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0Fib3J0TXVsdGlwYXJ0VXBsb2FkKClcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9EZWxldGVPYmplY3QoKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9MaXN0QnVja2V0KClcbiAgICogLSAudG9MaXN0QnVja2V0TXVsdGlwYXJ0VXBsb2FkcygpXG4gICAqIC0gLnRvTGlzdEJ1Y2tldFZlcnNpb25zKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b1B1dEFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9SZXN0b3JlT2JqZWN0KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW4odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpBY2Nlc3NQb2ludE5ldHdvcmtPcmlnaW5gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgQVdTIEFjY291bnQgSUQgdGhhdCBvd25zIHRoZSBhY2Nlc3MgcG9pbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvY3JlYXRpbmctYWNjZXNzLXBvaW50cy5odG1sI2FjY2Vzcy1wb2ludHMtcG9saWNpZXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0Fib3J0TXVsdGlwYXJ0VXBsb2FkKClcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9EZWxldGVPYmplY3QoKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9MaXN0QnVja2V0KClcbiAgICogLSAudG9MaXN0QnVja2V0TXVsdGlwYXJ0VXBsb2FkcygpXG4gICAqIC0gLnRvTGlzdEJ1Y2tldFZlcnNpb25zKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b1B1dEFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9SZXN0b3JlT2JqZWN0KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZEYXRhQWNjZXNzUG9pbnRBY2NvdW50KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6RGF0YUFjY2Vzc1BvaW50QWNjb3VudGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGFuIGFjY2VzcyBwb2ludCBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQWJvcnRNdWx0aXBhcnRVcGxvYWQoKVxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9EZWxldGVBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdCgpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VmVyc2lvbigpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50KClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludFBvbGljeSgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3lTdGF0dXMoKVxuICAgKiAtIC50b0dldE9iamVjdCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0QWNsKClcbiAgICogLSAudG9HZXRPYmplY3RMZWdhbEhvbGQoKVxuICAgKiAtIC50b0dldE9iamVjdFJldGVudGlvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvbkFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzKClcbiAgICogLSAudG9MaXN0QnVja2V0VmVyc2lvbnMoKVxuICAgKiAtIC50b0xpc3RNdWx0aXBhcnRVcGxvYWRQYXJ0cygpXG4gICAqIC0gLnRvUHV0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0QWNsKClcbiAgICogLSAudG9QdXRPYmplY3RMZWdhbEhvbGQoKVxuICAgKiAtIC50b1B1dE9iamVjdFJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvbkFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b1Jlc3RvcmVPYmplY3QoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkRhdGFBY2Nlc3NQb2ludEFybih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOkRhdGFBY2Nlc3NQb2ludEFybmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIHRvIHVwZGF0aW5nIHRoZSBqb2IgcHJpb3JpdHkgYnkgb3BlcmF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1qb2ItdGFncy1leGFtcGxlcy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVKb2JUYWdnaW5nKClcbiAgICogLSAudG9QdXRKb2JUYWdnaW5nKClcbiAgICogLSAudG9VcGRhdGVKb2JQcmlvcml0eSgpXG4gICAqIC0gLnRvVXBkYXRlSm9iU3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZFeGlzdGluZ0pvYk9wZXJhdGlvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOkV4aXN0aW5nSm9iT3BlcmF0aW9uYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgdG8gY2FuY2VsbGluZyBleGlzdGluZyBqb2JzIGJ5IHByaW9yaXR5IHJhbmdlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1qb2ItdGFncy1leGFtcGxlcy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVKb2JUYWdnaW5nKClcbiAgICogLSAudG9QdXRKb2JUYWdnaW5nKClcbiAgICogLSAudG9VcGRhdGVKb2JQcmlvcml0eSgpXG4gICAqIC0gLnRvVXBkYXRlSm9iU3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbbnVtZXJpYyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfTnVtZXJpYykuICoqRGVmYXVsdDoqKiBgTnVtZXJpY0VxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZkV4aXN0aW5nSm9iUHJpb3JpdHkodmFsdWU6IG51bWJlciB8IG51bWJlcltdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpFeGlzdGluZ0pvYlByaW9yaXR5YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdOdW1lcmljRXF1YWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhhdCBhbiBleGlzdGluZyBvYmplY3QgdGFnIGhhcyBhIHNwZWNpZmljIHRhZyBrZXkgYW5kIHZhbHVlLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9vYmplY3QtdGFnZ2luZy5odG1sI3RhZ2dpbmctYW5kLXBvbGljaWVzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVPYmplY3RUYWdnaW5nKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqIC0gLnRvR2V0T2JqZWN0KClcbiAgICogLSAudG9HZXRPYmplY3RBY2woKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9QdXRPYmplY3RBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICpcbiAgICogQHBhcmFtIGtleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkV4aXN0aW5nT2JqZWN0VGFnKGtleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOkV4aXN0aW5nT2JqZWN0VGFnLyR7IGtleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgdG8gY2FuY2VsbGluZyBzdXNwZW5kZWQgam9icyBieSBhIHNwZWNpZmljIGpvYiBzdXNwZW5kZWQgY2F1c2UgKGZvciBleGFtcGxlLCBBV0FJVElOR19DT05GSVJNQVRJT04pXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1qb2ItdGFncy1leGFtcGxlcy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9VcGRhdGVKb2JTdGF0dXMoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkpvYlN1c3BlbmRlZENhdXNlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6Sm9iU3VzcGVuZGVkQ2F1c2VgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHNwZWNpZmljIFJlZ2lvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hbWF6b24tczMtcG9saWN5LWtleXMuaHRtbCNjb25kaXRpb24ta2V5LWJ1Y2tldC1vcHMtMVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkxvY2F0aW9uQ29uc3RyYWludCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOkxvY2F0aW9uQ29uc3RyYWludGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIHRvIGNyZWF0aW5nIGpvYnMgYnkgb3BlcmF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2JhdGNoLW9wcy1qb2ItdGFncy1leGFtcGxlcy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVKb2IoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RKb2JPcGVyYXRpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpSZXF1ZXN0Sm9iT3BlcmF0aW9uYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgdG8gY3JlYXRpbmcgbmV3IGpvYnMgYnkgcHJpb3JpdHkgcmFuZ2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvYmF0Y2gtb3BzLWpvYi10YWdzLWV4YW1wbGVzLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUpvYigpXG4gICAqIC0gLnRvVXBkYXRlSm9iUHJpb3JpdHkoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtudW1lcmljIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19OdW1lcmljKS4gKipEZWZhdWx0OioqIGBOdW1lcmljRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdEpvYlByaW9yaXR5KHZhbHVlOiBudW1iZXIgfCBudW1iZXJbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6UmVxdWVzdEpvYlByaW9yaXR5YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdOdW1lcmljRXF1YWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVzdHJpY3RzIHRoZSB0YWcga2V5cyBhbmQgdmFsdWVzIGFsbG93ZWQgb24gb2JqZWN0c1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9vYmplY3QtdGFnZ2luZy5odG1sI3RhZ2dpbmctYW5kLXBvbGljaWVzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICpcbiAgICogQHBhcmFtIGtleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RPYmplY3RUYWcoa2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6UmVxdWVzdE9iamVjdFRhZy8keyBrZXkgfWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlc3RyaWN0cyB0aGUgdGFnIGtleXMgYWxsb3dlZCBvbiBvYmplY3RzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L29iamVjdC10YWdnaW5nLmh0bWwjdGFnZ2luZy1hbmQtcG9saWNpZXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RPYmplY3RUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6UmVxdWVzdE9iamVjdFRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHNwZWNpZmljIG9iamVjdCB2ZXJzaW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FtYXpvbi1zMy1wb2xpY3kta2V5cy5odG1sI2dldG9iamVjdHZlcnNpb24tbGltaXQtYWNjZXNzLXRvLXNwZWNpZmljLXZlcnNpb24tM1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlZlcnNpb25JZCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOlZlcnNpb25JZGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlc3RyaWN0cyBpbmNvbWluZyByZXF1ZXN0cyB0byBhIHNwZWNpZmljIGF1dGhlbnRpY2F0aW9uIG1ldGhvZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9idWNrZXQtcG9saWN5LXMzLXNpZ3Y0LWNvbmRpdGlvbnMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQWJvcnRNdWx0aXBhcnRVcGxvYWQoKVxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b0NyZWF0ZUpvYigpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9EZWxldGVCdWNrZXQoKVxuICAgKiAtIC50b0RlbGV0ZUJ1Y2tldFBvbGljeSgpXG4gICAqIC0gLnRvRGVsZXRlQnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvRGVsZXRlSm9iVGFnZ2luZygpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0KClcbiAgICogLSAudG9EZWxldGVPYmplY3RUYWdnaW5nKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqIC0gLnRvRGVzY3JpYmVKb2IoKVxuICAgKiAtIC50b0dldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKClcbiAgICogLSAudG9HZXRBY2NvdW50UHVibGljQWNjZXNzQmxvY2soKVxuICAgKiAtIC50b0dldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvR2V0QnVja2V0Q09SUygpXG4gICAqIC0gLnRvR2V0QnVja2V0TG9nZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0Tm90aWZpY2F0aW9uKClcbiAgICogLSAudG9HZXRCdWNrZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0QnVja2V0UG9saWN5KClcbiAgICogLSAudG9HZXRCdWNrZXRQb2xpY3lTdGF0dXMoKVxuICAgKiAtIC50b0dldEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9HZXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvR2V0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvR2V0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvR2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b0dldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFRvcnJlbnQoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Gb3JSZXBsaWNhdGlvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KClcbiAgICogLSAudG9HZXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0xpc3RBY2Nlc3NQb2ludHMoKVxuICAgKiAtIC50b0xpc3RBbGxNeUJ1Y2tldHMoKVxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzKClcbiAgICogLSAudG9MaXN0QnVja2V0VmVyc2lvbnMoKVxuICAgKiAtIC50b0xpc3RKb2JzKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b09iamVjdE93bmVyT3ZlcnJpZGVUb0J1Y2tldE93bmVyKClcbiAgICogLSAudG9QdXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b1B1dEFjY291bnRQdWJsaWNBY2Nlc3NCbG9jaygpXG4gICAqIC0gLnRvUHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QnVja2V0QWNsKClcbiAgICogLSAudG9QdXRCdWNrZXRDT1JTKClcbiAgICogLSAudG9QdXRCdWNrZXRMb2dnaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXROb3RpZmljYXRpb24oKVxuICAgKiAtIC50b1B1dEJ1Y2tldE9iamVjdExvY2tDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRCdWNrZXRQb2xpY3koKVxuICAgKiAtIC50b1B1dEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9QdXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvUHV0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvUHV0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvUHV0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9QdXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1JlcGxpY2F0ZURlbGV0ZSgpXG4gICAqIC0gLnRvUmVwbGljYXRlT2JqZWN0KClcbiAgICogLSAudG9SZXBsaWNhdGVUYWdzKClcbiAgICogLSAudG9SZXN0b3JlT2JqZWN0KClcbiAgICogLSAudG9VcGRhdGVKb2JQcmlvcml0eSgpXG4gICAqIC0gLnRvVXBkYXRlSm9iU3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBdXRoVHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOmF1dGhUeXBlYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhlIGRlbGltaXRlciBwYXJhbWV0ZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvd2Fsa3Rocm91Z2gxLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRWZXJzaW9ucygpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRGVsaW1pdGVyKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6ZGVsaW1pdGVyYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYSBzcGVjaWZpYyBSZWdpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvYW1hem9uLXMzLXBvbGljeS1rZXlzLmh0bWwjY29uZGl0aW9uLWtleS1idWNrZXQtb3BzLTFcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkxvY2F0aW9uY29uc3RyYWludCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOmxvY2F0aW9uY29uc3RyYWludGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpbWl0cyB0aGUgbWF4aW11bSBudW1iZXIgb2Yga2V5cyByZXR1cm5lZCBpbiBhIExpc3RCdWNrZXQgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hbWF6b24tczMtcG9saWN5LWtleXMuaHRtbCNleGFtcGxlLW51bWVyaWMtY29uZGl0aW9uLW9wZXJhdG9yc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvTGlzdEJ1Y2tldCgpXG4gICAqIC0gLnRvTGlzdEJ1Y2tldFZlcnNpb25zKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbbnVtZXJpYyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfTnVtZXJpYykuICoqRGVmYXVsdDoqKiBgTnVtZXJpY0VxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZk1heEtleXModmFsdWU6IG51bWJlciB8IG51bWJlcltdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzptYXgta2V5c2AsIHZhbHVlLCBvcGVyYXRvciB8fCAnTnVtZXJpY0VxdWFscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgZW5mb3JjZW1lbnQgb2YgdGhlIHNwZWNpZmllZCBvYmplY3QgbGVnYWwgaG9sZCBzdGF0dXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvb2JqZWN0LWxvY2stb3ZlcnZpZXcuaHRtbCNvYmplY3QtbG9jay1sZWdhbC1ob2xkc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RMZWdhbEhvbGQoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZk9iamVjdExvY2tMZWdhbEhvbGQodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpvYmplY3QtbG9jay1sZWdhbC1ob2xkYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyBlbmZvcmNlbWVudCBvZiB0aGUgc3BlY2lmaWVkIG9iamVjdCByZXRlbnRpb24gbW9kZSAoQ09NUExJQU5DRSBvciBHT1ZFUk5BTkNFKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZk9iamVjdExvY2tNb2RlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6b2JqZWN0LWxvY2stbW9kZWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgZW5mb3JjZW1lbnQgb2YgYW4gb2JqZWN0IHJlbGF0aXZlIHRvIHRoZSByZW1haW5pbmcgcmV0ZW50aW9uIGRheXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvb2JqZWN0LWxvY2stbWFuYWdpbmcuaHRtbCNvYmplY3QtbG9jay1tYW5hZ2luZy1yZXRlbnRpb24tbGltaXRzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdFJldGVudGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmT2JqZWN0TG9ja1JlbWFpbmluZ1JldGVudGlvbkRheXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpvYmplY3QtbG9jay1yZW1haW5pbmctcmV0ZW50aW9uLWRheXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGVzIGVuZm9yY2VtZW50IG9mIGEgc3BlY2lmaWMgcmV0YWluLXVudGlsLWRhdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvb2JqZWN0LWxvY2stb3ZlcnZpZXcuaHRtbCNvYmplY3QtbG9jay1yZXRlbnRpb24tcGVyaW9kc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZk9iamVjdExvY2tSZXRhaW5VbnRpbERhdGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpvYmplY3QtbG9jay1yZXRhaW4tdW50aWwtZGF0ZWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGtleSBuYW1lIHByZWZpeFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hbWF6b24tczMtcG9saWN5LWtleXMuaHRtbCNjb25kaXRpb24ta2V5LWJ1Y2tldC1vcHMtMlxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvTGlzdEJ1Y2tldCgpXG4gICAqIC0gLnRvTGlzdEJ1Y2tldFZlcnNpb25zKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZQcmVmaXgodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzpwcmVmaXhgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVzIHRoZSBsZW5ndGggb2YgdGltZSwgaW4gbWlsbGlzZWNvbmRzLCB0aGF0IGEgc2lnbmF0dXJlIGlzIHZhbGlkIGluIGFuIGF1dGhlbnRpY2F0ZWQgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9idWNrZXQtcG9saWN5LXMzLXNpZ3Y0LWNvbmRpdGlvbnMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQWJvcnRNdWx0aXBhcnRVcGxvYWQoKVxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b0NyZWF0ZUpvYigpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9EZWxldGVCdWNrZXQoKVxuICAgKiAtIC50b0RlbGV0ZUJ1Y2tldFBvbGljeSgpXG4gICAqIC0gLnRvRGVsZXRlQnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvRGVsZXRlSm9iVGFnZ2luZygpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0KClcbiAgICogLSAudG9EZWxldGVPYmplY3RUYWdnaW5nKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqIC0gLnRvRGVzY3JpYmVKb2IoKVxuICAgKiAtIC50b0dldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKClcbiAgICogLSAudG9HZXRBY2NvdW50UHVibGljQWNjZXNzQmxvY2soKVxuICAgKiAtIC50b0dldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvR2V0QnVja2V0Q09SUygpXG4gICAqIC0gLnRvR2V0QnVja2V0TG9nZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0Tm90aWZpY2F0aW9uKClcbiAgICogLSAudG9HZXRCdWNrZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0QnVja2V0UG9saWN5KClcbiAgICogLSAudG9HZXRCdWNrZXRQb2xpY3lTdGF0dXMoKVxuICAgKiAtIC50b0dldEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9HZXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvR2V0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvR2V0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvR2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b0dldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFRvcnJlbnQoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Gb3JSZXBsaWNhdGlvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KClcbiAgICogLSAudG9HZXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0xpc3RBY2Nlc3NQb2ludHMoKVxuICAgKiAtIC50b0xpc3RBbGxNeUJ1Y2tldHMoKVxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzKClcbiAgICogLSAudG9MaXN0QnVja2V0VmVyc2lvbnMoKVxuICAgKiAtIC50b0xpc3RKb2JzKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b09iamVjdE93bmVyT3ZlcnJpZGVUb0J1Y2tldE93bmVyKClcbiAgICogLSAudG9QdXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b1B1dEFjY291bnRQdWJsaWNBY2Nlc3NCbG9jaygpXG4gICAqIC0gLnRvUHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QnVja2V0QWNsKClcbiAgICogLSAudG9QdXRCdWNrZXRDT1JTKClcbiAgICogLSAudG9QdXRCdWNrZXRMb2dnaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXROb3RpZmljYXRpb24oKVxuICAgKiAtIC50b1B1dEJ1Y2tldE9iamVjdExvY2tDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRCdWNrZXRQb2xpY3koKVxuICAgKiAtIC50b1B1dEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9QdXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvUHV0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvUHV0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvUHV0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9QdXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1JlcGxpY2F0ZURlbGV0ZSgpXG4gICAqIC0gLnRvUmVwbGljYXRlT2JqZWN0KClcbiAgICogLSAudG9SZXBsaWNhdGVUYWdzKClcbiAgICogLSAudG9SZXN0b3JlT2JqZWN0KClcbiAgICogLSAudG9VcGRhdGVKb2JQcmlvcml0eSgpXG4gICAqIC0gLnRvVXBkYXRlSm9iU3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbbnVtZXJpYyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfTnVtZXJpYykuICoqRGVmYXVsdDoqKiBgTnVtZXJpY0VxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZlNpZ25hdHVyZUFnZSh2YWx1ZTogbnVtYmVyIHwgbnVtYmVyW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOnNpZ25hdHVyZUFnZWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnTnVtZXJpY0VxdWFscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXMgdGhlIHZlcnNpb24gb2YgQVdTIFNpZ25hdHVyZSB0aGF0IGlzIHN1cHBvcnRlZCBmb3IgYXV0aGVudGljYXRlZCByZXF1ZXN0c1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L0FQSS9idWNrZXQtcG9saWN5LXMzLXNpZ3Y0LWNvbmRpdGlvbnMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQWJvcnRNdWx0aXBhcnRVcGxvYWQoKVxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b0NyZWF0ZUpvYigpXG4gICAqIC0gLnRvRGVsZXRlQWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9EZWxldGVCdWNrZXQoKVxuICAgKiAtIC50b0RlbGV0ZUJ1Y2tldFBvbGljeSgpXG4gICAqIC0gLnRvRGVsZXRlQnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvRGVsZXRlSm9iVGFnZ2luZygpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0KClcbiAgICogLSAudG9EZWxldGVPYmplY3RUYWdnaW5nKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqIC0gLnRvRGVzY3JpYmVKb2IoKVxuICAgKiAtIC50b0dldEFjY2VsZXJhdGVDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5U3RhdHVzKClcbiAgICogLSAudG9HZXRBY2NvdW50UHVibGljQWNjZXNzQmxvY2soKVxuICAgKiAtIC50b0dldEFuYWx5dGljc0NvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvR2V0QnVja2V0Q09SUygpXG4gICAqIC0gLnRvR2V0QnVja2V0TG9nZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0Tm90aWZpY2F0aW9uKClcbiAgICogLSAudG9HZXRCdWNrZXRPYmplY3RMb2NrQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0QnVja2V0UG9saWN5KClcbiAgICogLSAudG9HZXRCdWNrZXRQb2xpY3lTdGF0dXMoKVxuICAgKiAtIC50b0dldEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9HZXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvR2V0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvR2V0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvR2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b0dldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFRvcnJlbnQoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Gb3JSZXBsaWNhdGlvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KClcbiAgICogLSAudG9HZXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0xpc3RBY2Nlc3NQb2ludHMoKVxuICAgKiAtIC50b0xpc3RBbGxNeUJ1Y2tldHMoKVxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzKClcbiAgICogLSAudG9MaXN0QnVja2V0VmVyc2lvbnMoKVxuICAgKiAtIC50b0xpc3RKb2JzKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b09iamVjdE93bmVyT3ZlcnJpZGVUb0J1Y2tldE93bmVyKClcbiAgICogLSAudG9QdXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b1B1dEFjY291bnRQdWJsaWNBY2Nlc3NCbG9jaygpXG4gICAqIC0gLnRvUHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QnVja2V0QWNsKClcbiAgICogLSAudG9QdXRCdWNrZXRDT1JTKClcbiAgICogLSAudG9QdXRCdWNrZXRMb2dnaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXROb3RpZmljYXRpb24oKVxuICAgKiAtIC50b1B1dEJ1Y2tldE9iamVjdExvY2tDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRCdWNrZXRQb2xpY3koKVxuICAgKiAtIC50b1B1dEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9QdXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvUHV0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvUHV0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvUHV0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9QdXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25UYWdnaW5nKClcbiAgICogLSAudG9QdXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1JlcGxpY2F0ZURlbGV0ZSgpXG4gICAqIC0gLnRvUmVwbGljYXRlT2JqZWN0KClcbiAgICogLSAudG9SZXBsaWNhdGVUYWdzKClcbiAgICogLSAudG9SZXN0b3JlT2JqZWN0KClcbiAgICogLSAudG9VcGRhdGVKb2JQcmlvcml0eSgpXG4gICAqIC0gLnRvVXBkYXRlSm9iU3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZTaWduYXR1cmV2ZXJzaW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6c2lnbmF0dXJldmVyc2lvbmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEgc3BlY2lmaWMgb2JqZWN0IHZlcnNpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvYW1hem9uLXMzLXBvbGljeS1rZXlzLmh0bWwjZ2V0b2JqZWN0dmVyc2lvbi1saW1pdC1hY2Nlc3MtdG8tc3BlY2lmaWMtdmVyc2lvbi0zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uKClcbiAgICogLSAudG9EZWxldGVPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvbkFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KClcbiAgICogLSAudG9QdXRPYmplY3RWZXJzaW9uQWNsKClcbiAgICogLSAudG9QdXRPYmplY3RWZXJzaW9uVGFnZ2luZygpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmVmVyc2lvbmlkKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6dmVyc2lvbmlkYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhlIHgtYW16LWFjbCBoZWFkZXIgd2l0aCBhIHNwZWNpZmljIGNhbm5lZCBBQ0wgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI3Blcm1pc3Npb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvQ3JlYXRlQnVja2V0KClcbiAgICogLSAudG9QdXRCdWNrZXRBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0QWNsKClcbiAgICogLSAudG9QdXRPYmplY3RWZXJzaW9uQWNsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZYQW16QWNsKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotYWNsYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRGlzYWxsb3dzIHVuc2lnbmVkIGNvbnRlbnQgaW4geW91ciBidWNrZXRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9BUEkvYnVja2V0LXBvbGljeS1zMy1zaWd2NC1jb25kaXRpb25zLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0Fib3J0TXVsdGlwYXJ0VXBsb2FkKClcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVBY2Nlc3NQb2ludCgpXG4gICAqIC0gLnRvQ3JlYXRlQnVja2V0KClcbiAgICogLSAudG9DcmVhdGVKb2IoKVxuICAgKiAtIC50b0RlbGV0ZUFjY2Vzc1BvaW50KClcbiAgICogLSAudG9EZWxldGVBY2Nlc3NQb2ludFBvbGljeSgpXG4gICAqIC0gLnRvRGVsZXRlQnVja2V0KClcbiAgICogLSAudG9EZWxldGVCdWNrZXRQb2xpY3koKVxuICAgKiAtIC50b0RlbGV0ZUJ1Y2tldFdlYnNpdGUoKVxuICAgKiAtIC50b0RlbGV0ZUpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b0RlbGV0ZU9iamVjdCgpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VmVyc2lvbigpXG4gICAqIC0gLnRvRGVsZXRlT2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0Rlc2NyaWJlSm9iKClcbiAgICogLSAudG9HZXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0QWNjZXNzUG9pbnQoKVxuICAgKiAtIC50b0dldEFjY2Vzc1BvaW50UG9saWN5KClcbiAgICogLSAudG9HZXRBY2Nlc3NQb2ludFBvbGljeVN0YXR1cygpXG4gICAqIC0gLnRvR2V0QWNjb3VudFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9HZXRBbmFseXRpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRCdWNrZXRBY2woKVxuICAgKiAtIC50b0dldEJ1Y2tldENPUlMoKVxuICAgKiAtIC50b0dldEJ1Y2tldExvZ2dpbmcoKVxuICAgKiAtIC50b0dldEJ1Y2tldE5vdGlmaWNhdGlvbigpXG4gICAqIC0gLnRvR2V0QnVja2V0UG9saWN5KClcbiAgICogLSAudG9HZXRCdWNrZXRQb2xpY3lTdGF0dXMoKVxuICAgKiAtIC50b0dldEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrKClcbiAgICogLSAudG9HZXRCdWNrZXRSZXF1ZXN0UGF5bWVudCgpXG4gICAqIC0gLnRvR2V0QnVja2V0VGFnZ2luZygpXG4gICAqIC0gLnRvR2V0QnVja2V0VmVyc2lvbmluZygpXG4gICAqIC0gLnRvR2V0QnVja2V0V2Vic2l0ZSgpXG4gICAqIC0gLnRvR2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEludmVudG9yeUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldEpvYlRhZ2dpbmcoKVxuICAgKiAtIC50b0dldExpZmVjeWNsZUNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0dldE1ldHJpY3NDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9HZXRPYmplY3QoKVxuICAgKiAtIC50b0dldE9iamVjdEFjbCgpXG4gICAqIC0gLnRvR2V0T2JqZWN0TGVnYWxIb2xkKClcbiAgICogLSAudG9HZXRPYmplY3RSZXRlbnRpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFRvcnJlbnQoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb24oKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25BY2woKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Gb3JSZXBsaWNhdGlvbigpXG4gICAqIC0gLnRvR2V0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b0dldE9iamVjdFZlcnNpb25Ub3JyZW50KClcbiAgICogLSAudG9HZXRSZXBsaWNhdGlvbkNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0xpc3RBY2Nlc3NQb2ludHMoKVxuICAgKiAtIC50b0xpc3RBbGxNeUJ1Y2tldHMoKVxuICAgKiAtIC50b0xpc3RCdWNrZXQoKVxuICAgKiAtIC50b0xpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzKClcbiAgICogLSAudG9MaXN0QnVja2V0VmVyc2lvbnMoKVxuICAgKiAtIC50b0xpc3RKb2JzKClcbiAgICogLSAudG9MaXN0TXVsdGlwYXJ0VXBsb2FkUGFydHMoKVxuICAgKiAtIC50b09iamVjdE93bmVyT3ZlcnJpZGVUb0J1Y2tldE93bmVyKClcbiAgICogLSAudG9QdXRBY2NlbGVyYXRlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QWNjZXNzUG9pbnRQb2xpY3koKVxuICAgKiAtIC50b1B1dEFjY291bnRQdWJsaWNBY2Nlc3NCbG9jaygpXG4gICAqIC0gLnRvUHV0QW5hbHl0aWNzQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0QnVja2V0QWNsKClcbiAgICogLSAudG9QdXRCdWNrZXRDT1JTKClcbiAgICogLSAudG9QdXRCdWNrZXRMb2dnaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXROb3RpZmljYXRpb24oKVxuICAgKiAtIC50b1B1dEJ1Y2tldFBvbGljeSgpXG4gICAqIC0gLnRvUHV0QnVja2V0UHVibGljQWNjZXNzQmxvY2soKVxuICAgKiAtIC50b1B1dEJ1Y2tldFJlcXVlc3RQYXltZW50KClcbiAgICogLSAudG9QdXRCdWNrZXRUYWdnaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXRWZXJzaW9uaW5nKClcbiAgICogLSAudG9QdXRCdWNrZXRXZWJzaXRlKClcbiAgICogLSAudG9QdXRFbmNyeXB0aW9uQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0SW52ZW50b3J5Q29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0Sm9iVGFnZ2luZygpXG4gICAqIC0gLnRvUHV0TGlmZWN5Y2xlQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUHV0TWV0cmljc0NvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0QWNsKClcbiAgICogLSAudG9QdXRPYmplY3RMZWdhbEhvbGQoKVxuICAgKiAtIC50b1B1dE9iamVjdFJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0VGFnZ2luZygpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvbkFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcoKVxuICAgKiAtIC50b1B1dFJlcGxpY2F0aW9uQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvUmVwbGljYXRlRGVsZXRlKClcbiAgICogLSAudG9SZXBsaWNhdGVPYmplY3QoKVxuICAgKiAtIC50b1JlcGxpY2F0ZVRhZ3MoKVxuICAgKiAtIC50b1Jlc3RvcmVPYmplY3QoKVxuICAgKiAtIC50b1VwZGF0ZUpvYlByaW9yaXR5KClcbiAgICogLSAudG9VcGRhdGVKb2JTdGF0dXMoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpDb250ZW50U2hhMjU2KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotY29udGVudC1zaGEyNTZgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXN0cmljdHMgdGhlIGNvcHkgc291cmNlIHRvIGEgc3BlY2lmaWMgYnVja2V0LCBwcmVmaXgsIG9yIG9iamVjdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hbWF6b24tczMtcG9saWN5LWtleXMuaHRtbCNwdXRvYmplY3QtbGltaXQtY29weS1zb3VyY2UtM1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZYQW16Q29weVNvdXJjZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOngtYW16LWNvcHktc291cmNlYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhlIHgtYW16LWdyYW50LWZ1bGwtY29udHJvbCAoZnVsbCBjb250cm9sKSBoZWFkZXIgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI3Blcm1pc3Npb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b1B1dEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpHcmFudEZ1bGxDb250cm9sKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotZ3JhbnQtZnVsbC1jb250cm9sYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhlIHgtYW16LWdyYW50LXJlYWQgKHJlYWQgYWNjZXNzKSBoZWFkZXIgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI3Blcm1pc3Npb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b1B1dEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpHcmFudFJlYWQodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzp4LWFtei1ncmFudC1yZWFkYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdGhlIHgtYW16LWdyYW50LXJlYWQtYWNwIChyZWFkIHBlcm1pc3Npb25zIGZvciB0aGUgQUNMKSBoZWFkZXIgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI3Blcm1pc3Npb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b1B1dEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpHcmFudFJlYWRBY3AodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzp4LWFtei1ncmFudC1yZWFkLWFjcGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVpcmVzIHRoZSB4LWFtei1ncmFudC13cml0ZSAod3JpdGUgYWNjZXNzKSBoZWFkZXIgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI3Blcm1pc3Npb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9DcmVhdGVCdWNrZXQoKVxuICAgKiAtIC50b1B1dEJ1Y2tldEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9QdXRPYmplY3RBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdFZlcnNpb25BY2woKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpHcmFudFdyaXRlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotZ3JhbnQtd3JpdGVgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1aXJlcyB0aGUgeC1hbXotZ3JhbnQtd3JpdGUtYWNwICh3cml0ZSBwZXJtaXNzaW9ucyBmb3IgdGhlIEFDTCkgaGVhZGVyIGluIGEgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hY2wtb3ZlcnZpZXcuaHRtbCNwZXJtaXNzaW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlQnVja2V0KClcbiAgICogLSAudG9QdXRCdWNrZXRBY2woKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0QWNsKClcbiAgICogLSAudG9QdXRPYmplY3RWZXJzaW9uQWNsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZYQW16R3JhbnRXcml0ZUFjcCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOngtYW16LWdyYW50LXdyaXRlLWFjcGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZXMgZW5mb3JjZW1lbnQgb2Ygb2JqZWN0IG1ldGFkYXRhIGJlaGF2aW9yIChDT1BZIG9yIFJFUExBQ0UpIHdoZW4gb2JqZWN0cyBhcmUgY29waWVkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvQVBJL0FQSV9Db3B5T2JqZWN0Lmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmWEFtek1ldGFkYXRhRGlyZWN0aXZlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotbWV0YWRhdGEtZGlyZWN0aXZlYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgc2VydmVyLXNpZGUgZW5jcnlwdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ1NlcnZlclNpZGVFbmNyeXB0aW9uLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0J5cGFzc0dvdmVybmFuY2VSZXRlbnRpb24oKVxuICAgKiAtIC50b1B1dE9iamVjdCgpXG4gICAqIC0gLnRvUmVwbGljYXRlT2JqZWN0KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZYQW16U2VydmVyU2lkZUVuY3J5cHRpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24ge1xuICAgIHJldHVybiB0aGlzLmlmKGBzMzp4LWFtei1zZXJ2ZXItc2lkZS1lbmNyeXB0aW9uYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgYSBzcGVjaWZpYyBBV1MgS01TIGN1c3RvbWVyIG1hbmFnZWQgQ01LIGZvciBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nS01TRW5jcnlwdGlvbi5odG1sI3JlcXVpcmUtc3NlLWttc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQnlwYXNzR292ZXJuYW5jZVJldGVudGlvbigpXG4gICAqIC0gLnRvUHV0T2JqZWN0KClcbiAgICogLSAudG9SZXBsaWNhdGVPYmplY3QoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpTZXJ2ZXJTaWRlRW5jcnlwdGlvbkF3c0ttc0tleUlkKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgczM6eC1hbXotc2VydmVyLXNpZGUtZW5jcnlwdGlvbi1hd3Mta21zLWtleS1pZGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHN0b3JhZ2UgY2xhc3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvc3RvcmFnZS1jbGFzcy1pbnRyby5odG1sI3NjLWhvd3Rvc2V0XG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKiAtIC50b1B1dE9iamVjdEFjbCgpXG4gICAqIC0gLnRvUHV0T2JqZWN0VmVyc2lvbkFjbCgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmWEFtelN0b3JhZ2VDbGFzcyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOngtYW16LXN0b3JhZ2UtY2xhc3NgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHNwZWNpZmljIHdlYnNpdGUgcmVkaXJlY3QgbG9jYXRpb24gZm9yIGJ1Y2tldHMgdGhhdCBhcmUgY29uZmlndXJlZCBhcyBzdGF0aWMgd2Vic2l0ZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvaG93LXRvLXBhZ2UtcmVkaXJlY3QuaHRtbCNwYWdlLXJlZGlyZWN0LXVzaW5nLXJlc3QtYXBpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CeXBhc3NHb3Zlcm5hbmNlUmV0ZW50aW9uKClcbiAgICogLSAudG9QdXRPYmplY3QoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlhBbXpXZWJzaXRlUmVkaXJlY3RMb2NhdGlvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHMzOngtYW16LXdlYnNpdGUtcmVkaXJlY3QtbG9jYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxufVxuIl19