"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ses = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ses extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses';
        this.actionList = {
            "CloneReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html",
                "description": "Creates a receipt rule set by cloning an existing one",
                "accessLevel": "Write"
            },
            "CreateConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html",
                "description": "Creates a new configuration set",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html",
                "description": "Creates a configuration set event destination",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html",
                "description": "Creates an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "CreateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html",
                "description": "Creates a new custom verification email template",
                "accessLevel": "Write"
            },
            "CreateReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html",
                "description": "Creates a new IP address filter",
                "accessLevel": "Write"
            },
            "CreateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html",
                "description": "Creates a receipt rule",
                "accessLevel": "Write"
            },
            "CreateReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html",
                "description": "Creates an empty receipt rule set",
                "accessLevel": "Write"
            },
            "CreateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html",
                "description": "Creates an email template",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html",
                "description": "Deletes the configuration set",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html",
                "description": "Deletes a configuration set event destination",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html",
                "description": "Deletes an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "DeleteCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html",
                "description": "Deletes an existing custom verification email template",
                "accessLevel": "Write"
            },
            "DeleteIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html",
                "description": "Deletes the specified IP address filter",
                "accessLevel": "Write"
            },
            "DeleteReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html",
                "description": "Deletes the specified receipt rule",
                "accessLevel": "Write"
            },
            "DeleteReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html",
                "description": "Deletes the specified receipt rule set and all of the receipt rules it contains",
                "accessLevel": "Write"
            },
            "DeleteTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html",
                "description": "Deletes an email template",
                "accessLevel": "Write"
            },
            "DeleteVerifiedEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html",
                "description": "Deletes the specified email address from the list of verified addresses",
                "accessLevel": "Write"
            },
            "DescribeActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html",
                "description": "Returns the metadata and receipt rules for the receipt rule set that is currently active",
                "accessLevel": "Read"
            },
            "DescribeConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html",
                "description": "Returns the details of the specified configuration set",
                "accessLevel": "Read"
            },
            "DescribeReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html",
                "description": "Returns the details of the specified receipt rule",
                "accessLevel": "Read"
            },
            "DescribeReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html",
                "description": "Returns the details of the specified receipt rule set",
                "accessLevel": "Read"
            },
            "GetAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html",
                "description": "Returns the email sending status of the Amazon SES account for the current region",
                "accessLevel": "Read"
            },
            "GetCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html",
                "description": "Returns the custom email verification template for the template name you specify",
                "accessLevel": "Read"
            },
            "GetIdentityDkimAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html",
                "description": "Returns the current status of Easy DKIM signing for an entity",
                "accessLevel": "Read"
            },
            "GetIdentityMailFromDomainAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html",
                "description": "Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)",
                "accessLevel": "Read"
            },
            "GetIdentityNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html",
                "description": "Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes",
                "accessLevel": "Read"
            },
            "GetIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html",
                "description": "Returns the requested sending authorization policies for the given identity (an email address or a domain)",
                "accessLevel": "Read"
            },
            "GetIdentityVerificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html",
                "description": "Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity",
                "accessLevel": "Read"
            },
            "GetSendQuota": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html",
                "description": "Returns the user's current sending limits",
                "accessLevel": "Read"
            },
            "GetSendStatistics": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html",
                "description": "Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity",
                "accessLevel": "Read"
            },
            "GetTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html",
                "description": "Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify",
                "accessLevel": "Read"
            },
            "ListConfigurationSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html",
                "description": "Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListCustomVerificationEmailTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html",
                "description": "Lists the existing custom verification email templates for your account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListIdentities": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html",
                "description": "Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status",
                "accessLevel": "List"
            },
            "ListIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html",
                "description": "Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)",
                "accessLevel": "List"
            },
            "ListReceiptFilters": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html",
                "description": "Lists the IP address filters associated with your AWS account",
                "accessLevel": "List"
            },
            "ListReceiptRuleSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html",
                "description": "Lists the receipt rule sets that exist under your AWS account",
                "accessLevel": "List"
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html",
                "description": "Lists the email templates present in your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListVerifiedEmailAddresses": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html",
                "description": "Returns a list containing all of the email addresses that have been verified",
                "accessLevel": "List"
            },
            "PutIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html",
                "description": "Adds or updates a sending authorization policy for the specified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "ReorderReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html",
                "description": "Reorders the receipt rules within a receipt rule set",
                "accessLevel": "Write"
            },
            "SendBounce": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html",
                "description": "Generates and sends a bounce message to the sender of an email you received through Amazon SES",
                "accessLevel": "Write",
                "conditions": [
                    "ses:FromAddress"
                ]
            },
            "SendBulkTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html",
                "description": "Composes an email message to multiple destinations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendCustomVerificationEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html",
                "description": "Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html",
                "description": "Composes an email message based on input data, and then immediately queues the message for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendRawEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html",
                "description": "Sends an email message, with header and content specified by the client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html",
                "description": "Composes an email message using an email template and immediately queues it for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SetActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html",
                "description": "Sets the specified receipt rule set as the active receipt rule set",
                "accessLevel": "Write"
            },
            "SetIdentityDkimEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html",
                "description": "Enables or disables Easy DKIM signing of email sent from an identity",
                "accessLevel": "Write"
            },
            "SetIdentityFeedbackForwardingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html",
                "description": "Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email",
                "accessLevel": "Write"
            },
            "SetIdentityHeadersInNotificationsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html",
                "description": "Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type",
                "accessLevel": "Write"
            },
            "SetIdentityMailFromDomain": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html",
                "description": "Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "SetIdentityNotificationTopic": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html",
                "description": "Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source",
                "accessLevel": "Write"
            },
            "SetReceiptRulePosition": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html",
                "description": "Sets the position of the specified receipt rule in the receipt rule set",
                "accessLevel": "Write"
            },
            "TestRenderTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html",
                "description": "Creates a preview of the MIME content of an email when provided with a template and a set of replacement data",
                "accessLevel": "Write"
            },
            "UpdateAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html",
                "description": "Enables or disables email sending across your entire Amazon SES account in the current AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html",
                "description": "Updates the event destination of a configuration set",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetReputationMetricsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html",
                "description": "Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html",
                "description": "Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html",
                "description": "Modifies an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "UpdateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html",
                "description": "Updates an existing custom verification email template",
                "accessLevel": "Write"
            },
            "UpdateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html",
                "description": "Updates a receipt rule",
                "accessLevel": "Write"
            },
            "UpdateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html",
                "description": "Updates an email template",
                "accessLevel": "Write"
            },
            "VerifyDomainDkim": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html",
                "description": "Returns a set of DKIM tokens for a domain",
                "accessLevel": "Read"
            },
            "VerifyDomainIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html",
                "description": "Verifies a domain",
                "accessLevel": "Read"
            },
            "VerifyEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            },
            "VerifyEmailIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "configuration-set": {
                "name": "configuration-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}",
                "conditionKeys": []
            },
            "custom-verification-email-template": {
                "name": "custom-verification-email-template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}",
                "conditionKeys": []
            },
            "event-destination": {
                "name": "event-destination",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}",
                "conditionKeys": []
            },
            "receipt-filter": {
                "name": "receipt-filter",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}",
                "conditionKeys": []
            },
            "receipt-rule": {
                "name": "receipt-rule",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}",
                "conditionKeys": []
            },
            "receipt-rule-set": {
                "name": "receipt-rule-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}",
                "conditionKeys": []
            },
            "template": {
                "name": "template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a receipt rule set by cloning an existing one
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html
     */
    toCloneReceiptRuleSet() {
        this.add('ses:CloneReceiptRuleSet');
        return this;
    }
    /**
     * Creates a new configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        this.add('ses:CreateConfigurationSet');
        return this;
    }
    /**
     * Creates a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        this.add('ses:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Creates an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html
     */
    toCreateConfigurationSetTrackingOptions() {
        this.add('ses:CreateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Creates a new custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate() {
        this.add('ses:CreateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Creates a new IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html
     */
    toCreateReceiptFilter() {
        this.add('ses:CreateReceiptFilter');
        return this;
    }
    /**
     * Creates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html
     */
    toCreateReceiptRule() {
        this.add('ses:CreateReceiptRule');
        return this;
    }
    /**
     * Creates an empty receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html
     */
    toCreateReceiptRuleSet() {
        this.add('ses:CreateReceiptRuleSet');
        return this;
    }
    /**
     * Creates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        this.add('ses:CreateTemplate');
        return this;
    }
    /**
     * Deletes the configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        this.add('ses:DeleteConfigurationSet');
        return this;
    }
    /**
     * Deletes a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        this.add('ses:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Deletes an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html
     */
    toDeleteConfigurationSetTrackingOptions() {
        this.add('ses:DeleteConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Deletes an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate() {
        this.add('ses:DeleteCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html
     */
    toDeleteIdentity() {
        this.add('ses:DeleteIdentity');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html
     */
    toDeleteIdentityPolicy() {
        this.add('ses:DeleteIdentityPolicy');
        return this;
    }
    /**
     * Deletes the specified IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html
     */
    toDeleteReceiptFilter() {
        this.add('ses:DeleteReceiptFilter');
        return this;
    }
    /**
     * Deletes the specified receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html
     */
    toDeleteReceiptRule() {
        this.add('ses:DeleteReceiptRule');
        return this;
    }
    /**
     * Deletes the specified receipt rule set and all of the receipt rules it contains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html
     */
    toDeleteReceiptRuleSet() {
        this.add('ses:DeleteReceiptRuleSet');
        return this;
    }
    /**
     * Deletes an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        this.add('ses:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the specified email address from the list of verified addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html
     */
    toDeleteVerifiedEmailAddress() {
        this.add('ses:DeleteVerifiedEmailAddress');
        return this;
    }
    /**
     * Returns the metadata and receipt rules for the receipt rule set that is currently active
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html
     */
    toDescribeActiveReceiptRuleSet() {
        this.add('ses:DescribeActiveReceiptRuleSet');
        return this;
    }
    /**
     * Returns the details of the specified configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html
     */
    toDescribeConfigurationSet() {
        this.add('ses:DescribeConfigurationSet');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html
     */
    toDescribeReceiptRule() {
        this.add('ses:DescribeReceiptRule');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html
     */
    toDescribeReceiptRuleSet() {
        this.add('ses:DescribeReceiptRuleSet');
        return this;
    }
    /**
     * Returns the email sending status of the Amazon SES account for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html
     */
    toGetAccountSendingEnabled() {
        this.add('ses:GetAccountSendingEnabled');
        return this;
    }
    /**
     * Returns the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate() {
        this.add('ses:GetCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Returns the current status of Easy DKIM signing for an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html
     */
    toGetIdentityDkimAttributes() {
        this.add('ses:GetIdentityDkimAttributes');
        return this;
    }
    /**
     * Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html
     */
    toGetIdentityMailFromDomainAttributes() {
        this.add('ses:GetIdentityMailFromDomainAttributes');
        return this;
    }
    /**
     * Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html
     */
    toGetIdentityNotificationAttributes() {
        this.add('ses:GetIdentityNotificationAttributes');
        return this;
    }
    /**
     * Returns the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html
     */
    toGetIdentityPolicies() {
        this.add('ses:GetIdentityPolicies');
        return this;
    }
    /**
     * Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html
     */
    toGetIdentityVerificationAttributes() {
        this.add('ses:GetIdentityVerificationAttributes');
        return this;
    }
    /**
     * Returns the user's current sending limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html
     */
    toGetSendQuota() {
        this.add('ses:GetSendQuota');
        return this;
    }
    /**
     * Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html
     */
    toGetSendStatistics() {
        this.add('ses:GetSendStatistics');
        return this;
    }
    /**
     * Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        this.add('ses:GetTemplate');
        return this;
    }
    /**
     * Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        this.add('ses:ListConfigurationSets');
        return this;
    }
    /**
     * Lists the existing custom verification email templates for your account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates() {
        this.add('ses:ListCustomVerificationEmailTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html
     */
    toListIdentities() {
        this.add('ses:ListIdentities');
        return this;
    }
    /**
     * Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html
     */
    toListIdentityPolicies() {
        this.add('ses:ListIdentityPolicies');
        return this;
    }
    /**
     * Lists the IP address filters associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html
     */
    toListReceiptFilters() {
        this.add('ses:ListReceiptFilters');
        return this;
    }
    /**
     * Lists the receipt rule sets that exist under your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html
     */
    toListReceiptRuleSets() {
        this.add('ses:ListReceiptRuleSets');
        return this;
    }
    /**
     * Lists the email templates present in your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        this.add('ses:ListTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the email addresses that have been verified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html
     */
    toListVerifiedEmailAddresses() {
        this.add('ses:ListVerifiedEmailAddresses');
        return this;
    }
    /**
     * Adds or updates a sending authorization policy for the specified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html
     */
    toPutIdentityPolicy() {
        this.add('ses:PutIdentityPolicy');
        return this;
    }
    /**
     * Reorders the receipt rules within a receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html
     */
    toReorderReceiptRuleSet() {
        this.add('ses:ReorderReceiptRuleSet');
        return this;
    }
    /**
     * Generates and sends a bounce message to the sender of an email you received through Amazon SES
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromAddress()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html
     */
    toSendBounce() {
        this.add('ses:SendBounce');
        return this;
    }
    /**
     * Composes an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html
     */
    toSendBulkTemplatedEmail() {
        this.add('ses:SendBulkTemplatedEmail');
        return this;
    }
    /**
     * Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail() {
        this.add('ses:SendCustomVerificationEmail');
        return this;
    }
    /**
     * Composes an email message based on input data, and then immediately queues the message for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html
     */
    toSendEmail() {
        this.add('ses:SendEmail');
        return this;
    }
    /**
     * Sends an email message, with header and content specified by the client
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html
     */
    toSendRawEmail() {
        this.add('ses:SendRawEmail');
        return this;
    }
    /**
     * Composes an email message using an email template and immediately queues it for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html
     */
    toSendTemplatedEmail() {
        this.add('ses:SendTemplatedEmail');
        return this;
    }
    /**
     * Sets the specified receipt rule set as the active receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html
     */
    toSetActiveReceiptRuleSet() {
        this.add('ses:SetActiveReceiptRuleSet');
        return this;
    }
    /**
     * Enables or disables Easy DKIM signing of email sent from an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html
     */
    toSetIdentityDkimEnabled() {
        this.add('ses:SetIdentityDkimEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html
     */
    toSetIdentityFeedbackForwardingEnabled() {
        this.add('ses:SetIdentityFeedbackForwardingEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html
     */
    toSetIdentityHeadersInNotificationsEnabled() {
        this.add('ses:SetIdentityHeadersInNotificationsEnabled');
        return this;
    }
    /**
     * Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html
     */
    toSetIdentityMailFromDomain() {
        this.add('ses:SetIdentityMailFromDomain');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html
     */
    toSetIdentityNotificationTopic() {
        this.add('ses:SetIdentityNotificationTopic');
        return this;
    }
    /**
     * Sets the position of the specified receipt rule in the receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html
     */
    toSetReceiptRulePosition() {
        this.add('ses:SetReceiptRulePosition');
        return this;
    }
    /**
     * Creates a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html
     */
    toTestRenderTemplate() {
        this.add('ses:TestRenderTemplate');
        return this;
    }
    /**
     * Enables or disables email sending across your entire Amazon SES account in the current AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html
     */
    toUpdateAccountSendingEnabled() {
        this.add('ses:UpdateAccountSendingEnabled');
        return this;
    }
    /**
     * Updates the event destination of a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        this.add('ses:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html
     */
    toUpdateConfigurationSetReputationMetricsEnabled() {
        this.add('ses:UpdateConfigurationSetReputationMetricsEnabled');
        return this;
    }
    /**
     * Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html
     */
    toUpdateConfigurationSetSendingEnabled() {
        this.add('ses:UpdateConfigurationSetSendingEnabled');
        return this;
    }
    /**
     * Modifies an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html
     */
    toUpdateConfigurationSetTrackingOptions() {
        this.add('ses:UpdateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Updates an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate() {
        this.add('ses:UpdateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Updates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html
     */
    toUpdateReceiptRule() {
        this.add('ses:UpdateReceiptRule');
        return this;
    }
    /**
     * Updates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        this.add('ses:UpdateTemplate');
        return this;
    }
    /**
     * Returns a set of DKIM tokens for a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html
     */
    toVerifyDomainDkim() {
        this.add('ses:VerifyDomainDkim');
        return this;
    }
    /**
     * Verifies a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html
     */
    toVerifyDomainIdentity() {
        this.add('ses:VerifyDomainIdentity');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html
     */
    toVerifyEmailAddress() {
        this.add('ses:VerifyEmailAddress');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html
     */
    toVerifyEmailIdentity() {
        this.add('ses:VerifyEmailIdentity');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCustomVerificationEmailTemplate(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-filter to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html
     *
     * @param receiptFilterName - Identifier for the receiptFilterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptFilter(receiptFilterName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}';
        arn = arn.replace('${ReceiptFilterName}', receiptFilterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param receiptRuleName - Identifier for the receiptRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRule(receiptRuleSetName, receiptRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${ReceiptRuleName}', receiptRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRuleSet(receiptRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTemplate(templateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBounce()
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.Ses = Ses;
//# sourceMappingURL=data:application/json;base64,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