"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sqs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sqs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsqs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sqs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sqs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsqs.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sqs';
        this.actionList = {
            "AddPermission": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_AddPermission.html",
                "description": "Adds a permission to a queue for a specific principal.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ChangeMessageVisibility": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibility.html",
                "description": "Changes the visibility timeout of a specified message in a queue to a new value.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ChangeMessageVisibilityBatch": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html",
                "description": "Changes the visibility timeout of multiple messages.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "CreateQueue": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_CreateQueue.html",
                "description": "Creates a new queue, or returns the URL of an existing one.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "DeleteMessage": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessage.html",
                "description": "Deletes the specified message from the specified queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "DeleteMessageBatch": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html",
                "description": "Deletes up to ten messages from the specified queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "DeleteQueue": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteQueue.html",
                "description": "Deletes the queue specified by the queue URL, regardless of whether the queue is empty.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "GetQueueAttributes": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueAttributes.html",
                "description": "Gets attributes for the specified queue.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "GetQueueUrl": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueUrl.html",
                "description": "Returns the URL of an existing queue.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ListDeadLetterSourceQueues": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListDeadLetterSourceQueues.html",
                "description": "Returns a list of your queues that have the RedrivePolicy queue attribute configured with a dead letter queue.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ListQueueTags": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueueTags.html",
                "description": "Lists tags added to an SQS queue.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ListQueues": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueues.html",
                "description": "Returns a list of your queues.",
                "accessLevel": "List"
            },
            "PurgeQueue": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_PurgeQueue.html",
                "description": "Deletes the messages in a queue specified by the queue URL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "ReceiveMessage": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ReceiveMessage.html",
                "description": "Retrieves one or more messages, with a maximum limit of 10 messages, from the specified queue.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_RemovePermission.html",
                "description": "Revokes any permissions in the queue policy that matches the specified Label parameter.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "SendMessage": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html",
                "description": "Delivers a message to the specified queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "SendMessageBatch": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html",
                "description": "Delivers up to ten messages to the specified queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "SetQueueAttributes": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SetQueueAttributes.html",
                "description": "Sets the value of one or more queue attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "TagQueue": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_TagQueue.html",
                "description": "Add tags to the specified SQS queue.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            },
            "UntagQueue": {
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_UntagQueue.html",
                "description": "Remove tags from the specified SQS queue.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "queue": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "queue": {
                "name": "queue",
                "url": "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-how-it-works.html",
                "arn": "arn:${Partition}:sqs:${Region}:${Account}:${QueueName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds a permission to a queue for a specific principal.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_AddPermission.html
     */
    toAddPermission() {
        this.add('sqs:AddPermission');
        return this;
    }
    /**
     * Changes the visibility timeout of a specified message in a queue to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibility.html
     */
    toChangeMessageVisibility() {
        this.add('sqs:ChangeMessageVisibility');
        return this;
    }
    /**
     * Changes the visibility timeout of multiple messages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html
     */
    toChangeMessageVisibilityBatch() {
        this.add('sqs:ChangeMessageVisibilityBatch');
        return this;
    }
    /**
     * Creates a new queue, or returns the URL of an existing one.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue() {
        this.add('sqs:CreateQueue');
        return this;
    }
    /**
     * Deletes the specified message from the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessage.html
     */
    toDeleteMessage() {
        this.add('sqs:DeleteMessage');
        return this;
    }
    /**
     * Deletes up to ten messages from the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html
     */
    toDeleteMessageBatch() {
        this.add('sqs:DeleteMessageBatch');
        return this;
    }
    /**
     * Deletes the queue specified by the queue URL, regardless of whether the queue is empty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue() {
        this.add('sqs:DeleteQueue');
        return this;
    }
    /**
     * Gets attributes for the specified queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueAttributes.html
     */
    toGetQueueAttributes() {
        this.add('sqs:GetQueueAttributes');
        return this;
    }
    /**
     * Returns the URL of an existing queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueUrl.html
     */
    toGetQueueUrl() {
        this.add('sqs:GetQueueUrl');
        return this;
    }
    /**
     * Returns a list of your queues that have the RedrivePolicy queue attribute configured with a dead letter queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListDeadLetterSourceQueues.html
     */
    toListDeadLetterSourceQueues() {
        this.add('sqs:ListDeadLetterSourceQueues');
        return this;
    }
    /**
     * Lists tags added to an SQS queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueueTags.html
     */
    toListQueueTags() {
        this.add('sqs:ListQueueTags');
        return this;
    }
    /**
     * Returns a list of your queues.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ListQueues.html
     */
    toListQueues() {
        this.add('sqs:ListQueues');
        return this;
    }
    /**
     * Deletes the messages in a queue specified by the queue URL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_PurgeQueue.html
     */
    toPurgeQueue() {
        this.add('sqs:PurgeQueue');
        return this;
    }
    /**
     * Retrieves one or more messages, with a maximum limit of 10 messages, from the specified queue.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ReceiveMessage.html
     */
    toReceiveMessage() {
        this.add('sqs:ReceiveMessage');
        return this;
    }
    /**
     * Revokes any permissions in the queue policy that matches the specified Label parameter.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        this.add('sqs:RemovePermission');
        return this;
    }
    /**
     * Delivers a message to the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html
     */
    toSendMessage() {
        this.add('sqs:SendMessage');
        return this;
    }
    /**
     * Delivers up to ten messages to the specified queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html
     */
    toSendMessageBatch() {
        this.add('sqs:SendMessageBatch');
        return this;
    }
    /**
     * Sets the value of one or more queue attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SetQueueAttributes.html
     */
    toSetQueueAttributes() {
        this.add('sqs:SetQueueAttributes');
        return this;
    }
    /**
     * Add tags to the specified SQS queue.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_TagQueue.html
     */
    toTagQueue() {
        this.add('sqs:TagQueue');
        return this;
    }
    /**
     * Remove tags from the specified SQS queue.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_UntagQueue.html
     */
    toUntagQueue() {
        this.add('sqs:UntagQueue');
        return this;
    }
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-how-it-works.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueue(queueName, account, region, partition) {
        var arn = 'arn:${Partition}:sqs:${Region}:${Account}:${QueueName}';
        arn = arn.replace('${QueueName}', queueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sqs = Sqs;
//# sourceMappingURL=data:application/json;base64,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