"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transcribe = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [transcribe](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transcribe extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [transcribe](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'transcribe';
        this.actionList = {
            "CreateLanguageModel": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateLanguageModel.html",
                "description": "Grants permission to create a new custom language model.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject",
                    "s3:ListBucket"
                ]
            },
            "CreateMedicalVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html",
                "description": "Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe Medical handles transcription of an audio file.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "CreateVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html",
                "description": "Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an audio file.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "CreateVocabularyFilter": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html",
                "description": "Grants permission to create a new vocabulary filter that you can use to filter out words from the transcription of an audio file generated by Amazon Transcribe",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "DeleteLanguageModel": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteLanguageModel.html",
                "description": "Grants permission to delete a previously created custom language model.",
                "accessLevel": "Write"
            },
            "DeleteMedicalTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalTranscriptionJob.html",
                "description": "Grants permission to delete a previously submitted medical transcription job.",
                "accessLevel": "Write"
            },
            "DeleteMedicalVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalVocabulary.html",
                "description": "Grants permission to delete a medical vocabulary from Amazon Transcribe.",
                "accessLevel": "Write"
            },
            "DeleteTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteTranscriptionJob.html",
                "description": "Grants permission to delete a previously submitted transcription job along with any other generated results such as the transcription, models, and so on.",
                "accessLevel": "Write"
            },
            "DeleteVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabulary.html",
                "description": "Grants permission to delete a vocabulary from Amazon Transcribe.",
                "accessLevel": "Write"
            },
            "DeleteVocabularyFilter": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabularyFilter.html",
                "description": "Grants permission to delete a vocabulary filter from Amazon Transcribe.",
                "accessLevel": "Write"
            },
            "DescribeLanguageModel": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_DescribeLanguageModel.html",
                "description": "Grants permission to return information about a custom language model.",
                "accessLevel": "Read"
            },
            "GetMedicalTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalTranscriptionJob.html",
                "description": "Grants permission to return information about a medical transcription job.",
                "accessLevel": "Read"
            },
            "GetMedicalVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalVocabulary.html",
                "description": "Grants permission to get information about a medical vocabulary.",
                "accessLevel": "Read"
            },
            "GetTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_GetTranscriptionJob.html",
                "description": "Grants permission to return information about a transcription job.",
                "accessLevel": "Read"
            },
            "GetVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabulary.html",
                "description": "Grants permission to to get information about a vocabulary.",
                "accessLevel": "Read"
            },
            "GetVocabularyFilter": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabularyFilter.html",
                "description": "Grants permission to get information about a vocabulary filter.",
                "accessLevel": "Read"
            },
            "ListLanguageModels": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListLanguageModels.html",
                "description": "Grants permission to list custom language models.",
                "accessLevel": "List"
            },
            "ListMedicalTranscriptionJobs": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalTranscriptionJobs.html",
                "description": "Grants permission to list medical transcription jobs with the specified status.",
                "accessLevel": "List"
            },
            "ListMedicalVocabularies": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalVocabularies.html",
                "description": "Grants permission to return a list of medical vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.",
                "accessLevel": "List"
            },
            "ListTranscriptionJobs": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTranscriptionJobs.html",
                "description": "Grants permission to list transcription jobs with the specified status.",
                "accessLevel": "List"
            },
            "ListVocabularies": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularies.html",
                "description": "Grants permission to return a list of vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.",
                "accessLevel": "List"
            },
            "ListVocabularyFilters": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularyFilters.html",
                "description": "Grants permission to return a list of vocabulary filters that match the specified criteria. If no criteria are specified, returns the at most 5 vocabulary filters.",
                "accessLevel": "List"
            },
            "StartMedicalStreamTranscription": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscription.html",
                "description": "Grants permission to start a protocol where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.",
                "accessLevel": "Write"
            },
            "StartMedicalStreamTranscriptionWebSocket": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscriptionWebSocket.html",
                "description": "Grants permission to start a WebSocket where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.",
                "accessLevel": "Write"
            },
            "StartMedicalTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalTranscriptionJob.html",
                "description": "Grants permission to start an asynchronous job to transcribe medical speech to text.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "StartStreamTranscription": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscription.html",
                "description": "Grants permission to start a bidirectional HTTP2 stream to transcribe speech to text in real time.",
                "accessLevel": "Write"
            },
            "StartStreamTranscriptionWebSocket": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscriptionWebSocket.html",
                "description": "Grants permission to start a websocket stream to transcribe speech to text in real time.",
                "accessLevel": "Write"
            },
            "StartTranscriptionJob": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_StartTranscriptionJob.html",
                "description": "Grants permission to start an asynchronous job to transcribe speech to text.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ],
                "conditions": [
                    "transcribe:OutputBucketName",
                    "transcribe:OutputEncryptionKMSKeyId"
                ]
            },
            "UpdateMedicalVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateMedicalVocabulary.html",
                "description": "Grants permission to update an existing medical vocabulary with new values. The UpdateMedicalVocabulary operation overwrites all of the existing information with the values that you provide in the request.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "UpdateVocabulary": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabulary.html",
                "description": "Grants permission to update an existing vocabulary with new values. The UpdateVocabulary operation overwrites all of the existing information with the values that you provide in the request.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            },
            "UpdateVocabularyFilter": {
                "url": "https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabularyFilter.html",
                "description": "Grants permission to update an existing vocabulary filter with new values. The UpdateVocabularyFilter operation overwrites all of the existing information with the values that you provide in the request.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:GetObject"
                ]
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Grants permission to create a new custom language model.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateLanguageModel.html
     */
    toCreateLanguageModel() {
        this.add('transcribe:CreateLanguageModel');
        return this;
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe Medical handles transcription of an audio file.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     */
    toCreateMedicalVocabulary() {
        this.add('transcribe:CreateMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an audio file.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     */
    toCreateVocabulary() {
        this.add('transcribe:CreateVocabulary');
        return this;
    }
    /**
     * Grants permission to create a new vocabulary filter that you can use to filter out words from the transcription of an audio file generated by Amazon Transcribe
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     */
    toCreateVocabularyFilter() {
        this.add('transcribe:CreateVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to delete a previously created custom language model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteLanguageModel.html
     */
    toDeleteLanguageModel() {
        this.add('transcribe:DeleteLanguageModel');
        return this;
    }
    /**
     * Grants permission to delete a previously submitted medical transcription job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalTranscriptionJob.html
     */
    toDeleteMedicalTranscriptionJob() {
        this.add('transcribe:DeleteMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to delete a medical vocabulary from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalVocabulary.html
     */
    toDeleteMedicalVocabulary() {
        this.add('transcribe:DeleteMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to delete a previously submitted transcription job along with any other generated results such as the transcription, models, and so on.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteTranscriptionJob.html
     */
    toDeleteTranscriptionJob() {
        this.add('transcribe:DeleteTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to delete a vocabulary from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabulary.html
     */
    toDeleteVocabulary() {
        this.add('transcribe:DeleteVocabulary');
        return this;
    }
    /**
     * Grants permission to delete a vocabulary filter from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabularyFilter.html
     */
    toDeleteVocabularyFilter() {
        this.add('transcribe:DeleteVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to return information about a custom language model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DescribeLanguageModel.html
     */
    toDescribeLanguageModel() {
        this.add('transcribe:DescribeLanguageModel');
        return this;
    }
    /**
     * Grants permission to return information about a medical transcription job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalTranscriptionJob.html
     */
    toGetMedicalTranscriptionJob() {
        this.add('transcribe:GetMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to get information about a medical vocabulary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalVocabulary.html
     */
    toGetMedicalVocabulary() {
        this.add('transcribe:GetMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to return information about a transcription job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetTranscriptionJob.html
     */
    toGetTranscriptionJob() {
        this.add('transcribe:GetTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to to get information about a vocabulary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabulary.html
     */
    toGetVocabulary() {
        this.add('transcribe:GetVocabulary');
        return this;
    }
    /**
     * Grants permission to get information about a vocabulary filter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabularyFilter.html
     */
    toGetVocabularyFilter() {
        this.add('transcribe:GetVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to list custom language models.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListLanguageModels.html
     */
    toListLanguageModels() {
        this.add('transcribe:ListLanguageModels');
        return this;
    }
    /**
     * Grants permission to list medical transcription jobs with the specified status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalTranscriptionJobs.html
     */
    toListMedicalTranscriptionJobs() {
        this.add('transcribe:ListMedicalTranscriptionJobs');
        return this;
    }
    /**
     * Grants permission to return a list of medical vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalVocabularies.html
     */
    toListMedicalVocabularies() {
        this.add('transcribe:ListMedicalVocabularies');
        return this;
    }
    /**
     * Grants permission to list transcription jobs with the specified status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTranscriptionJobs.html
     */
    toListTranscriptionJobs() {
        this.add('transcribe:ListTranscriptionJobs');
        return this;
    }
    /**
     * Grants permission to return a list of vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularies.html
     */
    toListVocabularies() {
        this.add('transcribe:ListVocabularies');
        return this;
    }
    /**
     * Grants permission to return a list of vocabulary filters that match the specified criteria. If no criteria are specified, returns the at most 5 vocabulary filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularyFilters.html
     */
    toListVocabularyFilters() {
        this.add('transcribe:ListVocabularyFilters');
        return this;
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscription.html
     */
    toStartMedicalStreamTranscription() {
        this.add('transcribe:StartMedicalStreamTranscription');
        return this;
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscriptionWebSocket.html
     */
    toStartMedicalStreamTranscriptionWebSocket() {
        this.add('transcribe:StartMedicalStreamTranscriptionWebSocket');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to transcribe medical speech to text.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalTranscriptionJob.html
     */
    toStartMedicalTranscriptionJob() {
        this.add('transcribe:StartMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream to transcribe speech to text in real time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscription.html
     */
    toStartStreamTranscription() {
        this.add('transcribe:StartStreamTranscription');
        return this;
    }
    /**
     * Grants permission to start a websocket stream to transcribe speech to text in real time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscriptionWebSocket.html
     */
    toStartStreamTranscriptionWebSocket() {
        this.add('transcribe:StartStreamTranscriptionWebSocket');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to transcribe speech to text.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartTranscriptionJob.html
     */
    toStartTranscriptionJob() {
        this.add('transcribe:StartTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to update an existing medical vocabulary with new values. The UpdateMedicalVocabulary operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateMedicalVocabulary.html
     */
    toUpdateMedicalVocabulary() {
        this.add('transcribe:UpdateMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to update an existing vocabulary with new values. The UpdateVocabulary operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabulary.html
     */
    toUpdateVocabulary() {
        this.add('transcribe:UpdateVocabulary');
        return this;
    }
    /**
     * Grants permission to update an existing vocabulary filter with new values. The UpdateVocabularyFilter operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabularyFilter.html
     */
    toUpdateVocabularyFilter() {
        this.add('transcribe:UpdateVocabularyFilter');
        return this;
    }
    /**
     * Enables you to control access based on the output bucket name included in the request
     *
     * Applies to actions:
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputBucketName(value, operator) {
        return this.if(`transcribe:OutputBucketName`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the KMS key id included in the request
     *
     * Applies to actions:
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputEncryptionKMSKeyId(value, operator) {
        return this.if(`transcribe:OutputEncryptionKMSKeyId`, value, operator || 'StringLike');
    }
}
exports.Transcribe = Transcribe;
//# sourceMappingURL=data:application/json;base64,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