"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspaces = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workspaces](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workspaces extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workspaces](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkspaces.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workspaces';
        this.actionList = {
            "AssociateIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html",
                "description": "Associates the specified IP access control group with the specified directory.",
                "accessLevel": "Write"
            },
            "AuthorizeIpRules": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html",
                "description": "Adds one or more rules to the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "CreateIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html",
                "description": "Creates an IP access control group.",
                "accessLevel": "Write"
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html",
                "description": "Creates tags for a WorkSpace.",
                "accessLevel": "Tagging"
            },
            "CreateWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html",
                "description": "Creates one or more WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    },
                    "workspacebundle": {
                        "required": true
                    }
                }
            },
            "DeleteIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html",
                "description": "Deletes the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html",
                "description": "Deletes tags from a Workspace.",
                "accessLevel": "Write"
            },
            "DeleteWorkspaceImage": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html",
                "description": "Deletes the specified workspace image.",
                "accessLevel": "Write"
            },
            "DescribeAccount": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html",
                "description": "Retrieves a list that describes the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "List"
            },
            "DescribeAccountModifications": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html",
                "description": "Retrieves a list that describes modifications to the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "List"
            },
            "DescribeClientProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html",
                "description": "Describe client properties about the specified resources.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    }
                }
            },
            "DescribeIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html",
                "description": "Retrieves information about the IP access control groups of your account in the region.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html",
                "description": "Describes tags for a WorkSpace.",
                "accessLevel": "List"
            },
            "DescribeWorkspaceBundles": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html",
                "description": "Obtains information about the WorkSpace bundles that are available to your account in the specified region.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workspacebundle": {
                        "required": true
                    }
                }
            },
            "DescribeWorkspaceDirectories": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html",
                "description": "Retrieves information about the AWS Directory Service directories in the region that are registered with Amazon WorkSpaces and are available to your account.",
                "accessLevel": "List"
            },
            "DescribeWorkspaceImages": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html",
                "description": "Retrieves a list that describes one or more specified images.",
                "accessLevel": "List"
            },
            "DescribeWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html",
                "description": "Obtains information about the specified WorkSpaces.",
                "accessLevel": "List"
            },
            "DescribeWorkspacesConnectionStatus": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html",
                "description": "Describes the connection status of a specified WorkSpace.",
                "accessLevel": "Read"
            },
            "DisassociateIpGroups": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html",
                "description": "Disassociates the specified IP access control group from the specified directory.",
                "accessLevel": "Write"
            },
            "ImportWorkspaceImage": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html",
                "description": "Import a licensed EC2 image to into Amazon WorkSpaces.",
                "accessLevel": "Write"
            },
            "ListAvailableManagementCidrRanges": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html",
                "description": "List available CIDR ranges for a CIDR range constraint.",
                "accessLevel": "List"
            },
            "ModifyAccount": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html",
                "description": "Modify the configuration of bring your own license (BYOL) for the specified account.",
                "accessLevel": "Write"
            },
            "ModifyClientProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html",
                "description": "Modify the client properties of a specified resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directoryid": {
                        "required": true
                    }
                }
            },
            "ModifyWorkspaceProperties": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html",
                "description": "Modifies the WorkSpace properties, including the running mode and AutoStop time.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "ModifyWorkspaceState": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html",
                "description": "Modify the state of specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RebootWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html",
                "description": "Reboots the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RebuildWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html",
                "description": "Rebuilds the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "RevokeIpRules": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html",
                "description": "Removes one or more rules from the specified IP access control group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            },
            "StartWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html",
                "description": "Starts the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "StopWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html",
                "description": "Stops the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "TerminateWorkspaces": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html",
                "description": "Terminates the specified WorkSpaces.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceid": {
                        "required": true
                    }
                }
            },
            "UpdateRulesOfIpGroup": {
                "url": "https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html",
                "description": "Replaces the current rules of the specified IP access control group with the specified rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workspaceipgroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "workspacebundle": {
                "name": "workspacebundle",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspacebundle/${BundleId}",
                "conditionKeys": []
            },
            "workspaceipgroup": {
                "name": "workspaceipgroup",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspaceipgroup/${GroupId}",
                "conditionKeys": []
            },
            "directoryid": {
                "name": "directoryid",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": []
            },
            "workspaceid": {
                "name": "workspaceid",
                "url": "https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html",
                "arn": "arn:${Partition}:workspaces:${Region}:${Account}:workspace/${WorkspaceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the specified IP access control group with the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html
     */
    toAssociateIpGroups() {
        this.add('workspaces:AssociateIpGroups');
        return this;
    }
    /**
     * Adds one or more rules to the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html
     */
    toAuthorizeIpRules() {
        this.add('workspaces:AuthorizeIpRules');
        return this;
    }
    /**
     * Creates an IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html
     */
    toCreateIpGroup() {
        this.add('workspaces:CreateIpGroup');
        return this;
    }
    /**
     * Creates tags for a WorkSpace.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html
     */
    toCreateTags() {
        this.add('workspaces:CreateTags');
        return this;
    }
    /**
     * Creates one or more WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html
     */
    toCreateWorkspaces() {
        this.add('workspaces:CreateWorkspaces');
        return this;
    }
    /**
     * Deletes the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html
     */
    toDeleteIpGroup() {
        this.add('workspaces:DeleteIpGroup');
        return this;
    }
    /**
     * Deletes tags from a Workspace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('workspaces:DeleteTags');
        return this;
    }
    /**
     * Deletes the specified workspace image.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html
     */
    toDeleteWorkspaceImage() {
        this.add('workspaces:DeleteWorkspaceImage');
        return this;
    }
    /**
     * Retrieves a list that describes the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html
     */
    toDescribeAccount() {
        this.add('workspaces:DescribeAccount');
        return this;
    }
    /**
     * Retrieves a list that describes modifications to the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html
     */
    toDescribeAccountModifications() {
        this.add('workspaces:DescribeAccountModifications');
        return this;
    }
    /**
     * Describe client properties about the specified resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html
     */
    toDescribeClientProperties() {
        this.add('workspaces:DescribeClientProperties');
        return this;
    }
    /**
     * Retrieves information about the IP access control groups of your account in the region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html
     */
    toDescribeIpGroups() {
        this.add('workspaces:DescribeIpGroups');
        return this;
    }
    /**
     * Describes tags for a WorkSpace.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('workspaces:DescribeTags');
        return this;
    }
    /**
     * Obtains information about the WorkSpace bundles that are available to your account in the specified region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html
     */
    toDescribeWorkspaceBundles() {
        this.add('workspaces:DescribeWorkspaceBundles');
        return this;
    }
    /**
     * Retrieves information about the AWS Directory Service directories in the region that are registered with Amazon WorkSpaces and are available to your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html
     */
    toDescribeWorkspaceDirectories() {
        this.add('workspaces:DescribeWorkspaceDirectories');
        return this;
    }
    /**
     * Retrieves a list that describes one or more specified images.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html
     */
    toDescribeWorkspaceImages() {
        this.add('workspaces:DescribeWorkspaceImages');
        return this;
    }
    /**
     * Obtains information about the specified WorkSpaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html
     */
    toDescribeWorkspaces() {
        this.add('workspaces:DescribeWorkspaces');
        return this;
    }
    /**
     * Describes the connection status of a specified WorkSpace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html
     */
    toDescribeWorkspacesConnectionStatus() {
        this.add('workspaces:DescribeWorkspacesConnectionStatus');
        return this;
    }
    /**
     * Disassociates the specified IP access control group from the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html
     */
    toDisassociateIpGroups() {
        this.add('workspaces:DisassociateIpGroups');
        return this;
    }
    /**
     * Import a licensed EC2 image to into Amazon WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html
     */
    toImportWorkspaceImage() {
        this.add('workspaces:ImportWorkspaceImage');
        return this;
    }
    /**
     * List available CIDR ranges for a CIDR range constraint.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html
     */
    toListAvailableManagementCidrRanges() {
        this.add('workspaces:ListAvailableManagementCidrRanges');
        return this;
    }
    /**
     * Modify the configuration of bring your own license (BYOL) for the specified account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html
     */
    toModifyAccount() {
        this.add('workspaces:ModifyAccount');
        return this;
    }
    /**
     * Modify the client properties of a specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html
     */
    toModifyClientProperties() {
        this.add('workspaces:ModifyClientProperties');
        return this;
    }
    /**
     * Modifies the WorkSpace properties, including the running mode and AutoStop time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html
     */
    toModifyWorkspaceProperties() {
        this.add('workspaces:ModifyWorkspaceProperties');
        return this;
    }
    /**
     * Modify the state of specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html
     */
    toModifyWorkspaceState() {
        this.add('workspaces:ModifyWorkspaceState');
        return this;
    }
    /**
     * Reboots the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html
     */
    toRebootWorkspaces() {
        this.add('workspaces:RebootWorkspaces');
        return this;
    }
    /**
     * Rebuilds the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html
     */
    toRebuildWorkspaces() {
        this.add('workspaces:RebuildWorkspaces');
        return this;
    }
    /**
     * Removes one or more rules from the specified IP access control group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html
     */
    toRevokeIpRules() {
        this.add('workspaces:RevokeIpRules');
        return this;
    }
    /**
     * Starts the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html
     */
    toStartWorkspaces() {
        this.add('workspaces:StartWorkspaces');
        return this;
    }
    /**
     * Stops the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html
     */
    toStopWorkspaces() {
        this.add('workspaces:StopWorkspaces');
        return this;
    }
    /**
     * Terminates the specified WorkSpaces.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html
     */
    toTerminateWorkspaces() {
        this.add('workspaces:TerminateWorkspaces');
        return this;
    }
    /**
     * Replaces the current rules of the specified IP access control group with the specified rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html
     */
    toUpdateRulesOfIpGroup() {
        this.add('workspaces:UpdateRulesOfIpGroup');
        return this;
    }
    /**
     * Adds a resource of type workspacebundle to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param bundleId - Identifier for the bundleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspacebundle(bundleId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspacebundle/${BundleId}';
        arn = arn.replace('${BundleId}', bundleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceipgroup to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceipgroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspaceipgroup/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directoryid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectoryid(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workspaceid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkspaceid(workspaceId, account, region, partition) {
        var arn = 'arn:${Partition}:workspaces:${Region}:${Account}:workspace/${WorkspaceId}';
        arn = arn.replace('${WorkspaceId}', workspaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Workspaces = Workspaces;
//# sourceMappingURL=data:application/json;base64,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