"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Batch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [batch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Batch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [batch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'batch';
        this.actionList = {
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_CancelJob.html",
                "description": "Cancels a job in an AWS Batch job queue.",
                "accessLevel": "Write"
            },
            "CreateComputeEnvironment": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html",
                "description": "Creates an AWS Batch compute environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compute-environment": {
                        "required": true
                    }
                }
            },
            "CreateJobQueue": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateJobQueue.html",
                "description": "Creates an AWS Batch job queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compute-environment": {
                        "required": true
                    },
                    "job-queue": {
                        "required": true
                    }
                }
            },
            "DeleteComputeEnvironment": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteComputeEnvironment.html",
                "description": "Deletes an AWS Batch compute environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compute-environment": {
                        "required": true
                    }
                }
            },
            "DeleteJobQueue": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteJobQueue.html",
                "description": "Deletes the specified job queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job-queue": {
                        "required": true
                    }
                }
            },
            "DeregisterJobDefinition": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DeregisterJobDefinition.html",
                "description": "Deregisters an AWS Batch job definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job-definition": {
                        "required": true
                    }
                }
            },
            "DescribeComputeEnvironments": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeComputeEnvironments.html",
                "description": "Describes one or more of your compute environments.",
                "accessLevel": "Read"
            },
            "DescribeJobDefinitions": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobDefinitions.html",
                "description": "Describes a list of job definitions.",
                "accessLevel": "Read"
            },
            "DescribeJobQueues": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobQueues.html",
                "description": "Describes one or more of your job queues.",
                "accessLevel": "Read"
            },
            "DescribeJobs": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobs.html",
                "description": "Describes a list of AWS Batch jobs.",
                "accessLevel": "Read"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_ListJobs.html",
                "description": "Returns a list of task jobs for a specified job queue.",
                "accessLevel": "List"
            },
            "RegisterJobDefinition": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_RegisterJobDefinition.html",
                "description": "Registers an AWS Batch job definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job-definition": {
                        "required": true
                    }
                },
                "conditions": [
                    "batch:User",
                    "batch:Privileged",
                    "batch:Image"
                ]
            },
            "SubmitJob": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html",
                "description": "Submits an AWS Batch job from a job definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job-definition": {
                        "required": true
                    },
                    "job-queue": {
                        "required": true
                    }
                }
            },
            "TerminateJob": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_TerminateJob.html",
                "description": "Terminates jobs in a job queue.",
                "accessLevel": "Write"
            },
            "UpdateComputeEnvironment": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html",
                "description": "Updates an AWS Batch compute environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compute-environment": {
                        "required": true
                    }
                }
            },
            "UpdateJobQueue": {
                "url": "https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateJobQueue.html",
                "description": "Updates a job queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job-queue": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "compute-environment": {
                "name": "compute-environment",
                "url": "",
                "arn": "arn:${Partition}:batch:${Region}:${Account}:compute-environment/${ComputeEnvironmentName}",
                "conditionKeys": []
            },
            "job-queue": {
                "name": "job-queue",
                "url": "",
                "arn": "arn:${Partition}:batch:${Region}:${Account}:job-queue/${JobQueueName}",
                "conditionKeys": []
            },
            "job-definition": {
                "name": "job-definition",
                "url": "",
                "arn": "arn:${Partition}:batch:${Region}:${Account}:job-definition/${JobDefinitionName}:${Revision}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels a job in an AWS Batch job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CancelJob.html
     */
    toCancelJob() {
        this.add('batch:CancelJob');
        return this;
    }
    /**
     * Creates an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html
     */
    toCreateComputeEnvironment() {
        this.add('batch:CreateComputeEnvironment');
        return this;
    }
    /**
     * Creates an AWS Batch job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateJobQueue.html
     */
    toCreateJobQueue() {
        this.add('batch:CreateJobQueue');
        return this;
    }
    /**
     * Deletes an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteComputeEnvironment.html
     */
    toDeleteComputeEnvironment() {
        this.add('batch:DeleteComputeEnvironment');
        return this;
    }
    /**
     * Deletes the specified job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteJobQueue.html
     */
    toDeleteJobQueue() {
        this.add('batch:DeleteJobQueue');
        return this;
    }
    /**
     * Deregisters an AWS Batch job definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeregisterJobDefinition.html
     */
    toDeregisterJobDefinition() {
        this.add('batch:DeregisterJobDefinition');
        return this;
    }
    /**
     * Describes one or more of your compute environments.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeComputeEnvironments.html
     */
    toDescribeComputeEnvironments() {
        this.add('batch:DescribeComputeEnvironments');
        return this;
    }
    /**
     * Describes a list of job definitions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobDefinitions.html
     */
    toDescribeJobDefinitions() {
        this.add('batch:DescribeJobDefinitions');
        return this;
    }
    /**
     * Describes one or more of your job queues.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobQueues.html
     */
    toDescribeJobQueues() {
        this.add('batch:DescribeJobQueues');
        return this;
    }
    /**
     * Describes a list of AWS Batch jobs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobs.html
     */
    toDescribeJobs() {
        this.add('batch:DescribeJobs');
        return this;
    }
    /**
     * Returns a list of task jobs for a specified job queue.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListJobs.html
     */
    toListJobs() {
        this.add('batch:ListJobs');
        return this;
    }
    /**
     * Registers an AWS Batch job definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUser()
     * - .ifPrivileged()
     * - .ifImage()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_RegisterJobDefinition.html
     */
    toRegisterJobDefinition() {
        this.add('batch:RegisterJobDefinition');
        return this;
    }
    /**
     * Submits an AWS Batch job from a job definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html
     */
    toSubmitJob() {
        this.add('batch:SubmitJob');
        return this;
    }
    /**
     * Terminates jobs in a job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_TerminateJob.html
     */
    toTerminateJob() {
        this.add('batch:TerminateJob');
        return this;
    }
    /**
     * Updates an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html
     */
    toUpdateComputeEnvironment() {
        this.add('batch:UpdateComputeEnvironment');
        return this;
    }
    /**
     * Updates a job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateJobQueue.html
     */
    toUpdateJobQueue() {
        this.add('batch:UpdateJobQueue');
        return this;
    }
    /**
     * Adds a resource of type compute-environment to the statement
     *
     * @param computeEnvironmentName - Identifier for the computeEnvironmentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onComputeEnvironment(computeEnvironmentName, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:compute-environment/${ComputeEnvironmentName}';
        arn = arn.replace('${ComputeEnvironmentName}', computeEnvironmentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job-queue to the statement
     *
     * @param jobQueueName - Identifier for the jobQueueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobQueue(jobQueueName, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:job-queue/${JobQueueName}';
        arn = arn.replace('${JobQueueName}', jobQueueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job-definition to the statement
     *
     * @param jobDefinitionName - Identifier for the jobDefinitionName.
     * @param revision - Identifier for the revision.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobDefinition(jobDefinitionName, revision, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:job-definition/${JobDefinitionName}:${Revision}';
        arn = arn.replace('${JobDefinitionName}', jobDefinitionName);
        arn = arn.replace('${Revision}', revision);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The image used to start a container.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImage(value, operator) {
        return this.if(`batch:Image`, value, operator || 'StringLike');
    }
    /**
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPrivileged(value) {
        return this.if(`batch:Privileged`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The user name or numeric uid to use inside the container.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUser(value, operator) {
        return this.if(`batch:User`, value, operator || 'StringLike');
    }
}
exports.Batch = Batch;
//# sourceMappingURL=data:application/json;base64,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