"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chatbot = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chatbot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awschatbot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chatbot extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chatbot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awschatbot.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chatbot';
        this.actionList = {
            "CreateChimeWebhookConfiguration": {
                "url": "",
                "description": "Creates an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "CreateSlackChannelConfiguration": {
                "url": "",
                "description": "Creates an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            },
            "DeleteChimeWebhookConfiguration": {
                "url": "",
                "description": "Deletes an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "DeleteSlackChannelConfiguration": {
                "url": "",
                "description": "Deletes an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            },
            "DescribeChimeWebhookConfigurations": {
                "url": "",
                "description": "Lists all AWS Chatbot Chime Webhook Configurations in an AWS Account.",
                "accessLevel": "Read"
            },
            "DescribeSlackChannelConfigurations": {
                "url": "",
                "description": "Lists all AWS Chatbot Slack Channel Configurations in an AWS account.",
                "accessLevel": "Read"
            },
            "DescribeSlackChannels": {
                "url": "",
                "description": "Lists all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "DescribeSlackWorkspaces": {
                "url": "",
                "description": "Lists all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "GetSlackOauthParameters": {
                "url": "",
                "description": "Generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "RedeemSlackOauthCode": {
                "url": "",
                "description": "Redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service.",
                "accessLevel": "Write"
            },
            "UpdateChimeWebhookConfiguration": {
                "url": "",
                "description": "Updates an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "UpdateSlackChannelConfiguration": {
                "url": "",
                "description": "Updates an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "ChatbotConfiguration": {
                "name": "ChatbotConfiguration",
                "url": "",
                "arn": "arn:${Partition}:chatbot::${Account}:${ResourceType}/${ResourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toCreateChimeWebhookConfiguration() {
        this.add('chatbot:CreateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Creates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toCreateSlackChannelConfiguration() {
        this.add('chatbot:CreateSlackChannelConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toDeleteChimeWebhookConfiguration() {
        this.add('chatbot:DeleteChimeWebhookConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toDeleteSlackChannelConfiguration() {
        this.add('chatbot:DeleteSlackChannelConfiguration');
        return this;
    }
    /**
     * Lists all AWS Chatbot Chime Webhook Configurations in an AWS Account.
     *
     * Access Level: Read
     */
    toDescribeChimeWebhookConfigurations() {
        this.add('chatbot:DescribeChimeWebhookConfigurations');
        return this;
    }
    /**
     * Lists all AWS Chatbot Slack Channel Configurations in an AWS account.
     *
     * Access Level: Read
     */
    toDescribeSlackChannelConfigurations() {
        this.add('chatbot:DescribeSlackChannelConfigurations');
        return this;
    }
    /**
     * Lists all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    toDescribeSlackChannels() {
        this.add('chatbot:DescribeSlackChannels');
        return this;
    }
    /**
     * Lists all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    toDescribeSlackWorkspaces() {
        this.add('chatbot:DescribeSlackWorkspaces');
        return this;
    }
    /**
     * Generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service.
     *
     * Access Level: Read
     */
    toGetSlackOauthParameters() {
        this.add('chatbot:GetSlackOauthParameters');
        return this;
    }
    /**
     * Redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service.
     *
     * Access Level: Write
     */
    toRedeemSlackOauthCode() {
        this.add('chatbot:RedeemSlackOauthCode');
        return this;
    }
    /**
     * Updates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    toUpdateChimeWebhookConfiguration() {
        this.add('chatbot:UpdateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Updates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    toUpdateSlackChannelConfiguration() {
        this.add('chatbot:UpdateSlackChannelConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ChatbotConfiguration to the statement
     *
     * @param resourceType - Identifier for the resourceType.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChatbotConfiguration(resourceType, resourceName, account, partition) {
        var arn = 'arn:${Partition}:chatbot::${Account}:${ResourceType}/${ResourceName}';
        arn = arn.replace('${ResourceType}', resourceType);
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chatbot = Chatbot;
//# sourceMappingURL=data:application/json;base64,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