"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicediscovery = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicediscovery extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicediscovery](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudmap.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicediscovery';
        this.actionList = {
            "CreateHttpNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html",
                "description": "Creates an HTTP namespace.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePrivateDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html",
                "description": "Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreatePublicDnsNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html",
                "description": "Creates a public namespace based on DNS, which will be visible on the internet.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html",
                "description": "Creates a service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:NamespaceArn",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html",
                "description": "Deletes a specified namespace.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "DeleteService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html",
                "description": "Deletes a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "DeregisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html",
                "description": "Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "DiscoverInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html",
                "description": "Discovers registered instances for a specified namespace and service.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:NamespaceName",
                    "servicediscovery:ServiceName"
                ]
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html",
                "description": "Gets information about a specified instance.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetInstancesHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html",
                "description": "Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.",
                "accessLevel": "Read",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "GetNamespace": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html",
                "description": "Gets information about a namespace.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "namespace": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html",
                "description": "Gets information about a specific operation.",
                "accessLevel": "Read"
            },
            "GetService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html",
                "description": "Gets the settings for a specified service.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            },
            "ListInstances": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html",
                "description": "Gets summary information about the instances that were registered with a specified service.",
                "accessLevel": "List",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "ListNamespaces": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html",
                "description": "Gets information about the namespaces.",
                "accessLevel": "List"
            },
            "ListOperations": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html",
                "description": "Lists operations that match the criteria that you specify.",
                "accessLevel": "List"
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html",
                "description": "Gets settings for all the services that match specified filters.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html",
                "description": "Lists tags for the specified resource.",
                "accessLevel": "List"
            },
            "RegisterInstance": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html",
                "description": "Registers an instance based on the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html",
                "description": "Adds one or more tags to the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html",
                "description": "Removes one or more tags from the specified resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateInstanceCustomHealthStatus": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html",
                "description": "Updates the current health status for an instance that has a custom health check.",
                "accessLevel": "Write",
                "conditions": [
                    "servicediscovery:ServiceArn"
                ]
            },
            "UpdateService": {
                "url": "https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html",
                "description": "Updates the settings in a specified service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "namespace": {
                "name": "namespace",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "service": {
                "name": "service",
                "url": "https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html",
                "arn": "arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an HTTP namespace.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateHttpNamespace.html
     */
    toCreateHttpNamespace() {
        this.add('servicediscovery:CreateHttpNamespace');
        return this;
    }
    /**
     * Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePrivateDnsNamespace.html
     */
    toCreatePrivateDnsNamespace() {
        this.add('servicediscovery:CreatePrivateDnsNamespace');
        return this;
    }
    /**
     * Creates a public namespace based on DNS, which will be visible on the internet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreatePublicDnsNamespace.html
     */
    toCreatePublicDnsNamespace() {
        this.add('servicediscovery:CreatePublicDnsNamespace');
        return this;
    }
    /**
     * Creates a service.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespaceArn()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html
     */
    toCreateService() {
        this.add('servicediscovery:CreateService');
        return this;
    }
    /**
     * Deletes a specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        this.add('servicediscovery:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeleteService.html
     */
    toDeleteService() {
        this.add('servicediscovery:DeleteService');
        return this;
    }
    /**
     * Deletes the records and the health check, if any, that Amazon Route 53 created for the specified instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DeregisterInstance.html
     */
    toDeregisterInstance() {
        this.add('servicediscovery:DeregisterInstance');
        return this;
    }
    /**
     * Discovers registered instances for a specified namespace and service.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespaceName()
     * - .ifServiceName()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html
     */
    toDiscoverInstances() {
        this.add('servicediscovery:DiscoverInstances');
        return this;
    }
    /**
     * Gets information about a specified instance.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstance.html
     */
    toGetInstance() {
        this.add('servicediscovery:GetInstance');
        return this;
    }
    /**
     * Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetInstancesHealthStatus.html
     */
    toGetInstancesHealthStatus() {
        this.add('servicediscovery:GetInstancesHealthStatus');
        return this;
    }
    /**
     * Gets information about a namespace.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetNamespace.html
     */
    toGetNamespace() {
        this.add('servicediscovery:GetNamespace');
        return this;
    }
    /**
     * Gets information about a specific operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetOperation.html
     */
    toGetOperation() {
        this.add('servicediscovery:GetOperation');
        return this;
    }
    /**
     * Gets the settings for a specified service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_GetService.html
     */
    toGetService() {
        this.add('servicediscovery:GetService');
        return this;
    }
    /**
     * Gets summary information about the instances that were registered with a specified service.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListInstances.html
     */
    toListInstances() {
        this.add('servicediscovery:ListInstances');
        return this;
    }
    /**
     * Gets information about the namespaces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListNamespaces.html
     */
    toListNamespaces() {
        this.add('servicediscovery:ListNamespaces');
        return this;
    }
    /**
     * Lists operations that match the criteria that you specify.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html
     */
    toListOperations() {
        this.add('servicediscovery:ListOperations');
        return this;
    }
    /**
     * Gets settings for all the services that match specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListServices.html
     */
    toListServices() {
        this.add('servicediscovery:ListServices');
        return this;
    }
    /**
     * Lists tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('servicediscovery:ListTagsForResource');
        return this;
    }
    /**
     * Registers an instance based on the settings in a specified service.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html
     */
    toRegisterInstance() {
        this.add('servicediscovery:RegisterInstance');
        return this;
    }
    /**
     * Adds one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.add('servicediscovery:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.add('servicediscovery:UntagResource');
        return this;
    }
    /**
     * Updates the current health status for an instance that has a custom health check.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServiceArn()
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateInstanceCustomHealthStatus.html
     */
    toUpdateInstanceCustomHealthStatus() {
        this.add('servicediscovery:UpdateInstanceCustomHealthStatus');
        return this;
    }
    /**
     * Updates the settings in a specified service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud-map/latest/api/API_UpdateService.html
     */
    toUpdateService() {
        this.add('servicediscovery:UpdateService');
        return this;
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Namespace.html
     *
     * @param namespaceId - Identifier for the namespaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNamespace(namespaceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:namespace/${NamespaceId}';
        arn = arn.replace('${NamespaceId}', namespaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/API_Service.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onService(serviceId, account, region, partition) {
        var arn = 'arn:${Partition}:servicediscovery:${Region}:${Account}:service/${ServiceId}';
        arn = arn.replace('${ServiceId}', serviceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateHttpNamespace()
     * - .toCreatePrivateDnsNamespace()
     * - .toCreatePublicDnsNamespace()
     * - .toCreateService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - namespace
     * - service
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateHttpNamespace()
     * - .toCreatePrivateDnsNamespace()
     * - .toCreatePublicDnsNamespace()
     * - .toCreateService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toCreateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceArn(value, operator) {
        return this.if(`servicediscovery:NamespaceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related namespace.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDiscoverInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceName(value, operator) {
        return this.if(`servicediscovery:NamespaceName`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the Amazon Resource Name (ARN) for the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDeregisterInstance()
     * - .toGetInstance()
     * - .toGetInstancesHealthStatus()
     * - .toListInstances()
     * - .toRegisterInstance()
     * - .toUpdateInstanceCustomHealthStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`servicediscovery:ServiceArn`, value, operator || 'StringLike');
    }
    /**
     * A filter that lets you get objects by specifying the name of the related service.
     *
     * https://docs.aws.amazon.com/cloud-map/latest/dg/access-control-overview.html#specifying-conditions
     *
     * Applies to actions:
     * - .toDiscoverInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceName(value, operator) {
        return this.if(`servicediscovery:ServiceName`, value, operator || 'StringLike');
    }
}
exports.Servicediscovery = Servicediscovery;
//# sourceMappingURL=data:application/json;base64,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