"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codeartifact = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeartifact](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodeartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codeartifact extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeartifact](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodeartifact.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeartifact';
        this.actionList = {
            "AssociateExternalConnection": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html",
                "description": "Grants permission to add an external connection to a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "AssociateWithDownstreamRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repos-upstream.html",
                "description": "Grants permission to associate an existing repository as an upstream repository to another repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CopyPackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html",
                "description": "Grants permission to copy package versions from one repository to another repository in the same domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html",
                "description": "Grants permission to create a new domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html",
                "description": "Grants permission to create a new repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html",
                "description": "Grants permission to delete a domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html",
                "description": "Grants permission to delete the resource policy set on a domain",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeletePackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html",
                "description": "Grants permission to delete package versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "DeleteRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html",
                "description": "Grants permission to delete a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html",
                "description": "Grants permission to delete the resource policy set on a repository",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html",
                "description": "Grants permission to return information about a domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribePackageVersion": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html",
                "description": "Grants permission to return information about a package version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "DescribeRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html",
                "description": "Grants permission to return detailed information about a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DisassociateExternalConnection": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html",
                "description": "Grants permission to disassociate an external connection from a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DisposePackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html",
                "description": "Grants permission to set the status of package versions to Disposed and delete their assets",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetAuthorizationToken": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html",
                "description": "Grants permission to generate a temporary authentication token for accessing repositories in a domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html",
                "description": "Grants permission to return a domain's resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetPackageVersionAsset": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html",
                "description": "Grants permission to return an asset (or file) that is part of a package version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetPackageVersionReadme": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html",
                "description": "Grants permission to return a package version's readme file",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetRepositoryEndpoint": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html",
                "description": "Grants permission to return an endpoint for a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html",
                "description": "Grants permission to return a repository's resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html",
                "description": "Grants permission to list the domains in the current user's AWS account",
                "accessLevel": "List"
            },
            "ListPackageVersionAssets": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html",
                "description": "Grants permission to list a package version's assets",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackageVersionDependencies": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html",
                "description": "Grants permission to list the direct dependencies of a package version",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html",
                "description": "Grants permission to list a package's versions",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackages": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html",
                "description": "Grants permission to list the packages in a repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html",
                "description": "Grants permission to list the repositories administered by the calling account",
                "accessLevel": "List"
            },
            "ListRepositoriesInDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html",
                "description": "Grants permission to list the repositories in a domain",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PublishPackageVersion": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to publish assets and metadata to a repository endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "PutDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html",
                "description": "Grants permission to attach a resource policy to a domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PutPackageMetadata": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to add, modify or remove package metadata using a repository endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "PutRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html",
                "description": "Grants permission to attach a resource policy to a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ReadFromRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to return package assets and metadata from a repository endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePackageVersionsStatus": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html",
                "description": "Grants permission to modify the status of one or more versions of a package",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "UpdateRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html",
                "description": "Grants permission to modify the properties of a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            },
            "repository": {
                "name": "repository",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:repository/${DomainName}/${RepositoryName}",
                "conditionKeys": []
            },
            "package": {
                "name": "package",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:package/${DomainName}/${RepositoryName}/${PackageFormat}/${PackageNamespace}/${PackageName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add an external connection to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html
     */
    toAssociateExternalConnection() {
        this.add('codeartifact:AssociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to associate an existing repository as an upstream repository to another repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repos-upstream.html
     */
    toAssociateWithDownstreamRepository() {
        this.add('codeartifact:AssociateWithDownstreamRepository');
        return this;
    }
    /**
     * Grants permission to copy package versions from one repository to another repository in the same domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html
     */
    toCopyPackageVersions() {
        this.add('codeartifact:CopyPackageVersions');
        return this;
    }
    /**
     * Grants permission to create a new domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        this.add('codeartifact:CreateDomain');
        return this;
    }
    /**
     * Grants permission to create a new repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        this.add('codeartifact:CreateRepository');
        return this;
    }
    /**
     * Grants permission to delete a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.add('codeartifact:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a domain
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html
     */
    toDeleteDomainPermissionsPolicy() {
        this.add('codeartifact:DeleteDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to delete package versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html
     */
    toDeletePackageVersions() {
        this.add('codeartifact:DeletePackageVersions');
        return this;
    }
    /**
     * Grants permission to delete a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        this.add('codeartifact:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a repository
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html
     */
    toDeleteRepositoryPermissionsPolicy() {
        this.add('codeartifact:DeleteRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return information about a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        this.add('codeartifact:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to return information about a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html
     */
    toDescribePackageVersion() {
        this.add('codeartifact:DescribePackageVersion');
        return this;
    }
    /**
     * Grants permission to return detailed information about a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html
     */
    toDescribeRepository() {
        this.add('codeartifact:DescribeRepository');
        return this;
    }
    /**
     * Grants permission to disassociate an external connection from a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html
     */
    toDisassociateExternalConnection() {
        this.add('codeartifact:DisassociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to set the status of package versions to Disposed and delete their assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html
     */
    toDisposePackageVersions() {
        this.add('codeartifact:DisposePackageVersions');
        return this;
    }
    /**
     * Grants permission to generate a temporary authentication token for accessing repositories in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html
     */
    toGetAuthorizationToken() {
        this.add('codeartifact:GetAuthorizationToken');
        return this;
    }
    /**
     * Grants permission to return a domain's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html
     */
    toGetDomainPermissionsPolicy() {
        this.add('codeartifact:GetDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return an asset (or file) that is part of a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html
     */
    toGetPackageVersionAsset() {
        this.add('codeartifact:GetPackageVersionAsset');
        return this;
    }
    /**
     * Grants permission to return a package version's readme file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html
     */
    toGetPackageVersionReadme() {
        this.add('codeartifact:GetPackageVersionReadme');
        return this;
    }
    /**
     * Grants permission to return an endpoint for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html
     */
    toGetRepositoryEndpoint() {
        this.add('codeartifact:GetRepositoryEndpoint');
        return this;
    }
    /**
     * Grants permission to return a repository's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html
     */
    toGetRepositoryPermissionsPolicy() {
        this.add('codeartifact:GetRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to list the domains in the current user's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        this.add('codeartifact:ListDomains');
        return this;
    }
    /**
     * Grants permission to list a package version's assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html
     */
    toListPackageVersionAssets() {
        this.add('codeartifact:ListPackageVersionAssets');
        return this;
    }
    /**
     * Grants permission to list the direct dependencies of a package version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html
     */
    toListPackageVersionDependencies() {
        this.add('codeartifact:ListPackageVersionDependencies');
        return this;
    }
    /**
     * Grants permission to list a package's versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html
     */
    toListPackageVersions() {
        this.add('codeartifact:ListPackageVersions');
        return this;
    }
    /**
     * Grants permission to list the packages in a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html
     */
    toListPackages() {
        this.add('codeartifact:ListPackages');
        return this;
    }
    /**
     * Grants permission to list the repositories administered by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        this.add('codeartifact:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list the repositories in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html
     */
    toListRepositoriesInDomain() {
        this.add('codeartifact:ListRepositoriesInDomain');
        return this;
    }
    /**
     * Grants permission to publish assets and metadata to a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toPublishPackageVersion() {
        this.add('codeartifact:PublishPackageVersion');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html
     */
    toPutDomainPermissionsPolicy() {
        this.add('codeartifact:PutDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to add, modify or remove package metadata using a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toPutPackageMetadata() {
        this.add('codeartifact:PutPackageMetadata');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html
     */
    toPutRepositoryPermissionsPolicy() {
        this.add('codeartifact:PutRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return package assets and metadata from a repository endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    toReadFromRepository() {
        this.add('codeartifact:ReadFromRepository');
        return this;
    }
    /**
     * Grants permission to modify the status of one or more versions of a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html
     */
    toUpdatePackageVersionsStatus() {
        this.add('codeartifact:UpdatePackageVersionsStatus');
        return this;
    }
    /**
     * Grants permission to modify the properties of a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html
     */
    toUpdateRepository() {
        this.add('codeartifact:UpdateRepository');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRepository(domainName, repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:repository/${DomainName}/${RepositoryName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type package to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param packageFormat - Identifier for the packageFormat.
     * @param packageNamespace - Identifier for the packageNamespace.
     * @param packageName - Identifier for the packageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPackage(domainName, repositoryName, packageFormat, packageNamespace, packageName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:package/${DomainName}/${RepositoryName}/${PackageFormat}/${PackageNamespace}/${PackageName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${PackageFormat}', packageFormat);
        arn = arn.replace('${PackageNamespace}', packageNamespace);
        arn = arn.replace('${PackageName}', packageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codeartifact = Codeartifact;
//# sourceMappingURL=data:application/json;base64,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