"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deepracer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deepracer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deepracer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deepracer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deepracer';
        this.actionList = {
            "CloneReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model",
                "description": "Grants permission to clone existing DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    },
                    "track": {
                        "required": true
                    }
                }
            },
            "CreateAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to create resources needed by DeepRacer on behalf of the user",
                "accessLevel": "Write"
            },
            "CreateLeaderboardSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to submit DeepRacer models to be evaluated for leaderboards",
                "accessLevel": "Write",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    },
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "CreateReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to create reinforcement learning models for DeepRacer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "track": {
                        "required": true
                    }
                }
            },
            "DeleteAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to delete resources created by DeepRacer on behalf of the user",
                "accessLevel": "Write"
            },
            "DeleteModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to delete DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "GetAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve the resources created by DeepRacer on behalf of the user",
                "accessLevel": "Read"
            },
            "GetAlias": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve the user's alias for submitting DeepRacer models to leaderboards",
                "accessLevel": "Read"
            },
            "GetEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models' evaluation jobs",
                "accessLevel": "Read",
                "resourceTypes": {
                    "evaluation_job": {
                        "required": true
                    }
                }
            },
            "GetLatestUserSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetLeaderboard": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about leaderboards",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models",
                "accessLevel": "Read",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "GetRankedUserSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetTrack": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "description": "Grants permission to retrieve information about DeepRacer tracks",
                "accessLevel": "Read",
                "resourceTypes": {
                    "track": {
                        "required": true
                    }
                }
            },
            "GetTrainingJob": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models' training job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training_job": {
                        "required": true
                    }
                }
            },
            "ListEvaluations": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to list DeepRacer models' evaluation jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "ListLeaderboardSubmissions": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to list all the submissions of DeepRacer models of a user on a leaderboard",
                "accessLevel": "List",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "ListLeaderboards": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to list all the available leaderboards",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to list all existing DeepRacer models",
                "accessLevel": "List"
            },
            "ListTracks": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "description": "Grants permission to list all DeepRacer tracks",
                "accessLevel": "List"
            },
            "ListTrainingJobs": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to list DeepRacer models' training jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "SetAlias": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to set the user's alias for submitting DeepRacer models to leaderboards",
                "accessLevel": "Write"
            },
            "StartEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to evaluate DeepRacer models in a simulated environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    },
                    "track": {
                        "required": true
                    }
                }
            },
            "StopEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to stop DeepRacer model evaluations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "evaluation_job": {
                        "required": true
                    }
                }
            },
            "StopTrainingReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to stop training DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "TestRewardFunction": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function",
                "description": "Grants permission to test reward functions for correctness",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "reinforcement_learning_model": {
                "name": "reinforcement_learning_model",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:model/reinforcement_learning/${ResourceId}",
                "conditionKeys": []
            },
            "training_job": {
                "name": "training_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:training_job/${ResourceId}",
                "conditionKeys": []
            },
            "evaluation_job": {
                "name": "evaluation_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:evaluation_job/${ResourceId}",
                "conditionKeys": []
            },
            "leaderboard_evaluation_job": {
                "name": "leaderboard_evaluation_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:leaderboard_evaluation_job/${ResourceId}",
                "conditionKeys": []
            },
            "track": {
                "name": "track",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "arn": "arn:${Partition}:deepracer:${Region}::track/${ResourceId}",
                "conditionKeys": []
            },
            "leaderboard": {
                "name": "leaderboard",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "arn": "arn:${Partition}:deepracer:${Region}::leaderboard/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to clone existing DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model
     */
    toCloneReinforcementLearningModel() {
        this.add('deepracer:CloneReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to create resources needed by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toCreateAccountResources() {
        this.add('deepracer:CreateAccountResources');
        return this;
    }
    /**
     * Grants permission to submit DeepRacer models to be evaluated for leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toCreateLeaderboardSubmission() {
        this.add('deepracer:CreateLeaderboardSubmission');
        return this;
    }
    /**
     * Grants permission to create reinforcement learning models for DeepRacer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toCreateReinforcementLearningModel() {
        this.add('deepracer:CreateReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to delete resources created by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toDeleteAccountResources() {
        this.add('deepracer:DeleteAccountResources');
        return this;
    }
    /**
     * Grants permission to delete DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toDeleteModel() {
        this.add('deepracer:DeleteModel');
        return this;
    }
    /**
     * Grants permission to retrieve the resources created by DeepRacer on behalf of the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetAccountResources() {
        this.add('deepracer:GetAccountResources');
        return this;
    }
    /**
     * Grants permission to retrieve the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetAlias() {
        this.add('deepracer:GetAlias');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' evaluation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toGetEvaluation() {
        this.add('deepracer:GetEvaluation');
        return this;
    }
    /**
     * Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLatestUserSubmission() {
        this.add('deepracer:GetLatestUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLeaderboard() {
        this.add('deepracer:GetLeaderboard');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetModel() {
        this.add('deepracer:GetModel');
        return this;
    }
    /**
     * Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetRankedUserSubmission() {
        this.add('deepracer:GetRankedUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about DeepRacer tracks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toGetTrack() {
        this.add('deepracer:GetTrack');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' training job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetTrainingJob() {
        this.add('deepracer:GetTrainingJob');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' evaluation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toListEvaluations() {
        this.add('deepracer:ListEvaluations');
        return this;
    }
    /**
     * Grants permission to list all the submissions of DeepRacer models of a user on a leaderboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboardSubmissions() {
        this.add('deepracer:ListLeaderboardSubmissions');
        return this;
    }
    /**
     * Grants permission to list all the available leaderboards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboards() {
        this.add('deepracer:ListLeaderboards');
        return this;
    }
    /**
     * Grants permission to list all existing DeepRacer models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListModels() {
        this.add('deepracer:ListModels');
        return this;
    }
    /**
     * Grants permission to list all DeepRacer tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toListTracks() {
        this.add('deepracer:ListTracks');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' training jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListTrainingJobs() {
        this.add('deepracer:ListTrainingJobs');
        return this;
    }
    /**
     * Grants permission to set the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toSetAlias() {
        this.add('deepracer:SetAlias');
        return this;
    }
    /**
     * Grants permission to evaluate DeepRacer models in a simulated environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStartEvaluation() {
        this.add('deepracer:StartEvaluation');
        return this;
    }
    /**
     * Grants permission to stop DeepRacer model evaluations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStopEvaluation() {
        this.add('deepracer:StopEvaluation');
        return this;
    }
    /**
     * Grants permission to stop training DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toStopTrainingReinforcementLearningModel() {
        this.add('deepracer:StopTrainingReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to test reward functions for correctness
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function
     */
    toTestRewardFunction() {
        this.add('deepracer:TestRewardFunction');
        return this;
    }
    /**
     * Adds a resource of type reinforcement_learning_model to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReinforcementLearningModel(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:model/reinforcement_learning/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type training_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrainingJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:training_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard_evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboardEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:leaderboard_evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type track to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrack(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::track/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboard(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::leaderboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deepracer = Deepracer;
//# sourceMappingURL=data:application/json;base64,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