"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Directconnect = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [directconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Directconnect extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [directconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'directconnect';
        this.actionList = {
            "AcceptDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html",
                "description": "Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "AllocateConnectionOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html",
                "description": "Creates a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "AllocateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html",
                "description": "Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html",
                "description": "Provisions a private virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html",
                "description": "Provisions a public virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html",
                "description": "Provisions a transit virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AssociateConnectionWithLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html",
                "description": "Associates a connection with a LAG.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "AssociateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html",
                "description": "Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "AssociateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html",
                "description": "Associates a virtual interface with a specified link aggregation group (LAG) or connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    },
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "ConfirmConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html",
                "description": "Confirm the creation of a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "ConfirmPrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html",
                "description": "Accept ownership of a private virtual interface created by another customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmPublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html",
                "description": "Accept ownership of a public virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html",
                "description": "Accept ownership of a transit virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html",
                "description": "Creates a BGP peer on the specified virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html",
                "description": "Creates a new connection between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html",
                "description": "Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.",
                "accessLevel": "Write"
            },
            "CreateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html",
                "description": "Creates an association between a Direct Connect gateway and a virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html",
                "description": "Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html",
                "description": "Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html",
                "description": "Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html",
                "description": "Creates a new private virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html",
                "description": "Creates a new public virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html",
                "description": "Creates a new transit virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html",
                "description": "Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html",
                "description": "Deletes the connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html",
                "description": "Deletes the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html",
                "description": "Deletes the association between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html",
                "description": "Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write"
            },
            "DeleteInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html",
                "description": "Deletes the specified interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html",
                "description": "Deletes the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html",
                "description": "Deletes a virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DescribeConnectionLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html",
                "description": "Returns the LOA-CFA for a Connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html",
                "description": "Displays all connections in this region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeConnectionsOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html",
                "description": "Return a list of connections that have been provisioned on the given interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociationProposals": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html",
                "description": "Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html",
                "description": "Lists the associations between your Direct Connect gateways and virtual private gateways.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html",
                "description": "Lists the attachments between your Direct Connect gateways and virtual interfaces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html",
                "description": "Lists all your Direct Connect gateways or only the specified Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeHostedConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html",
                "description": "Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeInterconnectLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html",
                "description": "Returns the LOA-CFA for an Interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeInterconnects": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html",
                "description": "Returns a list of interconnects owned by the AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeLags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html",
                "description": "Describes all your link aggregation groups (LAG) or the specified LAG.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html",
                "description": "Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLocations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html",
                "description": "Returns the list of AWS Direct Connect locations in the current AWS region.",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the tags associated with the specified AWS Direct Connect resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DescribeVirtualGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html",
                "description": "Returns a list of virtual private gateways owned by the AWS account.",
                "accessLevel": "Read"
            },
            "DescribeVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html",
                "description": "Displays all virtual interfaces for an AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DisassociateConnectionFromLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html",
                "description": "Disassociates a connection from a link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "ListVirtualInterfaceTestHistory": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html",
                "description": "Lists the virtual interface failover test history.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StartBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html",
                "description": "Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StopBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html",
                "description": "Stops the virtual interface failover test.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html",
                "description": "Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html",
                "description": "Removes one or more tags from the specified AWS Direct Connect resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html",
                "description": "Updates the specified attributes of the Direct Connect gateway association.",
                "accessLevel": "Write"
            },
            "UpdateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html",
                "description": "Updates the attributes of the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualInterfaceAttributes": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html",
                "description": "Updates the specified attributes of the specified virtual private interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dxcon": {
                "name": "dxcon",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxlag": {
                "name": "dxlag",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxvif": {
                "name": "dxvif",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dx-gateway": {
                "name": "dx-gateway",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html",
                "arn": "arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html
     */
    toAcceptDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:AcceptDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html
     */
    toAllocateConnectionOnInterconnect() {
        this.add('directconnect:AllocateConnectionOnInterconnect');
        return this;
    }
    /**
     * Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html
     */
    toAllocateHostedConnection() {
        this.add('directconnect:AllocateHostedConnection');
        return this;
    }
    /**
     * Provisions a private virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html
     */
    toAllocatePrivateVirtualInterface() {
        this.add('directconnect:AllocatePrivateVirtualInterface');
        return this;
    }
    /**
     * Provisions a public virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html
     */
    toAllocatePublicVirtualInterface() {
        this.add('directconnect:AllocatePublicVirtualInterface');
        return this;
    }
    /**
     * Provisions a transit virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html
     */
    toAllocateTransitVirtualInterface() {
        this.add('directconnect:AllocateTransitVirtualInterface');
        return this;
    }
    /**
     * Associates a connection with a LAG.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html
     */
    toAssociateConnectionWithLag() {
        this.add('directconnect:AssociateConnectionWithLag');
        return this;
    }
    /**
     * Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html
     */
    toAssociateHostedConnection() {
        this.add('directconnect:AssociateHostedConnection');
        return this;
    }
    /**
     * Associates a virtual interface with a specified link aggregation group (LAG) or connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html
     */
    toAssociateVirtualInterface() {
        this.add('directconnect:AssociateVirtualInterface');
        return this;
    }
    /**
     * Confirm the creation of a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html
     */
    toConfirmConnection() {
        this.add('directconnect:ConfirmConnection');
        return this;
    }
    /**
     * Accept ownership of a private virtual interface created by another customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html
     */
    toConfirmPrivateVirtualInterface() {
        this.add('directconnect:ConfirmPrivateVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a public virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html
     */
    toConfirmPublicVirtualInterface() {
        this.add('directconnect:ConfirmPublicVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a transit virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html
     */
    toConfirmTransitVirtualInterface() {
        this.add('directconnect:ConfirmTransitVirtualInterface');
        return this;
    }
    /**
     * Creates a BGP peer on the specified virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html
     */
    toCreateBGPPeer() {
        this.add('directconnect:CreateBGPPeer');
        return this;
    }
    /**
     * Creates a new connection between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        this.add('directconnect:CreateConnection');
        return this;
    }
    /**
     * Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html
     */
    toCreateDirectConnectGateway() {
        this.add('directconnect:CreateDirectConnectGateway');
        return this;
    }
    /**
     * Creates an association between a Direct Connect gateway and a virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html
     */
    toCreateDirectConnectGatewayAssociation() {
        this.add('directconnect:CreateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html
     */
    toCreateDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:CreateDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html
     */
    toCreateInterconnect() {
        this.add('directconnect:CreateInterconnect');
        return this;
    }
    /**
     * Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html
     */
    toCreateLag() {
        this.add('directconnect:CreateLag');
        return this;
    }
    /**
     * Creates a new private virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html
     */
    toCreatePrivateVirtualInterface() {
        this.add('directconnect:CreatePrivateVirtualInterface');
        return this;
    }
    /**
     * Creates a new public virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html
     */
    toCreatePublicVirtualInterface() {
        this.add('directconnect:CreatePublicVirtualInterface');
        return this;
    }
    /**
     * Creates a new transit virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html
     */
    toCreateTransitVirtualInterface() {
        this.add('directconnect:CreateTransitVirtualInterface');
        return this;
    }
    /**
     * Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html
     */
    toDeleteBGPPeer() {
        this.add('directconnect:DeleteBGPPeer');
        return this;
    }
    /**
     * Deletes the connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        this.add('directconnect:DeleteConnection');
        return this;
    }
    /**
     * Deletes the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html
     */
    toDeleteDirectConnectGateway() {
        this.add('directconnect:DeleteDirectConnectGateway');
        return this;
    }
    /**
     * Deletes the association between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html
     */
    toDeleteDirectConnectGatewayAssociation() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html
     */
    toDeleteDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Deletes the specified interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html
     */
    toDeleteInterconnect() {
        this.add('directconnect:DeleteInterconnect');
        return this;
    }
    /**
     * Deletes the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html
     */
    toDeleteLag() {
        this.add('directconnect:DeleteLag');
        return this;
    }
    /**
     * Deletes a virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html
     */
    toDeleteVirtualInterface() {
        this.add('directconnect:DeleteVirtualInterface');
        return this;
    }
    /**
     * Returns the LOA-CFA for a Connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html
     */
    toDescribeConnectionLoa() {
        this.add('directconnect:DescribeConnectionLoa');
        return this;
    }
    /**
     * Displays all connections in this region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        this.add('directconnect:DescribeConnections');
        return this;
    }
    /**
     * Return a list of connections that have been provisioned on the given interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html
     */
    toDescribeConnectionsOnInterconnect() {
        this.add('directconnect:DescribeConnectionsOnInterconnect');
        return this;
    }
    /**
     * Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html
     */
    toDescribeDirectConnectGatewayAssociationProposals() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociationProposals');
        return this;
    }
    /**
     * Lists the associations between your Direct Connect gateways and virtual private gateways.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html
     */
    toDescribeDirectConnectGatewayAssociations() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociations');
        return this;
    }
    /**
     * Lists the attachments between your Direct Connect gateways and virtual interfaces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html
     */
    toDescribeDirectConnectGatewayAttachments() {
        this.add('directconnect:DescribeDirectConnectGatewayAttachments');
        return this;
    }
    /**
     * Lists all your Direct Connect gateways or only the specified Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html
     */
    toDescribeDirectConnectGateways() {
        this.add('directconnect:DescribeDirectConnectGateways');
        return this;
    }
    /**
     * Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html
     */
    toDescribeHostedConnections() {
        this.add('directconnect:DescribeHostedConnections');
        return this;
    }
    /**
     * Returns the LOA-CFA for an Interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html
     */
    toDescribeInterconnectLoa() {
        this.add('directconnect:DescribeInterconnectLoa');
        return this;
    }
    /**
     * Returns a list of interconnects owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html
     */
    toDescribeInterconnects() {
        this.add('directconnect:DescribeInterconnects');
        return this;
    }
    /**
     * Describes all your link aggregation groups (LAG) or the specified LAG.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html
     */
    toDescribeLags() {
        this.add('directconnect:DescribeLags');
        return this;
    }
    /**
     * Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html
     */
    toDescribeLoa() {
        this.add('directconnect:DescribeLoa');
        return this;
    }
    /**
     * Returns the list of AWS Direct Connect locations in the current AWS region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html
     */
    toDescribeLocations() {
        this.add('directconnect:DescribeLocations');
        return this;
    }
    /**
     * Describes the tags associated with the specified AWS Direct Connect resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('directconnect:DescribeTags');
        return this;
    }
    /**
     * Returns a list of virtual private gateways owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html
     */
    toDescribeVirtualGateways() {
        this.add('directconnect:DescribeVirtualGateways');
        return this;
    }
    /**
     * Displays all virtual interfaces for an AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html
     */
    toDescribeVirtualInterfaces() {
        this.add('directconnect:DescribeVirtualInterfaces');
        return this;
    }
    /**
     * Disassociates a connection from a link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html
     */
    toDisassociateConnectionFromLag() {
        this.add('directconnect:DisassociateConnectionFromLag');
        return this;
    }
    /**
     * Lists the virtual interface failover test history.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html
     */
    toListVirtualInterfaceTestHistory() {
        this.add('directconnect:ListVirtualInterfaceTestHistory');
        return this;
    }
    /**
     * Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html
     */
    toStartBgpFailoverTest() {
        this.add('directconnect:StartBgpFailoverTest');
        return this;
    }
    /**
     * Stops the virtual interface failover test.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html
     */
    toStopBgpFailoverTest() {
        this.add('directconnect:StopBgpFailoverTest');
        return this;
    }
    /**
     * Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('directconnect:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS Direct Connect resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('directconnect:UntagResource');
        return this;
    }
    /**
     * Updates the specified attributes of the Direct Connect gateway association.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html
     */
    toUpdateDirectConnectGatewayAssociation() {
        this.add('directconnect:UpdateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Updates the attributes of the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html
     */
    toUpdateLag() {
        this.add('directconnect:UpdateLag');
        return this;
    }
    /**
     * Updates the specified attributes of the specified virtual private interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html
     */
    toUpdateVirtualInterfaceAttributes() {
        this.add('directconnect:UpdateVirtualInterfaceAttributes');
        return this;
    }
    /**
     * Adds a resource of type dxcon to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxcon(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxlag to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html
     *
     * @param lagId - Identifier for the lagId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxlag(lagId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}';
        arn = arn.replace('${LagId}', lagId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxvif to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html
     *
     * @param virtualInterfaceId - Identifier for the virtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxvif(virtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}';
        arn = arn.replace('${VirtualInterfaceId}', virtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dx-gateway to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html
     *
     * @param directConnectGatewayId - Identifier for the directConnectGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDxGateway(directConnectGatewayId, account, partition) {
        var arn = 'arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}';
        arn = arn.replace('${DirectConnectGatewayId}', directConnectGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAllocateHostedConnection()
     * - .toAllocatePrivateVirtualInterface()
     * - .toAllocatePublicVirtualInterface()
     * - .toAllocateTransitVirtualInterface()
     * - .toCreateConnection()
     * - .toCreateInterconnect()
     * - .toCreateLag()
     * - .toCreatePrivateVirtualInterface()
     * - .toCreatePublicVirtualInterface()
     * - .toCreateTransitVirtualInterface()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - dxcon
     * - dxlag
     * - dxvif
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAllocateHostedConnection()
     * - .toAllocatePrivateVirtualInterface()
     * - .toAllocatePublicVirtualInterface()
     * - .toAllocateTransitVirtualInterface()
     * - .toCreateConnection()
     * - .toCreateInterconnect()
     * - .toCreateLag()
     * - .toCreatePrivateVirtualInterface()
     * - .toCreatePublicVirtualInterface()
     * - .toCreateTransitVirtualInterface()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Directconnect = Directconnect;
//# sourceMappingURL=data:application/json;base64,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