"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ds = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ds';
        this.actionList = {
            "AcceptSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html",
                "description": "Accepts a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html",
                "description": "Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services",
                "accessLevel": "Write",
                "dependentActions": [
                    "ec2:AuthorizeSecurityGroupEgress",
                    "ec2:AuthorizeSecurityGroupIngress",
                    "ec2:DescribeSecurityGroups"
                ],
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html",
                "description": "Adds or overwrites one or more tags for the specified Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "dependentActions": [
                    "ec2:CreateTags"
                ],
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AuthorizeApplication": {
                "url": "",
                "description": "Authorizes an application for your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CancelSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html",
                "description": "Cancels an in-progress schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CheckAlias": {
                "url": "",
                "description": "Verifies that the alias is available for use.",
                "accessLevel": "Read"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html",
                "description": "Creates an AD Connector to connect to an on-premises directory.",
                "accessLevel": "Tagging",
                "dependentActions": [
                    "ec2:AuthorizeSecurityGroupEgress",
                    "ec2:AuthorizeSecurityGroupIngress",
                    "ec2:CreateNetworkInterface",
                    "ec2:CreateSecurityGroup",
                    "ec2:CreateTags",
                    "ec2:DescribeNetworkInterfaces",
                    "ec2:DescribeSubnets",
                    "ec2:DescribeVpcs"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html",
                "description": "Creates an alias for a directory and assigns the alias to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateComputer": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html",
                "description": "Creates a computer account in the specified directory, and joins the computer to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html",
                "description": "Creates a conditional forwarder associated with your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html",
                "description": "Creates a Simple AD directory.",
                "accessLevel": "Tagging",
                "dependentActions": [
                    "ec2:AuthorizeSecurityGroupEgress",
                    "ec2:AuthorizeSecurityGroupIngress",
                    "ec2:CreateNetworkInterface",
                    "ec2:CreateSecurityGroup",
                    "ec2:CreateTags",
                    "ec2:DescribeNetworkInterfaces",
                    "ec2:DescribeSubnets",
                    "ec2:DescribeVpcs"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateIdentityPoolDirectory": {
                "url": "",
                "description": "Creates a IdentityPool Directory in the AWS cloud.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html",
                "description": "Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateMicrosoftAD": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html",
                "description": "Creates a Microsoft AD in the AWS cloud.",
                "accessLevel": "Tagging",
                "dependentActions": [
                    "ec2:AuthorizeSecurityGroupEgress",
                    "ec2:AuthorizeSecurityGroupIngress",
                    "ec2:CreateNetworkInterface",
                    "ec2:CreateSecurityGroup",
                    "ec2:CreateTags",
                    "ec2:DescribeNetworkInterfaces",
                    "ec2:DescribeSubnets",
                    "ec2:DescribeVpcs"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html",
                "description": "Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html",
                "description": "Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html",
                "description": "Deletes a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html",
                "description": "Deletes an AWS Directory Service directory.",
                "accessLevel": "Write",
                "dependentActions": [
                    "ec2:DeleteNetworkInterface",
                    "ec2:DeleteSecurityGroup",
                    "ec2:DescribeNetworkInterfaces",
                    "ec2:RevokeSecurityGroupEgress",
                    "ec2:RevokeSecurityGroupIngress"
                ],
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html",
                "description": "Deletes the specified log subscription.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html",
                "description": "Deletes a directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html",
                "description": "Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html",
                "description": "Deletes from the system the certificate that was registered for a secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html",
                "description": "Removes the specified directory as a publisher to the specified SNS topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html",
                "description": "Displays information about the certificate registered for a secured LDAP connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeConditionalForwarders": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html",
                "description": "Obtains information about the conditional forwarders for this account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html",
                "description": "Obtains information about the directories that belong to this account.",
                "accessLevel": "List"
            },
            "DescribeDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html",
                "description": "Provides information about any domain controllers in your directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeEventTopics": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html",
                "description": "Obtains information about which SNS topics receive status messages from the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeLDAPSSettings": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html",
                "description": "Describes the status of LDAP security for the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSharedDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html",
                "description": "Returns the shared directories in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html",
                "description": "Obtains information about the directory snapshots that belong to this account.",
                "accessLevel": "Read"
            },
            "DescribeTrusts": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html",
                "description": "Obtains information about the trust relationships for this account.",
                "accessLevel": "Read"
            },
            "DisableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html",
                "description": "Deactivates LDAP secure calls for the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html",
                "description": "Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html",
                "description": "Disables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html",
                "description": "Activates the switch for the specific directory to always use LDAP secure calls.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html",
                "description": "Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html",
                "description": "Enables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetAuthorizedApplicationDetails": {
                "url": "",
                "description": "",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetDirectoryLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html",
                "description": "Obtains directory limit information for the current region.",
                "accessLevel": "Read"
            },
            "GetSnapshotLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html",
                "description": "Obtains the manual snapshot limits for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListAuthorizedApplications": {
                "url": "",
                "description": "Obtains the aws applications authorized for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html",
                "description": "For the specified directory, lists all the certificates registered for a secured LDAP connection.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html",
                "description": "Lists the address blocks that you have added to a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListLogSubscriptions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html",
                "description": "Lists the active log subscriptions for the AWS account.",
                "accessLevel": "Read"
            },
            "ListSchemaExtensions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html",
                "description": "Lists all schema extensions applied to a Microsoft AD Directory.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html",
                "description": "Lists all tags on an Amazon Directory Services directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html",
                "description": "Registers a certificate for secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html",
                "description": "Associates a directory with an SNS topic.",
                "accessLevel": "Write",
                "dependentActions": [
                    "sns:GetTopicAttributes"
                ],
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RejectSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html",
                "description": "Rejects a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html",
                "description": "Removes IP address blocks from a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html",
                "description": "Removes tags from an Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "dependentActions": [
                    "ec2:DeleteTags"
                ],
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResetUserPassword": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html",
                "description": "Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RestoreFromSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html",
                "description": "Restores a directory using an existing directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ShareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html",
                "description": "Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "StartSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html",
                "description": "Applies a schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnauthorizeApplication": {
                "url": "",
                "description": "Unauthorizes an application from your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnshareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html",
                "description": "Stops the directory sharing between the directory owner and consumer accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html",
                "description": "Updates a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateNumberOfDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html",
                "description": "Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html",
                "description": "Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html",
                "description": "Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "VerifyTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html",
                "description": "Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "directory": {
                "name": "directory",
                "url": "",
                "arn": "arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Accepts a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    toAcceptSharedDirectory() {
        this.add('ds:AcceptSharedDirectory');
        return this;
    }
    /**
     * Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:DescribeSecurityGroups
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    toAddIpRoutes() {
        this.add('ds:AddIpRoutes');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.add('ds:AddTagsToResource');
        return this;
    }
    /**
     * Authorizes an application for your AWS Directory.
     *
     * Access Level: Write
     */
    toAuthorizeApplication() {
        this.add('ds:AuthorizeApplication');
        return this;
    }
    /**
     * Cancels an in-progress schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    toCancelSchemaExtension() {
        this.add('ds:CancelSchemaExtension');
        return this;
    }
    /**
     * Verifies that the alias is available for use.
     *
     * Access Level: Read
     */
    toCheckAlias() {
        this.add('ds:CheckAlias');
        return this;
    }
    /**
     * Creates an AD Connector to connect to an on-premises directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    toConnectDirectory() {
        this.add('ds:ConnectDirectory');
        return this;
    }
    /**
     * Creates an alias for a directory and assigns the alias to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    toCreateAlias() {
        this.add('ds:CreateAlias');
        return this;
    }
    /**
     * Creates a computer account in the specified directory, and joins the computer to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    toCreateComputer() {
        this.add('ds:CreateComputer');
        return this;
    }
    /**
     * Creates a conditional forwarder associated with your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    toCreateConditionalForwarder() {
        this.add('ds:CreateConditionalForwarder');
        return this;
    }
    /**
     * Creates a Simple AD directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    toCreateDirectory() {
        this.add('ds:CreateDirectory');
        return this;
    }
    /**
     * Creates a IdentityPool Directory in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     */
    toCreateIdentityPoolDirectory() {
        this.add('ds:CreateIdentityPoolDirectory');
        return this;
    }
    /**
     * Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    toCreateLogSubscription() {
        this.add('ds:CreateLogSubscription');
        return this;
    }
    /**
     * Creates a Microsoft AD in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    toCreateMicrosoftAD() {
        this.add('ds:CreateMicrosoftAD');
        return this;
    }
    /**
     * Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.add('ds:CreateSnapshot');
        return this;
    }
    /**
     * Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    toCreateTrust() {
        this.add('ds:CreateTrust');
        return this;
    }
    /**
     * Deletes a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    toDeleteConditionalForwarder() {
        this.add('ds:DeleteConditionalForwarder');
        return this;
    }
    /**
     * Deletes an AWS Directory Service directory.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteSecurityGroup
     * - ec2:DescribeNetworkInterfaces
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RevokeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        this.add('ds:DeleteDirectory');
        return this;
    }
    /**
     * Deletes the specified log subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    toDeleteLogSubscription() {
        this.add('ds:DeleteLogSubscription');
        return this;
    }
    /**
     * Deletes a directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.add('ds:DeleteSnapshot');
        return this;
    }
    /**
     * Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html
     */
    toDeleteTrust() {
        this.add('ds:DeleteTrust');
        return this;
    }
    /**
     * Deletes from the system the certificate that was registered for a secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    toDeregisterCertificate() {
        this.add('ds:DeregisterCertificate');
        return this;
    }
    /**
     * Removes the specified directory as a publisher to the specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    toDeregisterEventTopic() {
        this.add('ds:DeregisterEventTopic');
        return this;
    }
    /**
     * Displays information about the certificate registered for a secured LDAP connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        this.add('ds:DescribeCertificate');
        return this;
    }
    /**
     * Obtains information about the conditional forwarders for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    toDescribeConditionalForwarders() {
        this.add('ds:DescribeConditionalForwarders');
        return this;
    }
    /**
     * Obtains information about the directories that belong to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    toDescribeDirectories() {
        this.add('ds:DescribeDirectories');
        return this;
    }
    /**
     * Provides information about any domain controllers in your directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    toDescribeDomainControllers() {
        this.add('ds:DescribeDomainControllers');
        return this;
    }
    /**
     * Obtains information about which SNS topics receive status messages from the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    toDescribeEventTopics() {
        this.add('ds:DescribeEventTopics');
        return this;
    }
    /**
     * Describes the status of LDAP security for the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    toDescribeLDAPSSettings() {
        this.add('ds:DescribeLDAPSSettings');
        return this;
    }
    /**
     * Returns the shared directories in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    toDescribeSharedDirectories() {
        this.add('ds:DescribeSharedDirectories');
        return this;
    }
    /**
     * Obtains information about the directory snapshots that belong to this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.add('ds:DescribeSnapshots');
        return this;
    }
    /**
     * Obtains information about the trust relationships for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    toDescribeTrusts() {
        this.add('ds:DescribeTrusts');
        return this;
    }
    /**
     * Deactivates LDAP secure calls for the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    toDisableLDAPS() {
        this.add('ds:DisableLDAPS');
        return this;
    }
    /**
     * Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    toDisableRadius() {
        this.add('ds:DisableRadius');
        return this;
    }
    /**
     * Disables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    toDisableSso() {
        this.add('ds:DisableSso');
        return this;
    }
    /**
     * Activates the switch for the specific directory to always use LDAP secure calls.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    toEnableLDAPS() {
        this.add('ds:EnableLDAPS');
        return this;
    }
    /**
     * Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    toEnableRadius() {
        this.add('ds:EnableRadius');
        return this;
    }
    /**
     * Enables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    toEnableSso() {
        this.add('ds:EnableSso');
        return this;
    }
    /**
     *
     *
     * Access Level: Read
     */
    toGetAuthorizedApplicationDetails() {
        this.add('ds:GetAuthorizedApplicationDetails');
        return this;
    }
    /**
     * Obtains directory limit information for the current region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    toGetDirectoryLimits() {
        this.add('ds:GetDirectoryLimits');
        return this;
    }
    /**
     * Obtains the manual snapshot limits for a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    toGetSnapshotLimits() {
        this.add('ds:GetSnapshotLimits');
        return this;
    }
    /**
     * Obtains the aws applications authorized for a directory.
     *
     * Access Level: Read
     */
    toListAuthorizedApplications() {
        this.add('ds:ListAuthorizedApplications');
        return this;
    }
    /**
     * For the specified directory, lists all the certificates registered for a secured LDAP connection.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    toListCertificates() {
        this.add('ds:ListCertificates');
        return this;
    }
    /**
     * Lists the address blocks that you have added to a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    toListIpRoutes() {
        this.add('ds:ListIpRoutes');
        return this;
    }
    /**
     * Lists the active log subscriptions for the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    toListLogSubscriptions() {
        this.add('ds:ListLogSubscriptions');
        return this;
    }
    /**
     * Lists all schema extensions applied to a Microsoft AD Directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    toListSchemaExtensions() {
        this.add('ds:ListSchemaExtensions');
        return this;
    }
    /**
     * Lists all tags on an Amazon Directory Services directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('ds:ListTagsForResource');
        return this;
    }
    /**
     * Registers a certificate for secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    toRegisterCertificate() {
        this.add('ds:RegisterCertificate');
        return this;
    }
    /**
     * Associates a directory with an SNS topic.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    toRegisterEventTopic() {
        this.add('ds:RegisterEventTopic');
        return this;
    }
    /**
     * Rejects a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    toRejectSharedDirectory() {
        this.add('ds:RejectSharedDirectory');
        return this;
    }
    /**
     * Removes IP address blocks from a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    toRemoveIpRoutes() {
        this.add('ds:RemoveIpRoutes');
        return this;
    }
    /**
     * Removes tags from an Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DeleteTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.add('ds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    toResetUserPassword() {
        this.add('ds:ResetUserPassword');
        return this;
    }
    /**
     * Restores a directory using an existing directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    toRestoreFromSnapshot() {
        this.add('ds:RestoreFromSnapshot');
        return this;
    }
    /**
     * Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    toShareDirectory() {
        this.add('ds:ShareDirectory');
        return this;
    }
    /**
     * Applies a schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    toStartSchemaExtension() {
        this.add('ds:StartSchemaExtension');
        return this;
    }
    /**
     * Unauthorizes an application from your AWS Directory.
     *
     * Access Level: Write
     */
    toUnauthorizeApplication() {
        this.add('ds:UnauthorizeApplication');
        return this;
    }
    /**
     * Stops the directory sharing between the directory owner and consumer accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    toUnshareDirectory() {
        this.add('ds:UnshareDirectory');
        return this;
    }
    /**
     * Updates a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    toUpdateConditionalForwarder() {
        this.add('ds:UpdateConditionalForwarder');
        return this;
    }
    /**
     * Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    toUpdateNumberOfDomainControllers() {
        this.add('ds:UpdateNumberOfDomainControllers');
        return this;
    }
    /**
     * Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    toUpdateRadius() {
        this.add('ds:UpdateRadius');
        return this;
    }
    /**
     * Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    toUpdateTrust() {
        this.add('ds:UpdateTrust');
        return this;
    }
    /**
     * Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    toVerifyTrust() {
        this.add('ds:VerifyTrust');
        return this;
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Applies to resource types:
     * - directory
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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