import { Actions, PolicyStatement, PolicyStatementWithCondition, ResourceTypes } from "../shared";
/**
 * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Mediaconvert extends PolicyStatement {
    servicePrefix: string;
    protected actionList: Actions;
    protected resourceTypes: ResourceTypes;
    /**
     * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     */
    toAssociateCertificate(): this;
    /**
     * Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCancelJob(): this;
    /**
     * Grants permission to create and submit an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toCreateJob(): this;
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCreateJobTemplate(): this;
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toCreatePreset(): this;
    /**
     * Grants permission to create an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toCreateQueue(): this;
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toDeleteJobTemplate(): this;
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toDeletePreset(): this;
    /**
     * Grants permission to delete an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toDeleteQueue(): this;
    /**
     * Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html
     */
    toDescribeEndpoints(): this;
    /**
     * Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html
     */
    toDisassociateCertificate(): this;
    /**
     * Grants permission to get an AWS Elemental MediaConvert job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toGetJob(): this;
    /**
     * Grants permission to get an AWS Elemental MediaConvert job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toGetJobTemplate(): this;
    /**
     * Grants permission to get an AWS Elemental MediaConvert output preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toGetPreset(): this;
    /**
     * Grants permission to get an AWS Elemental MediaConvert job queue
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toGetQueue(): this;
    /**
     * Grants permission to list AWS Elemental MediaConvert job templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     */
    toListJobTemplates(): this;
    /**
     * Grants permission to list AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toListJobs(): this;
    /**
     * Grants permission to list AWS Elemental MediaConvert output presets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toListPresets(): this;
    /**
     * Grants permission to list AWS Elemental MediaConvert job queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toListQueues(): this;
    /**
     * Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to add tags to a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toUpdateJobTemplate(): this;
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toUpdatePreset(): this;
    /**
     * Grants permission to update an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toUpdateQueue(): this;
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type Queue to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(queueName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type Preset to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     *
     * @param presetName - Identifier for the presetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPreset(presetName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type JobTemplate to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     *
     * @param jobTemplateName - Identifier for the jobTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobTemplate(jobTemplateName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type CertificateAssociation to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     *
     * @param certificateArn - Identifier for the certificateArn.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAssociation(certificateArn: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: string): PolicyStatementWithCondition;
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to resource types:
     * - Queue
     * - Preset
     * - JobTemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: string): PolicyStatementWithCondition;
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: string): PolicyStatementWithCondition;
}
