"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconvert = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconvert extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediaconvert';
        this.actionList = {
            "AssociateCertificate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html",
                "description": "Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Job": {
                        "required": true
                    }
                }
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "description": "Grants permission to create and submit an AWS Elemental MediaConvert job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "CreateJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    }
                }
            },
            "DeletePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "DeleteQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoints": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html",
                "description": "Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.",
                "accessLevel": "List"
            },
            "DisassociateCertificate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html",
                "description": "Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.",
                "accessLevel": "Write"
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Job": {
                        "required": true
                    }
                }
            },
            "GetJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    }
                }
            },
            "GetPreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert output preset",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "GetQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job queue",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            },
            "ListJobTemplates": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html",
                "description": "Grants permission to list AWS Elemental MediaConvert job templates",
                "accessLevel": "List"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "description": "Grants permission to list AWS Elemental MediaConvert jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "Queue": {
                        "required": false
                    }
                }
            },
            "ListPresets": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "description": "Grants permission to list AWS Elemental MediaConvert output presets",
                "accessLevel": "List"
            },
            "ListQueues": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "description": "Grants permission to list AWS Elemental MediaConvert job queues",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html",
                "description": "Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html",
                "description": "Grants permission to add tags to a MediaConvert queue, preset, or job template",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html",
                "description": "Grants permission to remove tags from a MediaConvert queue, preset, or job template",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "UpdatePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "UpdateQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "Job": {
                "name": "Job",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:jobs/${JobId}",
                "conditionKeys": []
            },
            "Queue": {
                "name": "Queue",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:queues/${QueueName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Preset": {
                "name": "Preset",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:presets/${PresetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "JobTemplate": {
                "name": "JobTemplate",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:jobTemplates/${JobTemplateName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CertificateAssociation": {
                "name": "CertificateAssociation",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:certificates/${CertificateArn}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     */
    toAssociateCertificate() {
        this.add('mediaconvert:AssociateCertificate');
        return this;
    }
    /**
     * Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCancelJob() {
        this.add('mediaconvert:CancelJob');
        return this;
    }
    /**
     * Grants permission to create and submit an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toCreateJob() {
        this.add('mediaconvert:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCreateJobTemplate() {
        this.add('mediaconvert:CreateJobTemplate');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toCreatePreset() {
        this.add('mediaconvert:CreatePreset');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toCreateQueue() {
        this.add('mediaconvert:CreateQueue');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toDeleteJobTemplate() {
        this.add('mediaconvert:DeleteJobTemplate');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toDeletePreset() {
        this.add('mediaconvert:DeletePreset');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toDeleteQueue() {
        this.add('mediaconvert:DeleteQueue');
        return this;
    }
    /**
     * Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html
     */
    toDescribeEndpoints() {
        this.add('mediaconvert:DescribeEndpoints');
        return this;
    }
    /**
     * Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html
     */
    toDisassociateCertificate() {
        this.add('mediaconvert:DisassociateCertificate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toGetJob() {
        this.add('mediaconvert:GetJob');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toGetJobTemplate() {
        this.add('mediaconvert:GetJobTemplate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert output preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toGetPreset() {
        this.add('mediaconvert:GetPreset');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job queue
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toGetQueue() {
        this.add('mediaconvert:GetQueue');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     */
    toListJobTemplates() {
        this.add('mediaconvert:ListJobTemplates');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toListJobs() {
        this.add('mediaconvert:ListJobs');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert output presets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toListPresets() {
        this.add('mediaconvert:ListPresets');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toListQueues() {
        this.add('mediaconvert:ListQueues');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toListTagsForResource() {
        this.add('mediaconvert:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     */
    toTagResource() {
        this.add('mediaconvert:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toUntagResource() {
        this.add('mediaconvert:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toUpdateJobTemplate() {
        this.add('mediaconvert:UpdateJobTemplate');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toUpdatePreset() {
        this.add('mediaconvert:UpdatePreset');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toUpdateQueue() {
        this.add('mediaconvert:UpdateQueue');
        return this;
    }
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Queue to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(queueName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:queues/${QueueName}';
        arn = arn.replace('${QueueName}', queueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Preset to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     *
     * @param presetName - Identifier for the presetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPreset(presetName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:presets/${PresetName}';
        arn = arn.replace('${PresetName}', presetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type JobTemplate to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     *
     * @param jobTemplateName - Identifier for the jobTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobTemplate(jobTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobTemplates/${JobTemplateName}';
        arn = arn.replace('${JobTemplateName}', jobTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CertificateAssociation to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     *
     * @param certificateArn - Identifier for the certificateArn.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAssociation(certificateArn, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:certificates/${CertificateArn}';
        arn = arn.replace('${CertificateArn}', certificateArn);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to resource types:
     * - Queue
     * - Preset
     * - JobTemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Mediaconvert = Mediaconvert;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzZWxlbWVudGFsbWVkaWFjb252ZXJ0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzZWxlbWVudGFsbWVkaWFjb252ZXJ0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNDQUFrRztBQUVsRzs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFpVC9DOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBdFROLGtCQUFhLEdBQUcsY0FBYyxDQUFDO1FBQzVCLGVBQVUsR0FBWTtZQUM5QixzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLDZIQUE2SDtnQkFDNUksYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLHdGQUF3RjtnQkFDdkcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLDBFQUEwRTtnQkFDekYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSwrRUFBK0U7Z0JBQzlGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixhQUFhLEVBQUUsZ0ZBQWdGO2dCQUMvRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGFBQWEsRUFBRSxxRUFBcUU7Z0JBQ3BGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLCtFQUErRTtnQkFDOUYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGdGQUFnRjtnQkFDL0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLGtOQUFrTjtnQkFDak8sYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLCtLQUErSztnQkFDOUwsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLGtFQUFrRTtnQkFDakYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLG9FQUFvRTtnQkFDbkYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLDBFQUEwRTtnQkFDakYsYUFBYSxFQUFFLGlFQUFpRTtnQkFDaEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLDBGQUEwRjtnQkFDekcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGFBQWEsRUFBRSxnRkFBZ0Y7Z0JBQy9GLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSw0RUFBNEU7Z0JBQ25GLGFBQWEsRUFBRSxxRkFBcUY7Z0JBQ3BHLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsK0VBQStFO2dCQUM5RixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGdGQUFnRjtnQkFDL0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsS0FBSyxFQUFFO2dCQUNMLE1BQU0sRUFBRSxLQUFLO2dCQUNiLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLEtBQUssRUFBRSxrRUFBa0U7Z0JBQ3pFLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxPQUFPO2dCQUNmLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLFFBQVE7Z0JBQ2hCLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsTUFBTSxFQUFFLGFBQWE7Z0JBQ3JCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsTUFBTSxFQUFFLHdCQUF3QjtnQkFDaEMsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsZUFBZSxFQUFFLEVBQUU7YUFDcEI7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNqRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxRQUFRO1FBQ2IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxLQUFLLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLElBQUksR0FBRyxHQUFHLGtFQUFrRSxDQUFDO1FBQzdFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JGLElBQUksR0FBRyxHQUFHLHdFQUF3RSxDQUFDO1FBQ25GLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM3QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFFBQVEsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZGLElBQUksR0FBRyxHQUFHLDBFQUEwRSxDQUFDO1FBQ3JGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pHLElBQUksR0FBRyxHQUFHLG9GQUFvRixDQUFDO1FBQy9GLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLG9CQUFvQixFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ3pELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCLENBQUMsY0FBc0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRyxJQUFJLEdBQUcsR0FBRyxtRkFBbUYsQ0FBQztRQUM5RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN2RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQWlCO1FBQ2hGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQWlCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztDQUNGO0FBeHhCRCxvQ0F3eEJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWN0aW9ucywgUG9saWN5U3RhdGVtZW50LCBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uLCBSZXNvdXJjZVR5cGVzIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVkaWFjb252ZXJ0XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NlbGVtZW50YWxtZWRpYWNvbnZlcnQuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBNZWRpYWNvbnZlcnQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdtZWRpYWNvbnZlcnQnO1xuICBwcm90ZWN0ZWQgYWN0aW9uTGlzdDogQWN0aW9ucyA9IHtcbiAgICBcIkFzc29jaWF0ZUNlcnRpZmljYXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2NlcnRpZmljYXRlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGFuIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIHdpdGggQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJDYW5jZWxKb2JcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy1pZC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY2FuY2VsIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiB0aGF0IGlzIHdhaXRpbmcgaW4gcXVldWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJKb2JcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUpvYlwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9qb2JzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW5kIHN1Ym1pdCBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBqb2JcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJKb2JUZW1wbGF0ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlByZXNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlF1ZXVlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlSm9iVGVtcGxhdGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy1pZC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGN1c3RvbSBqb2IgdGVtcGxhdGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJQcmVzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZVByZXNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgY3VzdG9tIG91dHB1dCBwcmVzZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZVF1ZXVlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3F1ZXVlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiBxdWV1ZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVsZXRlSm9iVGVtcGxhdGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9idGVtcGxhdGVzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBjdXN0b20gam9iIHRlbXBsYXRlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSm9iVGVtcGxhdGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVByZXNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBjdXN0b20gb3V0cHV0IHByZXNldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlByZXNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlUXVldWVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcXVldWVzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBqb2IgcXVldWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVFbmRwb2ludHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvZW5kcG9pbnRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzdWJzY3JpYmUgdG8gdGhlIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IHNlcnZpY2UsIGJ5IHNlbmRpbmcgYSByZXF1ZXN0IGZvciBhbiBhY2NvdW50LXNwZWNpZmljIGVuZHBvaW50LiBBbGwgdHJhbnNjb2RpbmcgcmVxdWVzdHMgbXVzdCBiZSBzZW50IHRvIHRoZSBlbmRwb2ludCB0aGF0IHRoZSBzZXJ2aWNlIHJldHVybnMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZUNlcnRpZmljYXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2NlcnRpZmljYXRlcy1hcm4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSBhbiBhc3NvY2lhdGlvbiBiZXR3ZWVuIHRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiBhbiBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlciAoQUNNKSBjZXJ0aWZpY2F0ZSBhbmQgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJHZXRKb2JcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy1pZC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSm9iXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRKb2JUZW1wbGF0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9qb2J0ZW1wbGF0ZXMtbmFtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiB0ZW1wbGF0ZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSm9iVGVtcGxhdGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFByZXNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBvdXRwdXQgcHJlc2V0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJQcmVzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFF1ZXVlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3F1ZXVlcy1uYW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9iIHF1ZXVlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdEpvYlRlbXBsYXRlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9qb2J0ZW1wbGF0ZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9iIHRlbXBsYXRlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0Sm9ic1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9qb2JzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlF1ZXVlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdFByZXNldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcHJlc2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBvdXRwdXQgcHJlc2V0c1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0UXVldWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3F1ZXVlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBqb2IgcXVldWVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RUYWdzRm9yUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvdGFncy1hcm4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSB0YWdzIGZvciBhIE1lZGlhQ29udmVydCBxdWV1ZSwgcHJlc2V0LCBvciBqb2IgdGVtcGxhdGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkpvYlRlbXBsYXRlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiUHJlc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiUXVldWVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJUYWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS90YWdzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byBhIE1lZGlhQ29udmVydCBxdWV1ZSwgcHJlc2V0LCBvciBqb2IgdGVtcGxhdGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkpvYlRlbXBsYXRlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiUHJlc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiUXVldWVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVbnRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3RhZ3MtYXJuLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGEgTWVkaWFDb252ZXJ0IHF1ZXVlLCBwcmVzZXQsIG9yIGpvYiB0ZW1wbGF0ZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSm9iVGVtcGxhdGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJQcmVzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZUpvYlRlbXBsYXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnRlbXBsYXRlcy1uYW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgY3VzdG9tIGpvYiB0ZW1wbGF0ZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkpvYlRlbXBsYXRlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJQcmVzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVByZXNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBjdXN0b20gb3V0cHV0IHByZXNldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlByZXNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUXVldWVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcXVldWVzLW5hbWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBqb2IgcXVldWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJRdWV1ZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG4gIHByb3RlY3RlZCByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiSm9iXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIkpvYlwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bWVkaWFjb252ZXJ0OiR7UmVnaW9ufToke0FjY291bnR9OmpvYnMvJHtKb2JJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJRdWV1ZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJRdWV1ZVwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcXVldWVzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTptZWRpYWNvbnZlcnQ6JHtSZWdpb259OiR7QWNjb3VudH06cXVldWVzLyR7UXVldWVOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUHJlc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIlByZXNldFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcHJlc2V0cy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bWVkaWFjb252ZXJ0OiR7UmVnaW9ufToke0FjY291bnR9OnByZXNldHMvJHtQcmVzZXROYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiSm9iVGVtcGxhdGVcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiSm9iVGVtcGxhdGVcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnRlbXBsYXRlcy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bWVkaWFjb252ZXJ0OiR7UmVnaW9ufToke0FjY291bnR9OmpvYlRlbXBsYXRlcy8ke0pvYlRlbXBsYXRlTmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNlcnRpZmljYXRlQXNzb2NpYXRpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiQ2VydGlmaWNhdGVBc3NvY2lhdGlvblwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvY2VydGlmaWNhdGVzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTptZWRpYWNvbnZlcnQ6JHtSZWdpb259OiR7QWNjb3VudH06Y2VydGlmaWNhdGVzLyR7Q2VydGlmaWNhdGVBcm59XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVkaWFjb252ZXJ0XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NlbGVtZW50YWxtZWRpYWNvbnZlcnQuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGFuIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIHdpdGggQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2NlcnRpZmljYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVDZXJ0aWZpY2F0ZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OkFzc29jaWF0ZUNlcnRpZmljYXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2FuY2VsIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiB0aGF0IGlzIHdhaXRpbmcgaW4gcXVldWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy1pZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxKb2IoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpDYW5jZWxKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW5kIHN1Ym1pdCBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVKb2IoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpDcmVhdGVKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgY3VzdG9tIGpvYiB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9qb2JzLWlkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUpvYlRlbXBsYXRlKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6Q3JlYXRlSm9iVGVtcGxhdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgY3VzdG9tIG91dHB1dCBwcmVzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcHJlc2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVQcmVzZXQoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpDcmVhdGVQcmVzZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9iIHF1ZXVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3F1ZXVlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVRdWV1ZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OkNyZWF0ZVF1ZXVlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGN1c3RvbSBqb2IgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9idGVtcGxhdGVzLW5hbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSm9iVGVtcGxhdGUoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpEZWxldGVKb2JUZW1wbGF0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBjdXN0b20gb3V0cHV0IHByZXNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLW5hbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUHJlc2V0KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6RGVsZXRlUHJlc2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiBxdWV1ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9xdWV1ZXMtbmFtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVRdWV1ZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OkRlbGV0ZVF1ZXVlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3Vic2NyaWJlIHRvIHRoZSBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBzZXJ2aWNlLCBieSBzZW5kaW5nIGEgcmVxdWVzdCBmb3IgYW4gYWNjb3VudC1zcGVjaWZpYyBlbmRwb2ludC4gQWxsIHRyYW5zY29kaW5nIHJlcXVlc3RzIG11c3QgYmUgc2VudCB0byB0aGUgZW5kcG9pbnQgdGhhdCB0aGUgc2VydmljZSByZXR1cm5zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2VuZHBvaW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUVuZHBvaW50cygpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OkRlc2NyaWJlRW5kcG9pbnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIGFuIGFzc29jaWF0aW9uIGJldHdlZW4gdGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIG9mIGFuIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIGNlcnRpZmljYXRlIGFuZCBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvY2VydGlmaWNhdGVzLWFybi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVDZXJ0aWZpY2F0ZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OkRpc2Fzc29jaWF0ZUNlcnRpZmljYXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnMtaWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Sm9iKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6R2V0Sm9iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnRlbXBsYXRlcy1uYW1lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEpvYlRlbXBsYXRlKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6R2V0Sm9iVGVtcGxhdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgb3V0cHV0IHByZXNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3ByZXNldHMtbmFtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQcmVzZXQoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpHZXRQcmVzZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYW4gQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9iIHF1ZXVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvcXVldWVzLW5hbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UXVldWUoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpHZXRRdWV1ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9iIHRlbXBsYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnRlbXBsYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Sm9iVGVtcGxhdGVzKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6TGlzdEpvYlRlbXBsYXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgam9ic1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEpvYnMoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpMaXN0Sm9icycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgQVdTIEVsZW1lbnRhbCBNZWRpYUNvbnZlcnQgb3V0cHV0IHByZXNldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQcmVzZXRzKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6TGlzdFByZXNldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiBxdWV1ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9xdWV1ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFF1ZXVlcygpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0Okxpc3RRdWV1ZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgdGFncyBmb3IgYSBNZWRpYUNvbnZlcnQgcXVldWUsIHByZXNldCwgb3Igam9iIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2UvdGFncy1hcm4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0Okxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byBhIE1lZGlhQ29udmVydCBxdWV1ZSwgcHJlc2V0LCBvciBqb2IgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS90YWdzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGEgTWVkaWFDb252ZXJ0IHF1ZXVlLCBwcmVzZXQsIG9yIGpvYiB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3RhZ3MtYXJuLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGN1c3RvbSBqb2IgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFjb252ZXJ0L2xhdGVzdC9hcGlyZWZlcmVuY2Uvam9idGVtcGxhdGVzLW5hbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSm9iVGVtcGxhdGUoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhY29udmVydDpVcGRhdGVKb2JUZW1wbGF0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBV1MgRWxlbWVudGFsIE1lZGlhQ29udmVydCBjdXN0b20gb3V0cHV0IHByZXNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLW5hbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUHJlc2V0KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYWNvbnZlcnQ6VXBkYXRlUHJlc2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIEFXUyBFbGVtZW50YWwgTWVkaWFDb252ZXJ0IGpvYiBxdWV1ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9xdWV1ZXMtbmFtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVRdWV1ZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFjb252ZXJ0OlVwZGF0ZVF1ZXVlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgSm9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Kb2Ioam9iSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bWVkaWFjb252ZXJ0OiR7UmVnaW9ufToke0FjY291bnR9OmpvYnMvJHtKb2JJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0pvYklkfScsIGpvYklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBRdWV1ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9xdWV1ZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcXVldWVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHF1ZXVlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25RdWV1ZShxdWV1ZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bWVkaWFjb252ZXJ0OiR7UmVnaW9ufToke0FjY291bnR9OnF1ZXVlcy8ke1F1ZXVlTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1F1ZXVlTmFtZX0nLCBxdWV1ZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFByZXNldCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9wcmVzZXRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHByZXNldE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcHJlc2V0TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25QcmVzZXQocHJlc2V0TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTptZWRpYWNvbnZlcnQ6JHtSZWdpb259OiR7QWNjb3VudH06cHJlc2V0cy8ke1ByZXNldE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQcmVzZXROYW1lfScsIHByZXNldE5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEpvYlRlbXBsYXRlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL2pvYnRlbXBsYXRlcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBqb2JUZW1wbGF0ZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iVGVtcGxhdGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkpvYlRlbXBsYXRlKGpvYlRlbXBsYXRlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTptZWRpYWNvbnZlcnQ6JHtSZWdpb259OiR7QWNjb3VudH06am9iVGVtcGxhdGVzLyR7Sm9iVGVtcGxhdGVOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Sm9iVGVtcGxhdGVOYW1lfScsIGpvYlRlbXBsYXRlTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQ2VydGlmaWNhdGVBc3NvY2lhdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWNvbnZlcnQvbGF0ZXN0L2FwaXJlZmVyZW5jZS9jZXJ0aWZpY2F0ZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY2VydGlmaWNhdGVBcm4gLSBJZGVudGlmaWVyIGZvciB0aGUgY2VydGlmaWNhdGVBcm4uXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25DZXJ0aWZpY2F0ZUFzc29jaWF0aW9uKGNlcnRpZmljYXRlQXJuOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259Om1lZGlhY29udmVydDoke1JlZ2lvbn06JHtBY2NvdW50fTpjZXJ0aWZpY2F0ZXMvJHtDZXJ0aWZpY2F0ZUFybn0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0NlcnRpZmljYXRlQXJufScsIGNlcnRpZmljYXRlQXJuKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIHByZXNlbmNlIG9mIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3RhZ3MuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSm9iVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZVByZXNldCgpXG4gICAqIC0gLnRvQ3JlYXRlUXVldWUoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGFnIGtleS12YWx1ZSBwYWlycyBhdHRhY2hlZCB0byB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3RhZ3MuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFF1ZXVlXG4gICAqIC0gUHJlc2V0XG4gICAqIC0gSm9iVGVtcGxhdGVcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKTogUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleXMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhY29udmVydC9sYXRlc3QvYXBpcmVmZXJlbmNlL3RhZ3MuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSm9iVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZVByZXNldCgpXG4gICAqIC0gLnRvQ3JlYXRlUXVldWUoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxufVxuIl19