"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Globalaccelerator = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglobalaccelerator.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Globalaccelerator extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglobalaccelerator.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'globalaccelerator';
        this.actionList = {
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html",
                "description": "Advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP).",
                "accessLevel": "Write"
            },
            "CreateAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html",
                "description": "Create an accelerator.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html",
                "description": "Add an endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "CreateListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html",
                "description": "Add a listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DeleteAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html",
                "description": "Delete the accelerator.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DeleteEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html",
                "description": "Delete the endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "DeleteListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html",
                "description": "Delete the listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html",
                "description": "Releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP) and deletes the corresponding address pool.",
                "accessLevel": "Write"
            },
            "DescribeAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html",
                "description": "Describe the accelerator.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DescribeAcceleratorAttributes": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html",
                "description": "Describe the accelerator Attributes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DescribeEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html",
                "description": "Describe the endpoint group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "DescribeListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html",
                "description": "Describe the listener.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "ListAccelerators": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html",
                "description": "List the accelerators.",
                "accessLevel": "List"
            },
            "ListByoipCidrs": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html",
                "description": "List the byoip cidrs.",
                "accessLevel": "List"
            },
            "ListEndpointGroups": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html",
                "description": "List the endpoint groups.",
                "accessLevel": "List",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "ListListeners": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html",
                "description": "List the listeners.",
                "accessLevel": "List",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html",
                "description": "List tags for a globalaccelerator resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                }
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html",
                "description": "Provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP) and creates a corresponding address pool.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html",
                "description": "Add tags to globalaccelerator resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html",
                "description": "Remove tags from globalaccelerator resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html",
                "description": "Update the accelerator.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "UpdateAcceleratorAttributes": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html",
                "description": "Update the accelerator attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html",
                "description": "Update the endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "UpdateListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html",
                "description": "Update the listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html",
                "description": "Stops advertising an IPv4 address range that is provisioned as an address pool.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "accelerator": {
                "name": "accelerator",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "listener": {
                "name": "listener",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "endpointgroup": {
                "name": "endpointgroup",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}/endpoint-group/${EndpointGroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html
     */
    toAdvertiseByoipCidr() {
        this.add('globalaccelerator:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Create an accelerator.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html
     */
    toCreateAccelerator() {
        this.add('globalaccelerator:CreateAccelerator');
        return this;
    }
    /**
     * Add an endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html
     */
    toCreateEndpointGroup() {
        this.add('globalaccelerator:CreateEndpointGroup');
        return this;
    }
    /**
     * Add a listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html
     */
    toCreateListener() {
        this.add('globalaccelerator:CreateListener');
        return this;
    }
    /**
     * Delete the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html
     */
    toDeleteAccelerator() {
        this.add('globalaccelerator:DeleteAccelerator');
        return this;
    }
    /**
     * Delete the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html
     */
    toDeleteEndpointGroup() {
        this.add('globalaccelerator:DeleteEndpointGroup');
        return this;
    }
    /**
     * Delete the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html
     */
    toDeleteListener() {
        this.add('globalaccelerator:DeleteListener');
        return this;
    }
    /**
     * Releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP) and deletes the corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html
     */
    toDeprovisionByoipCidr() {
        this.add('globalaccelerator:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Describe the accelerator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html
     */
    toDescribeAccelerator() {
        this.add('globalaccelerator:DescribeAccelerator');
        return this;
    }
    /**
     * Describe the accelerator Attributes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html
     */
    toDescribeAcceleratorAttributes() {
        this.add('globalaccelerator:DescribeAcceleratorAttributes');
        return this;
    }
    /**
     * Describe the endpoint group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html
     */
    toDescribeEndpointGroup() {
        this.add('globalaccelerator:DescribeEndpointGroup');
        return this;
    }
    /**
     * Describe the listener.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html
     */
    toDescribeListener() {
        this.add('globalaccelerator:DescribeListener');
        return this;
    }
    /**
     * List the accelerators.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html
     */
    toListAccelerators() {
        this.add('globalaccelerator:ListAccelerators');
        return this;
    }
    /**
     * List the byoip cidrs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html
     */
    toListByoipCidrs() {
        this.add('globalaccelerator:ListByoipCidrs');
        return this;
    }
    /**
     * List the endpoint groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html
     */
    toListEndpointGroups() {
        this.add('globalaccelerator:ListEndpointGroups');
        return this;
    }
    /**
     * List the listeners.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html
     */
    toListListeners() {
        this.add('globalaccelerator:ListListeners');
        return this;
    }
    /**
     * List tags for a globalaccelerator resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('globalaccelerator:ListTagsForResource');
        return this;
    }
    /**
     * Provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP) and creates a corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html
     */
    toProvisionByoipCidr() {
        this.add('globalaccelerator:ProvisionByoipCidr');
        return this;
    }
    /**
     * Add tags to globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.add('globalaccelerator:TagResource');
        return this;
    }
    /**
     * Remove tags from globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.add('globalaccelerator:UntagResource');
        return this;
    }
    /**
     * Update the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html
     */
    toUpdateAccelerator() {
        this.add('globalaccelerator:UpdateAccelerator');
        return this;
    }
    /**
     * Update the accelerator attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html
     */
    toUpdateAcceleratorAttributes() {
        this.add('globalaccelerator:UpdateAcceleratorAttributes');
        return this;
    }
    /**
     * Update the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html
     */
    toUpdateEndpointGroup() {
        this.add('globalaccelerator:UpdateEndpointGroup');
        return this;
    }
    /**
     * Update the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html
     */
    toUpdateListener() {
        this.add('globalaccelerator:UpdateListener');
        return this;
    }
    /**
     * Stops advertising an IPv4 address range that is provisioned as an address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html
     */
    toWithdrawByoipCidr() {
        this.add('globalaccelerator:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type accelerator to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccelerator(acceleratorId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onListener(acceleratorId, listenerId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpointgroup to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param endpointGroupId - Identifier for the endpointGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEndpointgroup(acceleratorId, listenerId, endpointGroupId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}/endpoint-group/${EndpointGroupId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${EndpointGroupId}', endpointGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccelerator()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - accelerator
     * - listener
     * - endpointgroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccelerator()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Globalaccelerator = Globalaccelerator;
//# sourceMappingURL=data:application/json;base64,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