"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Health = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Health extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'health';
        this.actionList = {
            "DescribeAffectedAccountsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html",
                "description": "Gets a list of accounts that have been affected by the specified events in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeAffectedEntities": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html",
                "description": "Gets a list of entities that have been affected by the specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeAffectedEntitiesForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html",
                "description": "Gets a list of entities that have been affected by the specified events and accounts in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeEntityAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html",
                "description": "Returns the number of entities that are affected by each of the specified events.",
                "accessLevel": "Read"
            },
            "DescribeEventAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html",
                "description": "Returns the number of events of each event type (issue, scheduled change, and account notification).",
                "accessLevel": "Read"
            },
            "DescribeEventDetails": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html",
                "description": "Returns detailed information about one or more specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeEventDetailsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html",
                "description": "Returns detailed information about one or more specified events for provided accounts in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeEventTypes": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html",
                "description": "Returns the event types that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html",
                "description": "Returns information about events that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEventsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html",
                "description": "Returns information about events that meet the specified filter criteria in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeHealthServiceStatusForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html",
                "description": "Returns the status of enabling or disabling the Organizational View feature",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DisableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html",
                "description": "Disables the Organizational View feature.",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "organizations:DisableAWSServiceAccess",
                    "organizations:ListAccounts"
                ]
            },
            "EnableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html",
                "description": "Enables the Organizational View feature.",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole",
                    "organizations:EnableAWSServiceAccess",
                    "organizations:ListAccounts"
                ]
            }
        };
        this.resourceTypes = {
            "event": {
                "name": "event",
                "url": "https://docs.aws.amazon.com/health/latest/ug/supported-operations.html",
                "arn": "arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Gets a list of accounts that have been affected by the specified events in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html
     */
    toDescribeAffectedAccountsForOrganization() {
        this.add('health:DescribeAffectedAccountsForOrganization');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html
     */
    toDescribeAffectedEntities() {
        this.add('health:DescribeAffectedEntities');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events and accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html
     */
    toDescribeAffectedEntitiesForOrganization() {
        this.add('health:DescribeAffectedEntitiesForOrganization');
        return this;
    }
    /**
     * Returns the number of entities that are affected by each of the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html
     */
    toDescribeEntityAggregates() {
        this.add('health:DescribeEntityAggregates');
        return this;
    }
    /**
     * Returns the number of events of each event type (issue, scheduled change, and account notification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html
     */
    toDescribeEventAggregates() {
        this.add('health:DescribeEventAggregates');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html
     */
    toDescribeEventDetails() {
        this.add('health:DescribeEventDetails');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events for provided accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html
     */
    toDescribeEventDetailsForOrganization() {
        this.add('health:DescribeEventDetailsForOrganization');
        return this;
    }
    /**
     * Returns the event types that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html
     */
    toDescribeEventTypes() {
        this.add('health:DescribeEventTypes');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.add('health:DescribeEvents');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html
     */
    toDescribeEventsForOrganization() {
        this.add('health:DescribeEventsForOrganization');
        return this;
    }
    /**
     * Returns the status of enabling or disabling the Organizational View feature
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html
     */
    toDescribeHealthServiceStatusForOrganization() {
        this.add('health:DescribeHealthServiceStatusForOrganization');
        return this;
    }
    /**
     * Disables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:DisableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html
     */
    toDisableHealthServiceAccessForOrganization() {
        this.add('health:DisableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Enables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html
     */
    toEnableHealthServiceAccessForOrganization() {
        this.add('health:EnableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Adds a resource of type event to the statement
     *
     * https://docs.aws.amazon.com/health/latest/ug/supported-operations.html
     *
     * @param service - Identifier for the service.
     * @param eventTypeCode - Identifier for the eventTypeCode.
     * @param eventTypePlusId - Identifier for the eventTypePlusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvent(service, eventTypeCode, eventTypePlusId, account, region, partition) {
        var arn = 'arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}';
        arn = arn.replace('${Service}', service);
        arn = arn.replace('${EventTypeCode}', eventTypeCode);
        arn = arn.replace('${EventTypePlusId}', eventTypePlusId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The type of event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .toDescribeAffectedEntities()
     * - .toDescribeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventTypeCode(value, operator) {
        return this.if(`health:eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * The service of the event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .toDescribeAffectedEntities()
     * - .toDescribeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`health:service`, value, operator || 'StringLike');
    }
}
exports.Health = Health;
//# sourceMappingURL=data:application/json;base64,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