"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kms';
        this.actionList = {
            "CancelKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html",
                "description": "Controls permission to cancel the scheduled deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ConnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html",
                "description": "Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html",
                "description": "Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "CreateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html",
                "description": "Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.",
                "accessLevel": "Write",
                "dependentActions": [
                    "cloudhsm:DescribeClusters"
                ]
            },
            "CreateGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html",
                "description": "Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantConstraintType",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "CreateKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html",
                "description": "Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.",
                "accessLevel": "Write",
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CustomerMasterKeySpec",
                    "kms:CustomerMasterKeyUsage",
                    "kms:KeyOrigin"
                ]
            },
            "Decrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html",
                "description": "Controls permission to decrypt ciphertext that was encrypted under a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html",
                "description": "Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DeleteCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html",
                "description": "Controls permission to delete a custom key store.",
                "accessLevel": "Write"
            },
            "DeleteImportedKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html",
                "description": "Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DescribeCustomKeyStores": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html",
                "description": "Controls permission to view detailed information about custom key stores in the account and region.",
                "accessLevel": "Read"
            },
            "DescribeKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html",
                "description": "Controls permission to view detailed information about a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html",
                "description": "Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html",
                "description": "Controls permission to disable automatic rotation of a customer managed customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisconnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html",
                "description": "Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "EnableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html",
                "description": "Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "EnableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html",
                "description": "Controls permission to enable automatic rotation of the cryptographic material in a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Encrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html",
                "description": "Controls permission to use the specified customer master key to encrypt data and data keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html",
                "description": "Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPair": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html",
                "description": "Controls permission to use the customer master key to generate data key pairs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPairWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateRandom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html",
                "description": "Controls permission to get a cryptographically secure random byte string from AWS KMS.",
                "accessLevel": "Write"
            },
            "GetKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html",
                "description": "Controls permission to view the key policy for the specified customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetKeyRotationStatus": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html",
                "description": "Controls permission to determine whether automatic key rotation is enabled on the customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetParametersForImport": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html",
                "description": "Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService",
                    "kms:WrappingAlgorithm",
                    "kms:WrappingKeySpec"
                ]
            },
            "GetPublicKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html",
                "description": "Controls permission to download the public key of an asymmetric customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ImportKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html",
                "description": "Controls permission to import cryptographic material into a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ExpirationModel",
                    "kms:ValidTo",
                    "kms:ViaService"
                ]
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html",
                "description": "Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "List"
            },
            "ListGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html",
                "description": "Controls permission to view all grants for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ListKeyPolicies": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html",
                "description": "Controls permission to view the names of key policies for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListKeys": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html",
                "description": "Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.",
                "accessLevel": "List"
            },
            "ListResourceTags": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html",
                "description": "Controls permission to view all tags that are attached to a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListRetirableGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html",
                "description": "Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "PutKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html",
                "description": "Controls permission to replace the key policy for the specified customer master key.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ReEncryptFrom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "ReEncryptTo": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "RetireGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html",
                "description": "Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "RevokeGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html",
                "description": "Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ScheduleKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html",
                "description": "Controls permission to schedule deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Sign": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html",
                "description": "Controls permission to produce a digital signature for a message.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html",
                "description": "Controls permission to create or update tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html",
                "description": "Controls permission to delete tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html",
                "description": "Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html",
                "description": "Controls permission to change the properties of a custom key store.",
                "accessLevel": "Write"
            },
            "UpdateKeyDescription": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html",
                "description": "Controls permission to delete or change the description of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Verify": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html",
                "description": "Controls permission to use the specified customer master key to verify digital signatures.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            }
        };
        this.resourceTypes = {
            "alias": {
                "name": "alias",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}",
                "conditionKeys": []
            },
            "key": {
                "name": "key",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Controls permission to cancel the scheduled deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html
     */
    toCancelKeyDeletion() {
        this.add('kms:CancelKeyDeletion');
        return this;
    }
    /**
     * Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html
     */
    toConnectCustomKeyStore() {
        this.add('kms:ConnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        this.add('kms:CreateAlias');
        return this;
    }
    /**
     * Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudhsm:DescribeClusters
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html
     */
    toCreateCustomKeyStore() {
        this.add('kms:CreateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantConstraintType()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html
     */
    toCreateGrant() {
        this.add('kms:CreateGrant');
        return this;
    }
    /**
     * Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCustomerMasterKeySpec()
     * - .ifCustomerMasterKeyUsage()
     * - .ifKeyOrigin()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html
     */
    toCreateKey() {
        this.add('kms:CreateKey');
        return this;
    }
    /**
     * Controls permission to decrypt ciphertext that was encrypted under a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html
     */
    toDecrypt() {
        this.add('kms:Decrypt');
        return this;
    }
    /**
     * Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.add('kms:DeleteAlias');
        return this;
    }
    /**
     * Controls permission to delete a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html
     */
    toDeleteCustomKeyStore() {
        this.add('kms:DeleteCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html
     */
    toDeleteImportedKeyMaterial() {
        this.add('kms:DeleteImportedKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view detailed information about custom key stores in the account and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html
     */
    toDescribeCustomKeyStores() {
        this.add('kms:DescribeCustomKeyStores');
        return this;
    }
    /**
     * Controls permission to view detailed information about a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html
     */
    toDescribeKey() {
        this.add('kms:DescribeKey');
        return this;
    }
    /**
     * Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html
     */
    toDisableKey() {
        this.add('kms:DisableKey');
        return this;
    }
    /**
     * Controls permission to disable automatic rotation of a customer managed customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html
     */
    toDisableKeyRotation() {
        this.add('kms:DisableKeyRotation');
        return this;
    }
    /**
     * Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html
     */
    toDisconnectCustomKeyStore() {
        this.add('kms:DisconnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html
     */
    toEnableKey() {
        this.add('kms:EnableKey');
        return this;
    }
    /**
     * Controls permission to enable automatic rotation of the cryptographic material in a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html
     */
    toEnableKeyRotation() {
        this.add('kms:EnableKeyRotation');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to encrypt data and data keys.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html
     */
    toEncrypt() {
        this.add('kms:Encrypt');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html
     */
    toGenerateDataKey() {
        this.add('kms:GenerateDataKey');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html
     */
    toGenerateDataKeyPair() {
        this.add('kms:GenerateDataKeyPair');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html
     */
    toGenerateDataKeyPairWithoutPlaintext() {
        this.add('kms:GenerateDataKeyPairWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html
     */
    toGenerateDataKeyWithoutPlaintext() {
        this.add('kms:GenerateDataKeyWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to get a cryptographically secure random byte string from AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html
     */
    toGenerateRandom() {
        this.add('kms:GenerateRandom');
        return this;
    }
    /**
     * Controls permission to view the key policy for the specified customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html
     */
    toGetKeyPolicy() {
        this.add('kms:GetKeyPolicy');
        return this;
    }
    /**
     * Controls permission to determine whether automatic key rotation is enabled on the customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html
     */
    toGetKeyRotationStatus() {
        this.add('kms:GetKeyRotationStatus');
        return this;
    }
    /**
     * Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     * - .ifWrappingAlgorithm()
     * - .ifWrappingKeySpec()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html
     */
    toGetParametersForImport() {
        this.add('kms:GetParametersForImport');
        return this;
    }
    /**
     * Controls permission to download the public key of an asymmetric customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        this.add('kms:GetPublicKey');
        return this;
    }
    /**
     * Controls permission to import cryptographic material into a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifExpirationModel()
     * - .ifValidTo()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html
     */
    toImportKeyMaterial() {
        this.add('kms:ImportKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        this.add('kms:ListAliases');
        return this;
    }
    /**
     * Controls permission to view all grants for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html
     */
    toListGrants() {
        this.add('kms:ListGrants');
        return this;
    }
    /**
     * Controls permission to view the names of key policies for a customer master key.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html
     */
    toListKeyPolicies() {
        this.add('kms:ListKeyPolicies');
        return this;
    }
    /**
     * Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html
     */
    toListKeys() {
        this.add('kms:ListKeys');
        return this;
    }
    /**
     * Controls permission to view all tags that are attached to a customer master key.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html
     */
    toListResourceTags() {
        this.add('kms:ListResourceTags');
        return this;
    }
    /**
     * Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html
     */
    toListRetirableGrants() {
        this.add('kms:ListRetirableGrants');
        return this;
    }
    /**
     * Controls permission to replace the key policy for the specified customer master key.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html
     */
    toPutKeyPolicy() {
        this.add('kms:PutKeyPolicy');
        return this;
    }
    /**
     * Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptFrom() {
        this.add('kms:ReEncryptFrom');
        return this;
    }
    /**
     * Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptTo() {
        this.add('kms:ReEncryptTo');
        return this;
    }
    /**
     * Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html
     */
    toRetireGrant() {
        this.add('kms:RetireGrant');
        return this;
    }
    /**
     * Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html
     */
    toRevokeGrant() {
        this.add('kms:RevokeGrant');
        return this;
    }
    /**
     * Controls permission to schedule deletion of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html
     */
    toScheduleKeyDeletion() {
        this.add('kms:ScheduleKeyDeletion');
        return this;
    }
    /**
     * Controls permission to produce a digital signature for a message.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html
     */
    toSign() {
        this.add('kms:Sign');
        return this;
    }
    /**
     * Controls permission to create or update tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('kms:TagResource');
        return this;
    }
    /**
     * Controls permission to delete tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('kms:UntagResource');
        return this;
    }
    /**
     * Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        this.add('kms:UpdateAlias');
        return this;
    }
    /**
     * Controls permission to change the properties of a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html
     */
    toUpdateCustomKeyStore() {
        this.add('kms:UpdateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete or change the description of a customer master key.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html
     */
    toUpdateKeyDescription() {
        this.add('kms:UpdateKeyDescription');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to verify digital signatures.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html
     */
    toVerify() {
        this.add('kms:Verify');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlias(alias, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}';
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Controls access to the CreateKey and PutKeyPolicy operations based on the value of the BypassPolicyLockoutSafetyCheck parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-bypass-policy-lockout-safety-check
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toPutKeyPolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBypassPolicyLockoutSafetyCheck(value) {
        return this.if(`kms:BypassPolicyLockoutSafetyCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to specified AWS KMS operations based on the AWS account ID of the caller. You can use this condition key to allow or deny access to all IAM users and roles in an AWS account in a single policy statement.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-caller-account
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toCreateAlias()
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteImportedKeyMaterial()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRevokeGrant()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateKeyDescription()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCallerAccount(value, operator) {
        return this.if(`kms:CallerAccount`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the CustomerMasterKeySpec property of the CMK that is created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-spec
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeySpec(value, operator) {
        return this.if(`kms:CustomerMasterKeySpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the KeyUsage property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-usage
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeyUsage(value, operator) {
        return this.if(`kms:CustomerMasterKeyUsage`, value, operator || 'StringLike');
    }
    /**
     * Controls access to GenerateDataKeyPair and GenerateDataKeyPairWithoutPlaintext operations based on the value of the DataKeyPairSpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-data-key-pair-spec
     *
     * Applies to actions:
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataKeyPairSpec(value, operator) {
        return this.if(`kms:DataKeyPairSpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to encryption operations based on the value of the encryption algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionAlgorithm(value, operator) {
        return this.if(`kms:EncryptionAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access based on the presence of specified keys in the encryption context. The encryption context is an optional element in a cryptographic operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-context-keys
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContextKeys(value, operator) {
        return this.if(`kms:EncryptionContextKeys`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ExpirationModel parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-expiration-model
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExpirationModel(value, operator) {
        return this.if(`kms:ExpirationModel`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grant constraint in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-constraint-type
     *
     * Applies to actions:
     * - .toCreateGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantConstraintType(value, operator) {
        return this.if(`kms:GrantConstraintType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation when the request comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-is-for-aws-resource
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toListGrants()
     * - .toRevokeGrant()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifGrantIsForAWSResource(value) {
        return this.if(`kms:GrantIsForAWSResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the operations in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantOperations(value, operator) {
        return this.if(`kms:GrantOperations`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grantee principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grantee-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGranteePrincipal(value, operator) {
        return this.if(`kms:GranteePrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the Origin property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-key-origin
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyOrigin(value, operator) {
        return this.if(`kms:KeyOrigin`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the value of the MessageType parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-message-type
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMessageType(value, operator) {
        return this.if(`kms:MessageType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ReEncrypt operation when it uses the same customer master key that was used for the Encrypt operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-reencrypt-on-same-key
     *
     * Applies to actions:
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifReEncryptOnSameKey(value) {
        return this.if(`kms:ReEncryptOnSameKey`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the retiring principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-retiring-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRetiringPrincipal(value, operator) {
        return this.if(`kms:RetiringPrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the signing algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSigningAlgorithm(value, operator) {
        return this.if(`kms:SigningAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ValidTo parameter in the request. You can use this condition key to allow users to import key material only when it expires by the specified date.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-valid-to
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifValidTo(value, operator) {
        return this.if(`kms:ValidTo`, value, operator || 'NumericEquals');
    }
    /**
     * Controls access when a request made on the principal's behalf comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-via-service
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toCreateAlias()
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteImportedKeyMaterial()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRevokeGrant()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateKeyDescription()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaService(value, operator) {
        return this.if(`kms:ViaService`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingAlgorithm parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-algorithm
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingAlgorithm(value, operator) {
        return this.if(`kms:WrappingAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingKeySpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-key-spec
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeySpec(value, operator) {
        return this.if(`kms:WrappingKeySpec`, value, operator || 'StringLike');
    }
}
exports.Kms = Kms;
//# sourceMappingURL=data:application/json;base64,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