"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mgh = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mgh](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmigrationhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mgh extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mgh](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmigrationhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mgh';
        this.actionList = {
            "AssociateCreatedArtifact": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html",
                "description": "Associate a given AWS artifact to a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "AssociateDiscoveredResource": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html",
                "description": "Associate a given ADS resource to a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "CreateHomeRegionControl": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateHomeRegionControl.html",
                "description": "Create a Migration Hub Home Region Control",
                "accessLevel": "Write"
            },
            "CreateProgressUpdateStream": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html",
                "description": "Create a ProgressUpdateStream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "progressUpdateStream": {
                        "required": true
                    }
                }
            },
            "DeleteProgressUpdateStream": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html",
                "description": "Delete a ProgressUpdateStream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "progressUpdateStream": {
                        "required": true
                    }
                }
            },
            "DescribeApplicationState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html",
                "description": "Get an Application Discovery Service Application's state",
                "accessLevel": "Read"
            },
            "DescribeHomeRegionControls": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeHomeRegionControls.html",
                "description": "List Home Region Controls",
                "accessLevel": "List"
            },
            "DescribeMigrationTask": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html",
                "description": "Describe a MigrationTask",
                "accessLevel": "Read",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "DisassociateCreatedArtifact": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html",
                "description": "Disassociate a given AWS artifact from a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "DisassociateDiscoveredResource": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html",
                "description": "Disassociate a given ADS resource from a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "GetHomeRegion": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_GetHomeRegion.html",
                "description": "Get the Migration Hub Home Region",
                "accessLevel": "Read"
            },
            "ImportMigrationTask": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html",
                "description": "Import a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListCreatedArtifacts": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html",
                "description": "List associated created artifacts for a MigrationTask",
                "accessLevel": "List",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html",
                "description": "List associated ADS resources from MigrationTask",
                "accessLevel": "List",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListMigrationTasks": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html",
                "description": "List MigrationTasks",
                "accessLevel": "List"
            },
            "ListProgressUpdateStreams": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html",
                "description": "List ProgressUpdateStreams",
                "accessLevel": "List"
            },
            "NotifyApplicationState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html",
                "description": "Update an Application Discovery Service Application's state",
                "accessLevel": "Write"
            },
            "NotifyMigrationTaskState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html",
                "description": "Notify latest MigrationTask state",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "PutResourceAttributes": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html",
                "description": "Put ResourceAttributes",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "progressUpdateStream": {
                "name": "progressUpdateStream",
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html",
                "arn": "arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}",
                "conditionKeys": []
            },
            "migrationTask": {
                "name": "migrationTask",
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html",
                "arn": "arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}/migrationTask/${Task}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associate a given AWS artifact to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html
     */
    toAssociateCreatedArtifact() {
        this.add('mgh:AssociateCreatedArtifact');
        return this;
    }
    /**
     * Associate a given ADS resource to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html
     */
    toAssociateDiscoveredResource() {
        this.add('mgh:AssociateDiscoveredResource');
        return this;
    }
    /**
     * Create a Migration Hub Home Region Control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateHomeRegionControl.html
     */
    toCreateHomeRegionControl() {
        this.add('mgh:CreateHomeRegionControl');
        return this;
    }
    /**
     * Create a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html
     */
    toCreateProgressUpdateStream() {
        this.add('mgh:CreateProgressUpdateStream');
        return this;
    }
    /**
     * Delete a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html
     */
    toDeleteProgressUpdateStream() {
        this.add('mgh:DeleteProgressUpdateStream');
        return this;
    }
    /**
     * Get an Application Discovery Service Application's state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html
     */
    toDescribeApplicationState() {
        this.add('mgh:DescribeApplicationState');
        return this;
    }
    /**
     * List Home Region Controls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeHomeRegionControls.html
     */
    toDescribeHomeRegionControls() {
        this.add('mgh:DescribeHomeRegionControls');
        return this;
    }
    /**
     * Describe a MigrationTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html
     */
    toDescribeMigrationTask() {
        this.add('mgh:DescribeMigrationTask');
        return this;
    }
    /**
     * Disassociate a given AWS artifact from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html
     */
    toDisassociateCreatedArtifact() {
        this.add('mgh:DisassociateCreatedArtifact');
        return this;
    }
    /**
     * Disassociate a given ADS resource from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html
     */
    toDisassociateDiscoveredResource() {
        this.add('mgh:DisassociateDiscoveredResource');
        return this;
    }
    /**
     * Get the Migration Hub Home Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_GetHomeRegion.html
     */
    toGetHomeRegion() {
        this.add('mgh:GetHomeRegion');
        return this;
    }
    /**
     * Import a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html
     */
    toImportMigrationTask() {
        this.add('mgh:ImportMigrationTask');
        return this;
    }
    /**
     * List associated created artifacts for a MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html
     */
    toListCreatedArtifacts() {
        this.add('mgh:ListCreatedArtifacts');
        return this;
    }
    /**
     * List associated ADS resources from MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        this.add('mgh:ListDiscoveredResources');
        return this;
    }
    /**
     * List MigrationTasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html
     */
    toListMigrationTasks() {
        this.add('mgh:ListMigrationTasks');
        return this;
    }
    /**
     * List ProgressUpdateStreams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html
     */
    toListProgressUpdateStreams() {
        this.add('mgh:ListProgressUpdateStreams');
        return this;
    }
    /**
     * Update an Application Discovery Service Application's state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html
     */
    toNotifyApplicationState() {
        this.add('mgh:NotifyApplicationState');
        return this;
    }
    /**
     * Notify latest MigrationTask state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html
     */
    toNotifyMigrationTaskState() {
        this.add('mgh:NotifyMigrationTaskState');
        return this;
    }
    /**
     * Put ResourceAttributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html
     */
    toPutResourceAttributes() {
        this.add('mgh:PutResourceAttributes');
        return this;
    }
    /**
     * Adds a resource of type progressUpdateStream to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html
     *
     * @param stream - Identifier for the stream.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProgressUpdateStream(stream, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type migrationTask to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html
     *
     * @param stream - Identifier for the stream.
     * @param task - Identifier for the task.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMigrationTask(stream, task, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}/migrationTask/${Task}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Task}', task);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mgh = Mgh;
//# sourceMappingURL=data:application/json;base64,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