"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobilehub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mobilehub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobilehub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mobilehub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mobilehub';
        this.actionList = {
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a project",
                "accessLevel": "Write"
            },
            "CreateServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Enable AWS Mobile Hub in the account by creating the required service role",
                "accessLevel": "Write"
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete the specified project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete a saved snapshot of project configuration",
                "accessLevel": "Write"
            },
            "DeployToStage": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Deploy changes to the specified stage",
                "accessLevel": "Write"
            },
            "DescribeBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Describe the download bundle",
                "accessLevel": "Read"
            },
            "ExportBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the download bundle",
                "accessLevel": "Read"
            },
            "ExportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the project configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GenerateProjectParameters": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Generate project parameters required for code generation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Get project configuration and resources",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Fetch the previously exported project configuration snapshot",
                "accessLevel": "Read"
            },
            "ImportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a new project from the previously exported project configuration",
                "accessLevel": "Write"
            },
            "InstallBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Install a bundle in the project deployments S3 bucket",
                "accessLevel": "Write"
            },
            "ListAvailableConnectors": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available SaaS (Software as a Service) connectors",
                "accessLevel": "List"
            },
            "ListAvailableFeatures": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available features",
                "accessLevel": "List"
            },
            "ListAvailableRegions": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available regions for projects",
                "accessLevel": "List"
            },
            "ListBundles": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available download bundles",
                "accessLevel": "List"
            },
            "ListProjectSnapshots": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List saved snapshots of project configuration",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List projects",
                "accessLevel": "List"
            },
            "SynchronizeProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Synchronize state of resources into project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Update project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ValidateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Validate a mobile hub project.",
                "accessLevel": "Read"
            },
            "VerifyServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Verify AWS Mobile Hub is enabled in the account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html",
                "arn": "arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toCreateProject() {
        this.add('mobilehub:CreateProject');
        return this;
    }
    /**
     * Enable AWS Mobile Hub in the account by creating the required service role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toCreateServiceRole() {
        this.add('mobilehub:CreateServiceRole');
        return this;
    }
    /**
     * Delete the specified project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeleteProject() {
        this.add('mobilehub:DeleteProject');
        return this;
    }
    /**
     * Delete a saved snapshot of project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeleteProjectSnapshot() {
        this.add('mobilehub:DeleteProjectSnapshot');
        return this;
    }
    /**
     * Deploy changes to the specified stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDeployToStage() {
        this.add('mobilehub:DeployToStage');
        return this;
    }
    /**
     * Describe the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toDescribeBundle() {
        this.add('mobilehub:DescribeBundle');
        return this;
    }
    /**
     * Export the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toExportBundle() {
        this.add('mobilehub:ExportBundle');
        return this;
    }
    /**
     * Export the project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toExportProject() {
        this.add('mobilehub:ExportProject');
        return this;
    }
    /**
     * Generate project parameters required for code generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGenerateProjectParameters() {
        this.add('mobilehub:GenerateProjectParameters');
        return this;
    }
    /**
     * Get project configuration and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGetProject() {
        this.add('mobilehub:GetProject');
        return this;
    }
    /**
     * Fetch the previously exported project configuration snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toGetProjectSnapshot() {
        this.add('mobilehub:GetProjectSnapshot');
        return this;
    }
    /**
     * Create a new project from the previously exported project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toImportProject() {
        this.add('mobilehub:ImportProject');
        return this;
    }
    /**
     * Install a bundle in the project deployments S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toInstallBundle() {
        this.add('mobilehub:InstallBundle');
        return this;
    }
    /**
     * List the available SaaS (Software as a Service) connectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableConnectors() {
        this.add('mobilehub:ListAvailableConnectors');
        return this;
    }
    /**
     * List available features
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableFeatures() {
        this.add('mobilehub:ListAvailableFeatures');
        return this;
    }
    /**
     * List available regions for projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListAvailableRegions() {
        this.add('mobilehub:ListAvailableRegions');
        return this;
    }
    /**
     * List the available download bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListBundles() {
        this.add('mobilehub:ListBundles');
        return this;
    }
    /**
     * List saved snapshots of project configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListProjectSnapshots() {
        this.add('mobilehub:ListProjectSnapshots');
        return this;
    }
    /**
     * List projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toListProjects() {
        this.add('mobilehub:ListProjects');
        return this;
    }
    /**
     * Synchronize state of resources into project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toSynchronizeProject() {
        this.add('mobilehub:SynchronizeProject');
        return this;
    }
    /**
     * Update project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toUpdateProject() {
        this.add('mobilehub:UpdateProject');
        return this;
    }
    /**
     * Validate a mobile hub project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toValidateProject() {
        this.add('mobilehub:ValidateProject');
        return this;
    }
    /**
     * Verify AWS Mobile Hub is enabled in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    toVerifyServiceRole() {
        this.add('mobilehub:VerifyServiceRole');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobilehub = Mobilehub;
//# sourceMappingURL=data:application/json;base64,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