"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Opsworks = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [opsworks](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsopsworks.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Opsworks extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [opsworks](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsopsworks.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'opsworks';
        this.actionList = {
            "AssignInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignInstance.html",
                "description": "Assign a registered instance to a layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AssignVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignVolume.html",
                "description": "Assigns one of the stack's registered Amazon EBS volumes to a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AssociateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssociateElasticIp.html",
                "description": "Associates one of the stack's registered Elastic IP addresses with a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AttachElasticLoadBalancer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AttachElasticLoadBalancer.html",
                "description": "Attaches an Elastic Load Balancing load balancer to a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CloneStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CloneStack.html",
                "description": "Creates a clone of a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateApp.html",
                "description": "Creates an app for a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateDeployment.html",
                "description": "Runs deployment or stack commands",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateInstance.html",
                "description": "Creates an instance in a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateLayer.html",
                "description": "Creates a layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateStack.html",
                "description": "Creates a new stack",
                "accessLevel": "Write"
            },
            "CreateUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateUserProfile.html",
                "description": "Creates a new user profile",
                "accessLevel": "Write"
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteApp.html",
                "description": "Deletes a specified app",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteInstance.html",
                "description": "Deletes a specified instance, which terminates the associated Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteLayer.html",
                "description": "Deletes a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteStack.html",
                "description": "Deletes a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteUserProfile.html",
                "description": "Deletes a user profile",
                "accessLevel": "Write"
            },
            "DeregisterEcsCluster": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterEcsCluster.html",
                "description": "Deletes a user profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterElasticIp.html",
                "description": "Deregisters a specified Elastic IP address",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterInstance.html",
                "description": "Deregister a registered Amazon EC2 or on-premises instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterRdsDbInstance.html",
                "description": "Deregisters an Amazon RDS instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterVolume.html",
                "description": "Deregisters an Amazon EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeAgentVersions": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeAgentVersions.html",
                "description": "Describes the available AWS OpsWorks agent versions",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeApps": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeApps.html",
                "description": "Requests a description of a specified set of apps",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeCommands": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeCommands.html",
                "description": "Describes the results of specified commands",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeDeployments": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeDeployments.html",
                "description": "Requests a description of a specified set of deployments",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeEcsClusters": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeEcsClusters.html",
                "description": "Describes Amazon ECS clusters that are registered with a stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeElasticIps": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticIps.html",
                "description": "Describes Elastic IP addresses",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeElasticLoadBalancers": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticLoadBalancers.html",
                "description": "Describes a stack's Elastic Load Balancing instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeInstances.html",
                "description": "Requests a description of a set of instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeLayers": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLayers.html",
                "description": "Requests a description of one or more layers in a specified stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeLoadBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLoadBasedAutoScaling.html",
                "description": "Describes load-based auto scaling configurations for specified layers",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeMyUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeMyUserProfile.html",
                "description": "Describes a user's SSH information",
                "accessLevel": "List"
            },
            "DescribePermissions": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribePermissions.html",
                "description": "Describes the permissions for a specified stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeRaidArrays": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRaidArrays.html",
                "description": "Describe an instance's RAID arrays",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeRdsDbInstances": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRdsDbInstances.html",
                "description": "Describes Amazon RDS instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeServiceErrors": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeServiceErrors.html",
                "description": "Describes AWS OpsWorks service errors",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStackProvisioningParameters": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackProvisioningParameters.html",
                "description": "Requests a description of a stack's provisioning parameters",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStackSummary": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackSummary.html",
                "description": "Describes the number of layers and apps in a specified stack, and the number of instances in each state, such as running_setup or online",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStacks": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStacks.html",
                "description": "Requests a description of one or more stacks",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeTimeBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeTimeBasedAutoScaling.html",
                "description": "Describes time-based auto scaling configurations for specified instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeUserProfiles": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeUserProfiles.html",
                "description": "Describe specified users",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeVolumes.html",
                "description": "Describes an instance's Amazon EBS volumes",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DetachElasticLoadBalancer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DetachElasticLoadBalancer.html",
                "description": "Detaches a specified Elastic Load Balancing instance from its layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DisassociateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DisassociateElasticIp.html",
                "description": "Disassociates an Elastic IP address from its instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "GetHostnameSuggestion": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_GetHostnameSuggestion.html",
                "description": "Gets a generated host name for the specified layer, based on the current host name theme",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "GrantAccess": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html",
                "description": "Grants RDP access to a Windows instance for a specified time period",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_ListTags.html",
                "description": "Returns a list of tags that are applied to the specified stack or layer",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html",
                "description": "Reboots a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterEcsCluster": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterEcsCluster.html",
                "description": "Registers a specified Amazon ECS cluster with a stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterElasticIp.html",
                "description": "Registers an Elastic IP address with a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterInstance.html",
                "description": "Registers instances with a specified stack that were created outside of AWS OpsWorks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterRdsDbInstance.html",
                "description": "Registers an Amazon RDS instance with a stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterVolume.html",
                "description": "Registers an Amazon EBS volume with a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetLoadBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetLoadBasedAutoScaling.html",
                "description": "Specify the load-based auto scaling configuration for a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetPermission": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetPermission.html",
                "description": "Specifies a user's permissions",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetTimeBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetTimeBasedAutoScaling.html",
                "description": "Specify the time-based auto scaling configuration for a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartInstance.html",
                "description": "Starts a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StartStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartStack.html",
                "description": "Starts a stack's instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopInstance.html",
                "description": "Stops a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StopStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopStack.html",
                "description": "Stops a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_TagResource.html",
                "description": "Apply tags to a specified stack or layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UnassignInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignInstance.html",
                "description": "Unassigns a registered instance from all of it's layers",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UnassignVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignVolume.html",
                "description": "Unassigns an assigned Amazon EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a specified stack or layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateApp.html",
                "description": "Updates a specified app",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateElasticIp.html",
                "description": "Updates a registered Elastic IP address's name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateInstance.html",
                "description": "Updates a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateLayer.html",
                "description": "Updates a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateMyUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateMyUserProfile.html",
                "description": "Updates a user's SSH public key",
                "accessLevel": "Write"
            },
            "UpdateRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateRdsDbInstance.html",
                "description": "Updates an Amazon RDS instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateStack.html",
                "description": "Updates a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateUserProfile.html",
                "description": "Updates a specified user profile",
                "accessLevel": "Permissions management"
            },
            "UpdateVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateVolume.html",
                "description": "Updates an Amazon EBS volume's name or mount point",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "stack": {
                "name": "stack",
                "url": "https://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks.html",
                "arn": "arn:${Partition}:opsworks:${Region}:${Account}:stack/${StackId}/",
                "conditionKeys": []
            }
        };
    }
    /**
     * Assign a registered instance to a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignInstance.html
     */
    toAssignInstance() {
        this.add('opsworks:AssignInstance');
        return this;
    }
    /**
     * Assigns one of the stack's registered Amazon EBS volumes to a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignVolume.html
     */
    toAssignVolume() {
        this.add('opsworks:AssignVolume');
        return this;
    }
    /**
     * Associates one of the stack's registered Elastic IP addresses with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssociateElasticIp.html
     */
    toAssociateElasticIp() {
        this.add('opsworks:AssociateElasticIp');
        return this;
    }
    /**
     * Attaches an Elastic Load Balancing load balancer to a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AttachElasticLoadBalancer.html
     */
    toAttachElasticLoadBalancer() {
        this.add('opsworks:AttachElasticLoadBalancer');
        return this;
    }
    /**
     * Creates a clone of a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CloneStack.html
     */
    toCloneStack() {
        this.add('opsworks:CloneStack');
        return this;
    }
    /**
     * Creates an app for a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateApp.html
     */
    toCreateApp() {
        this.add('opsworks:CreateApp');
        return this;
    }
    /**
     * Runs deployment or stack commands
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        this.add('opsworks:CreateDeployment');
        return this;
    }
    /**
     * Creates an instance in a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateInstance.html
     */
    toCreateInstance() {
        this.add('opsworks:CreateInstance');
        return this;
    }
    /**
     * Creates a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateLayer.html
     */
    toCreateLayer() {
        this.add('opsworks:CreateLayer');
        return this;
    }
    /**
     * Creates a new stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        this.add('opsworks:CreateStack');
        return this;
    }
    /**
     * Creates a new user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateUserProfile.html
     */
    toCreateUserProfile() {
        this.add('opsworks:CreateUserProfile');
        return this;
    }
    /**
     * Deletes a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteApp.html
     */
    toDeleteApp() {
        this.add('opsworks:DeleteApp');
        return this;
    }
    /**
     * Deletes a specified instance, which terminates the associated Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        this.add('opsworks:DeleteInstance');
        return this;
    }
    /**
     * Deletes a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteLayer.html
     */
    toDeleteLayer() {
        this.add('opsworks:DeleteLayer');
        return this;
    }
    /**
     * Deletes a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        this.add('opsworks:DeleteStack');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteUserProfile.html
     */
    toDeleteUserProfile() {
        this.add('opsworks:DeleteUserProfile');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterEcsCluster.html
     */
    toDeregisterEcsCluster() {
        this.add('opsworks:DeregisterEcsCluster');
        return this;
    }
    /**
     * Deregisters a specified Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterElasticIp.html
     */
    toDeregisterElasticIp() {
        this.add('opsworks:DeregisterElasticIp');
        return this;
    }
    /**
     * Deregister a registered Amazon EC2 or on-premises instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterInstance.html
     */
    toDeregisterInstance() {
        this.add('opsworks:DeregisterInstance');
        return this;
    }
    /**
     * Deregisters an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterRdsDbInstance.html
     */
    toDeregisterRdsDbInstance() {
        this.add('opsworks:DeregisterRdsDbInstance');
        return this;
    }
    /**
     * Deregisters an Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterVolume.html
     */
    toDeregisterVolume() {
        this.add('opsworks:DeregisterVolume');
        return this;
    }
    /**
     * Describes the available AWS OpsWorks agent versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeAgentVersions.html
     */
    toDescribeAgentVersions() {
        this.add('opsworks:DescribeAgentVersions');
        return this;
    }
    /**
     * Requests a description of a specified set of apps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeApps.html
     */
    toDescribeApps() {
        this.add('opsworks:DescribeApps');
        return this;
    }
    /**
     * Describes the results of specified commands
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeCommands.html
     */
    toDescribeCommands() {
        this.add('opsworks:DescribeCommands');
        return this;
    }
    /**
     * Requests a description of a specified set of deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeDeployments.html
     */
    toDescribeDeployments() {
        this.add('opsworks:DescribeDeployments');
        return this;
    }
    /**
     * Describes Amazon ECS clusters that are registered with a stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeEcsClusters.html
     */
    toDescribeEcsClusters() {
        this.add('opsworks:DescribeEcsClusters');
        return this;
    }
    /**
     * Describes Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticIps.html
     */
    toDescribeElasticIps() {
        this.add('opsworks:DescribeElasticIps');
        return this;
    }
    /**
     * Describes a stack's Elastic Load Balancing instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticLoadBalancers.html
     */
    toDescribeElasticLoadBalancers() {
        this.add('opsworks:DescribeElasticLoadBalancers');
        return this;
    }
    /**
     * Requests a description of a set of instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        this.add('opsworks:DescribeInstances');
        return this;
    }
    /**
     * Requests a description of one or more layers in a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLayers.html
     */
    toDescribeLayers() {
        this.add('opsworks:DescribeLayers');
        return this;
    }
    /**
     * Describes load-based auto scaling configurations for specified layers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLoadBasedAutoScaling.html
     */
    toDescribeLoadBasedAutoScaling() {
        this.add('opsworks:DescribeLoadBasedAutoScaling');
        return this;
    }
    /**
     * Describes a user's SSH information
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeMyUserProfile.html
     */
    toDescribeMyUserProfile() {
        this.add('opsworks:DescribeMyUserProfile');
        return this;
    }
    /**
     * Describes the permissions for a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribePermissions.html
     */
    toDescribePermissions() {
        this.add('opsworks:DescribePermissions');
        return this;
    }
    /**
     * Describe an instance's RAID arrays
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRaidArrays.html
     */
    toDescribeRaidArrays() {
        this.add('opsworks:DescribeRaidArrays');
        return this;
    }
    /**
     * Describes Amazon RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRdsDbInstances.html
     */
    toDescribeRdsDbInstances() {
        this.add('opsworks:DescribeRdsDbInstances');
        return this;
    }
    /**
     * Describes AWS OpsWorks service errors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeServiceErrors.html
     */
    toDescribeServiceErrors() {
        this.add('opsworks:DescribeServiceErrors');
        return this;
    }
    /**
     * Requests a description of a stack's provisioning parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackProvisioningParameters.html
     */
    toDescribeStackProvisioningParameters() {
        this.add('opsworks:DescribeStackProvisioningParameters');
        return this;
    }
    /**
     * Describes the number of layers and apps in a specified stack, and the number of instances in each state, such as running_setup or online
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackSummary.html
     */
    toDescribeStackSummary() {
        this.add('opsworks:DescribeStackSummary');
        return this;
    }
    /**
     * Requests a description of one or more stacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        this.add('opsworks:DescribeStacks');
        return this;
    }
    /**
     * Describes time-based auto scaling configurations for specified instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeTimeBasedAutoScaling.html
     */
    toDescribeTimeBasedAutoScaling() {
        this.add('opsworks:DescribeTimeBasedAutoScaling');
        return this;
    }
    /**
     * Describe specified users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeUserProfiles.html
     */
    toDescribeUserProfiles() {
        this.add('opsworks:DescribeUserProfiles');
        return this;
    }
    /**
     * Describes an instance's Amazon EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        this.add('opsworks:DescribeVolumes');
        return this;
    }
    /**
     * Detaches a specified Elastic Load Balancing instance from its layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DetachElasticLoadBalancer.html
     */
    toDetachElasticLoadBalancer() {
        this.add('opsworks:DetachElasticLoadBalancer');
        return this;
    }
    /**
     * Disassociates an Elastic IP address from its instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DisassociateElasticIp.html
     */
    toDisassociateElasticIp() {
        this.add('opsworks:DisassociateElasticIp');
        return this;
    }
    /**
     * Gets a generated host name for the specified layer, based on the current host name theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_GetHostnameSuggestion.html
     */
    toGetHostnameSuggestion() {
        this.add('opsworks:GetHostnameSuggestion');
        return this;
    }
    /**
     * Grants RDP access to a Windows instance for a specified time period
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    toGrantAccess() {
        this.add('opsworks:GrantAccess');
        return this;
    }
    /**
     * Returns a list of tags that are applied to the specified stack or layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.add('opsworks:ListTags');
        return this;
    }
    /**
     * Reboots a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    toRebootInstance() {
        this.add('opsworks:RebootInstance');
        return this;
    }
    /**
     * Registers a specified Amazon ECS cluster with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterEcsCluster.html
     */
    toRegisterEcsCluster() {
        this.add('opsworks:RegisterEcsCluster');
        return this;
    }
    /**
     * Registers an Elastic IP address with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterElasticIp.html
     */
    toRegisterElasticIp() {
        this.add('opsworks:RegisterElasticIp');
        return this;
    }
    /**
     * Registers instances with a specified stack that were created outside of AWS OpsWorks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterInstance.html
     */
    toRegisterInstance() {
        this.add('opsworks:RegisterInstance');
        return this;
    }
    /**
     * Registers an Amazon RDS instance with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterRdsDbInstance.html
     */
    toRegisterRdsDbInstance() {
        this.add('opsworks:RegisterRdsDbInstance');
        return this;
    }
    /**
     * Registers an Amazon EBS volume with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterVolume.html
     */
    toRegisterVolume() {
        this.add('opsworks:RegisterVolume');
        return this;
    }
    /**
     * Specify the load-based auto scaling configuration for a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetLoadBasedAutoScaling.html
     */
    toSetLoadBasedAutoScaling() {
        this.add('opsworks:SetLoadBasedAutoScaling');
        return this;
    }
    /**
     * Specifies a user's permissions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetPermission.html
     */
    toSetPermission() {
        this.add('opsworks:SetPermission');
        return this;
    }
    /**
     * Specify the time-based auto scaling configuration for a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetTimeBasedAutoScaling.html
     */
    toSetTimeBasedAutoScaling() {
        this.add('opsworks:SetTimeBasedAutoScaling');
        return this;
    }
    /**
     * Starts a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartInstance.html
     */
    toStartInstance() {
        this.add('opsworks:StartInstance');
        return this;
    }
    /**
     * Starts a stack's instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartStack.html
     */
    toStartStack() {
        this.add('opsworks:StartStack');
        return this;
    }
    /**
     * Stops a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopInstance.html
     */
    toStopInstance() {
        this.add('opsworks:StopInstance');
        return this;
    }
    /**
     * Stops a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopStack.html
     */
    toStopStack() {
        this.add('opsworks:StopStack');
        return this;
    }
    /**
     * Apply tags to a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('opsworks:TagResource');
        return this;
    }
    /**
     * Unassigns a registered instance from all of it's layers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignInstance.html
     */
    toUnassignInstance() {
        this.add('opsworks:UnassignInstance');
        return this;
    }
    /**
     * Unassigns an assigned Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignVolume.html
     */
    toUnassignVolume() {
        this.add('opsworks:UnassignVolume');
        return this;
    }
    /**
     * Removes tags from a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('opsworks:UntagResource');
        return this;
    }
    /**
     * Updates a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateApp.html
     */
    toUpdateApp() {
        this.add('opsworks:UpdateApp');
        return this;
    }
    /**
     * Updates a registered Elastic IP address's name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateElasticIp.html
     */
    toUpdateElasticIp() {
        this.add('opsworks:UpdateElasticIp');
        return this;
    }
    /**
     * Updates a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateInstance.html
     */
    toUpdateInstance() {
        this.add('opsworks:UpdateInstance');
        return this;
    }
    /**
     * Updates a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateLayer.html
     */
    toUpdateLayer() {
        this.add('opsworks:UpdateLayer');
        return this;
    }
    /**
     * Updates a user's SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateMyUserProfile.html
     */
    toUpdateMyUserProfile() {
        this.add('opsworks:UpdateMyUserProfile');
        return this;
    }
    /**
     * Updates an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateRdsDbInstance.html
     */
    toUpdateRdsDbInstance() {
        this.add('opsworks:UpdateRdsDbInstance');
        return this;
    }
    /**
     * Updates a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        this.add('opsworks:UpdateStack');
        return this;
    }
    /**
     * Updates a specified user profile
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateUserProfile.html
     */
    toUpdateUserProfile() {
        this.add('opsworks:UpdateUserProfile');
        return this;
    }
    /**
     * Updates an Amazon EBS volume's name or mount point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateVolume.html
     */
    toUpdateVolume() {
        this.add('opsworks:UpdateVolume');
        return this;
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks.html
     *
     * @param stackId - Identifier for the stackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStack(stackId, account, region, partition) {
        var arn = 'arn:${Partition}:opsworks:${Region}:${Account}:stack/${StackId}/';
        arn = arn.replace('${StackId}', stackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Opsworks = Opsworks;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzb3Bzd29ya3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3NvcHN3b3Jrcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBb3NCM0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUF6c0JOLGtCQUFhLEdBQUcsVUFBVSxDQUFDO1FBQ3hCLGVBQVUsR0FBWTtZQUM5QixnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHlDQUF5QztnQkFDeEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGtGQUFrRjtnQkFDakcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHlGQUF5RjtnQkFDeEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLG1DQUFtQztnQkFDbEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLGlCQUFpQjtnQkFDaEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLHFCQUFxQjtnQkFDcEMsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDRCQUE0QjtnQkFDM0MsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHlCQUF5QjtnQkFDeEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLG1GQUFtRjtnQkFDbEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDJCQUEyQjtnQkFDMUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDJCQUEyQjtnQkFDMUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtDQUFrQztnQkFDakQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDZDQUE2QztnQkFDNUQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQ0FBcUMsRUFBRTtnQkFDckMsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDBJQUEwSTtnQkFDekosYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDBFQUEwRTtnQkFDekYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLDBGQUEwRjtnQkFDekcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDhCQUE4QjtnQkFDN0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdEQUF3RDtnQkFDdkUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHNGQUFzRjtnQkFDckcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsNEVBQTRFO2dCQUMzRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsNkJBQTZCO2dCQUM1QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsNEJBQTRCO2dCQUMzQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixhQUFhLEVBQUUsNEJBQTRCO2dCQUMzQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUseUJBQXlCO2dCQUN4QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMENBQTBDO2dCQUN6RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUseURBQXlEO2dCQUN4RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsOENBQThDO2dCQUM3RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUseUJBQXlCO2dCQUN4QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUsZ0RBQWdEO2dCQUMvRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsOEJBQThCO2dCQUM3QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMkJBQTJCO2dCQUMxQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsZ0NBQWdDO2dCQUMvQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMkJBQTJCO2dCQUMxQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsa0NBQWtDO2dCQUNqRCxhQUFhLEVBQUUsd0JBQXdCO2FBQ3hDO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixLQUFLLEVBQUUsa0VBQWtFO2dCQUN6RSxlQUFlLEVBQUUsRUFBRTthQUNwQjtTQUNGLENBQUM7SUFTRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLElBQUksQ0FBQyxHQUFHLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN6RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxPQUFPLENBQUMsT0FBZSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25GLElBQUksR0FBRyxHQUFHLGtFQUFrRSxDQUFDO1FBQzdFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN6QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBM2tERCw0QkEya0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWN0aW9ucywgUG9saWN5U3RhdGVtZW50LCBSZXNvdXJjZVR5cGVzIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbb3Bzd29ya3NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c29wc3dvcmtzLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgT3Bzd29ya3MgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdvcHN3b3Jrcyc7XG4gIHByb3RlY3RlZCBhY3Rpb25MaXN0OiBBY3Rpb25zID0ge1xuICAgIFwiQXNzaWduSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzaWduSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkFzc2lnbiBhIHJlZ2lzdGVyZWQgaW5zdGFuY2UgdG8gYSBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXNzaWduVm9sdW1lXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc2lnblZvbHVtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXNzaWducyBvbmUgb2YgdGhlIHN0YWNrJ3MgcmVnaXN0ZXJlZCBBbWF6b24gRUJTIHZvbHVtZXMgdG8gYSBzcGVjaWZpZWQgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkFzc29jaWF0ZUVsYXN0aWNJcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVFbGFzdGljSXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkFzc29jaWF0ZXMgb25lIG9mIHRoZSBzdGFjaydzIHJlZ2lzdGVyZWQgRWxhc3RpYyBJUCBhZGRyZXNzZXMgd2l0aCBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXR0YWNoRWxhc3RpY0xvYWRCYWxhbmNlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BdHRhY2hFbGFzdGljTG9hZEJhbGFuY2VyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBdHRhY2hlcyBhbiBFbGFzdGljIExvYWQgQmFsYW5jaW5nIGxvYWQgYmFsYW5jZXIgdG8gYSBzcGVjaWZpZWQgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNsb25lU3RhY2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2xvbmVTdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIGNsb25lIG9mIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVBcHBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQXBwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGFuIGFwcCBmb3IgYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZURlcGxveW1lbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRGVwbG95bWVudC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUnVucyBkZXBsb3ltZW50IG9yIHN0YWNrIGNvbW1hbmRzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBpbnN0YW5jZSBpbiBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlTGF5ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTGF5ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlU3RhY2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU3RhY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBuZXcgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkNyZWF0ZVVzZXJQcm9maWxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVVzZXJQcm9maWxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbmV3IHVzZXIgcHJvZmlsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlQXBwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFwcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHNwZWNpZmllZCBhcHBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc3BlY2lmaWVkIGluc3RhbmNlLCB3aGljaCB0ZXJtaW5hdGVzIHRoZSBhc3NvY2lhdGVkIEFtYXpvbiBFQzIgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUxheWVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUxheWVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc3BlY2lmaWVkIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlVXNlclByb2ZpbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVXNlclByb2ZpbGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSB1c2VyIHByb2ZpbGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkRlcmVnaXN0ZXJFY3NDbHVzdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJFY3NDbHVzdGVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgdXNlciBwcm9maWxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXJlZ2lzdGVyRWxhc3RpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJFbGFzdGljSXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlcmVnaXN0ZXJzIGEgc3BlY2lmaWVkIEVsYXN0aWMgSVAgYWRkcmVzc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVyZWdpc3Rlckluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVyZWdpc3RlciBhIHJlZ2lzdGVyZWQgQW1hem9uIEVDMiBvciBvbi1wcmVtaXNlcyBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVyZWdpc3RlclJkc0RiSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlclJkc0RiSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlcmVnaXN0ZXJzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlcmVnaXN0ZXJWb2x1bWVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlclZvbHVtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVyZWdpc3RlcnMgYW4gQW1hem9uIEVCUyB2b2x1bWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQWdlbnRWZXJzaW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFnZW50VmVyc2lvbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyB0aGUgYXZhaWxhYmxlIEFXUyBPcHNXb3JrcyBhZ2VudCB2ZXJzaW9uc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUFwcHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBcHBzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXF1ZXN0cyBhIGRlc2NyaXB0aW9uIG9mIGEgc3BlY2lmaWVkIHNldCBvZiBhcHBzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQ29tbWFuZHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb21tYW5kcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRoZSByZXN1bHRzIG9mIHNwZWNpZmllZCBjb21tYW5kc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURlcGxveW1lbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRGVwbG95bWVudHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzcGVjaWZpZWQgc2V0IG9mIGRlcGxveW1lbnRzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlRWNzQ2x1c3RlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFY3NDbHVzdGVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIEFtYXpvbiBFQ1MgY2x1c3RlcnMgdGhhdCBhcmUgcmVnaXN0ZXJlZCB3aXRoIGEgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVFbGFzdGljSXBzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRWxhc3RpY0lwcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIEVsYXN0aWMgSVAgYWRkcmVzc2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlRWxhc3RpY0xvYWRCYWxhbmNlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2Vycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIGEgc3RhY2sncyBFbGFzdGljIExvYWQgQmFsYW5jaW5nIGluc3RhbmNlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUluc3RhbmNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUluc3RhbmNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHNldCBvZiBpbnN0YW5jZXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVMYXllcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMYXllcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2Ygb25lIG9yIG1vcmUgbGF5ZXJzIGluIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMb2FkQmFzZWRBdXRvU2NhbGluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIGxvYWQtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb25zIGZvciBzcGVjaWZpZWQgbGF5ZXJzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlTXlVc2VyUHJvZmlsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU15VXNlclByb2ZpbGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBhIHVzZXIncyBTU0ggaW5mb3JtYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVQZXJtaXNzaW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVBlcm1pc3Npb25zLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgdGhlIHBlcm1pc3Npb25zIGZvciBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVJhaWRBcnJheXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSYWlkQXJyYXlzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZSBhbiBpbnN0YW5jZSdzIFJBSUQgYXJyYXlzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlUmRzRGJJbnN0YW5jZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZHNEYkluc3RhbmNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIEFtYXpvbiBSRFMgaW5zdGFuY2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlU2VydmljZUVycm9yc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVNlcnZpY2VFcnJvcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBBV1MgT3BzV29ya3Mgc2VydmljZSBlcnJvcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVTdGFja1Byb3Zpc2lvbmluZ1BhcmFtZXRlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja1Byb3Zpc2lvbmluZ1BhcmFtZXRlcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzdGFjaydzIHByb3Zpc2lvbmluZyBwYXJhbWV0ZXJzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlU3RhY2tTdW1tYXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3RhY2tTdW1tYXJ5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgdGhlIG51bWJlciBvZiBsYXllcnMgYW5kIGFwcHMgaW4gYSBzcGVjaWZpZWQgc3RhY2ssIGFuZCB0aGUgbnVtYmVyIG9mIGluc3RhbmNlcyBpbiBlYWNoIHN0YXRlLCBzdWNoIGFzIHJ1bm5pbmdfc2V0dXAgb3Igb25saW5lXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlU3RhY2tzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3RhY2tzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXF1ZXN0cyBhIGRlc2NyaXB0aW9uIG9mIG9uZSBvciBtb3JlIHN0YWNrc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGltZUJhc2VkQXV0b1NjYWxpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyB0aW1lLWJhc2VkIGF1dG8gc2NhbGluZyBjb25maWd1cmF0aW9ucyBmb3Igc3BlY2lmaWVkIGluc3RhbmNlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVVzZXJQcm9maWxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVVzZXJQcm9maWxlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmUgc3BlY2lmaWVkIHVzZXJzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVm9sdW1lc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVZvbHVtZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBhbiBpbnN0YW5jZSdzIEFtYXpvbiBFQlMgdm9sdW1lc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXRhY2hFbGFzdGljTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRldGFjaGVzIGEgc3BlY2lmaWVkIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgaW5zdGFuY2UgZnJvbSBpdHMgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZUVsYXN0aWNJcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVFbGFzdGljSXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRpc2Fzc29jaWF0ZXMgYW4gRWxhc3RpYyBJUCBhZGRyZXNzIGZyb20gaXRzIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRIb3N0bmFtZVN1Z2dlc3Rpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SG9zdG5hbWVTdWdnZXN0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHZXRzIGEgZ2VuZXJhdGVkIGhvc3QgbmFtZSBmb3IgdGhlIHNwZWNpZmllZCBsYXllciwgYmFzZWQgb24gdGhlIGN1cnJlbnQgaG9zdCBuYW1lIHRoZW1lXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdyYW50QWNjZXNzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlYm9vdEluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgUkRQIGFjY2VzcyB0byBhIFdpbmRvd3MgaW5zdGFuY2UgZm9yIGEgc3BlY2lmaWVkIHRpbWUgcGVyaW9kXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0VGFnc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFncy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgdGFncyB0aGF0IGFyZSBhcHBsaWVkIHRvIHRoZSBzcGVjaWZpZWQgc3RhY2sgb3IgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUmVib290SW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVib290SW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlYm9vdHMgYSBzcGVjaWZpZWQgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlZ2lzdGVyRWNzQ2x1c3RlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckVjc0NsdXN0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlZ2lzdGVycyBhIHNwZWNpZmllZCBBbWF6b24gRUNTIGNsdXN0ZXIgd2l0aCBhIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZWdpc3RlckVsYXN0aWNJcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckVsYXN0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGFuIEVsYXN0aWMgSVAgYWRkcmVzcyB3aXRoIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZWdpc3Rlckluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVySW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlZ2lzdGVycyBpbnN0YW5jZXMgd2l0aCBhIHNwZWNpZmllZCBzdGFjayB0aGF0IHdlcmUgY3JlYXRlZCBvdXRzaWRlIG9mIEFXUyBPcHNXb3Jrc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUmVnaXN0ZXJSZHNEYkluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyUmRzRGJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2Ugd2l0aCBhIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZWdpc3RlclZvbHVtZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlclZvbHVtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGFuIEFtYXpvbiBFQlMgdm9sdW1lIHdpdGggYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlNldExvYWRCYXNlZEF1dG9TY2FsaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldExvYWRCYXNlZEF1dG9TY2FsaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTcGVjaWZ5IHRoZSBsb2FkLWJhc2VkIGF1dG8gc2NhbGluZyBjb25maWd1cmF0aW9uIGZvciBhIHNwZWNpZmllZCBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU2V0UGVybWlzc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TZXRQZXJtaXNzaW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTcGVjaWZpZXMgYSB1c2VyJ3MgcGVybWlzc2lvbnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU2V0VGltZUJhc2VkQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0VGltZUJhc2VkQXV0b1NjYWxpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNwZWNpZnkgdGhlIHRpbWUtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb24gZm9yIGEgc3BlY2lmaWVkIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdGFydEluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0SW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlN0YXJ0cyBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RhcnRTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFN0YWNrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdGFydHMgYSBzdGFjaydzIGluc3RhbmNlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RvcEluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RvcHMgYSBzcGVjaWZpZWQgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0b3BTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdG9wU3RhY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlN0b3BzIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJUYWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXBwbHkgdGFncyB0byBhIHNwZWNpZmllZCBzdGFjayBvciBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVW5hc3NpZ25JbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbmFzc2lnbkluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVbmFzc2lnbnMgYSByZWdpc3RlcmVkIGluc3RhbmNlIGZyb20gYWxsIG9mIGl0J3MgbGF5ZXJzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVbmFzc2lnblZvbHVtZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbmFzc2lnblZvbHVtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVW5hc3NpZ25zIGFuIGFzc2lnbmVkIEFtYXpvbiBFQlMgdm9sdW1lXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVbnRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlbW92ZXMgdGFncyBmcm9tIGEgc3BlY2lmaWVkIHN0YWNrIG9yIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVBcHBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQXBwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgc3BlY2lmaWVkIGFwcFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlRWxhc3RpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUVsYXN0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHJlZ2lzdGVyZWQgRWxhc3RpYyBJUCBhZGRyZXNzJ3MgbmFtZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBzcGVjaWZpZWQgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUxheWVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUxheWVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgc3BlY2lmaWVkIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVNeVVzZXJQcm9maWxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZU15VXNlclByb2ZpbGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSB1c2VyJ3MgU1NIIHB1YmxpYyBrZXlcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIlVwZGF0ZVJkc0RiSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlUmRzRGJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhbiBBbWF6b24gUkRTIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlVXNlclByb2ZpbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVXNlclByb2ZpbGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBzcGVjaWZpZWQgdXNlciBwcm9maWxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiXG4gICAgfSxcbiAgICBcIlVwZGF0ZVZvbHVtZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVWb2x1bWUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYW4gQW1hem9uIEVCUyB2b2x1bWUncyBuYW1lIG9yIG1vdW50IHBvaW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfTtcbiAgcHJvdGVjdGVkIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJzdGFja1wiOiB7XG4gICAgICBcIm5hbWVcIjogXCJzdGFja1wiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L3VzZXJndWlkZS93b3JraW5nc3RhY2tzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpvcHN3b3Jrczoke1JlZ2lvbn06JHtBY2NvdW50fTpzdGFjay8ke1N0YWNrSWR9L1wiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW29wc3dvcmtzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NvcHN3b3Jrcy5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NpZ24gYSByZWdpc3RlcmVkIGluc3RhbmNlIHRvIGEgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzaWduSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzaWduSW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkFzc2lnbkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXNzaWducyBvbmUgb2YgdGhlIHN0YWNrJ3MgcmVnaXN0ZXJlZCBBbWF6b24gRUJTIHZvbHVtZXMgdG8gYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzaWduVm9sdW1lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc2lnblZvbHVtZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6QXNzaWduVm9sdW1lJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXNzb2NpYXRlcyBvbmUgb2YgdGhlIHN0YWNrJ3MgcmVnaXN0ZXJlZCBFbGFzdGljIElQIGFkZHJlc3NlcyB3aXRoIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZUVsYXN0aWNJcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVFbGFzdGljSXAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkFzc29jaWF0ZUVsYXN0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaGVzIGFuIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgbG9hZCBiYWxhbmNlciB0byBhIHNwZWNpZmllZCBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BdHRhY2hFbGFzdGljTG9hZEJhbGFuY2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0F0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkF0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgY2xvbmUgb2YgYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2xvbmVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DbG9uZVN0YWNrKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDbG9uZVN0YWNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBhcHAgZm9yIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUFwcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBcHAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkNyZWF0ZUFwcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJ1bnMgZGVwbG95bWVudCBvciBzdGFjayBjb21tYW5kc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEZXBsb3ltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURlcGxveW1lbnQoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkNyZWF0ZURlcGxveW1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIGluc3RhbmNlIGluIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDcmVhdGVJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMYXllci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMYXllcigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6Q3JlYXRlTGF5ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVN0YWNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN0YWNrKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDcmVhdGVTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgdXNlciBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVVzZXJQcm9maWxlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDcmVhdGVVc2VyUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBzcGVjaWZpZWQgYXBwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFwcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBcHAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlbGV0ZUFwcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBzcGVjaWZpZWQgaW5zdGFuY2UsIHdoaWNoIHRlcm1pbmF0ZXMgdGhlIGFzc29jaWF0ZWQgQW1hem9uIEVDMiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWVkIGxheWVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUxheWVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxheWVyKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZWxldGVMYXllcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU3RhY2soKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlbGV0ZVN0YWNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHVzZXIgcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVVc2VyUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyUHJvZmlsZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlVXNlclByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgdXNlciBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJFY3NDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJFY3NDbHVzdGVyKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXJlZ2lzdGVyRWNzQ2x1c3RlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXJzIGEgc3BlY2lmaWVkIEVsYXN0aWMgSVAgYWRkcmVzc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVyRWxhc3RpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJFbGFzdGljSXAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlcmVnaXN0ZXJFbGFzdGljSXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXJlZ2lzdGVyIGEgcmVnaXN0ZXJlZCBBbWF6b24gRUMyIG9yIG9uLXByZW1pc2VzIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXJlZ2lzdGVySW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlcmVnaXN0ZXJJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXJzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlclJkc0RiSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVyZWdpc3RlclJkc0RiSW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlcmVnaXN0ZXJSZHNEYkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVyZWdpc3RlcnMgYW4gQW1hem9uIEVCUyB2b2x1bWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlclZvbHVtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXJlZ2lzdGVyVm9sdW1lKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXJlZ2lzdGVyVm9sdW1lJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBhdmFpbGFibGUgQVdTIE9wc1dvcmtzIGFnZW50IHZlcnNpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBZ2VudFZlcnNpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWdlbnRWZXJzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVBZ2VudFZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHNwZWNpZmllZCBzZXQgb2YgYXBwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQXBwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFwcHMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlQXBwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyB0aGUgcmVzdWx0cyBvZiBzcGVjaWZpZWQgY29tbWFuZHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNvbW1hbmRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ29tbWFuZHMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlQ29tbWFuZHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1ZXN0cyBhIGRlc2NyaXB0aW9uIG9mIGEgc3BlY2lmaWVkIHNldCBvZiBkZXBsb3ltZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRGVwbG95bWVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEZXBsb3ltZW50cygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVEZXBsb3ltZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBBbWF6b24gRUNTIGNsdXN0ZXJzIHRoYXQgYXJlIHJlZ2lzdGVyZWQgd2l0aCBhIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFY3NDbHVzdGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUVjc0NsdXN0ZXJzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZUVjc0NsdXN0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEVsYXN0aWMgSVAgYWRkcmVzc2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFbGFzdGljSXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRWxhc3RpY0lwcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVFbGFzdGljSXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGEgc3RhY2sncyBFbGFzdGljIExvYWQgQmFsYW5jaW5nIGluc3RhbmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRWxhc3RpY0xvYWRCYWxhbmNlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2VycygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzZXQgb2YgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbnN0YW5jZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbnN0YW5jZXMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlSW5zdGFuY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBvbmUgb3IgbW9yZSBsYXllcnMgaW4gYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxheWVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUxheWVycygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVMYXllcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgbG9hZC1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbnMgZm9yIHNwZWNpZmllZCBsYXllcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxvYWRCYXNlZEF1dG9TY2FsaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSB1c2VyJ3MgU1NIIGluZm9ybWF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVNeVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTXlVc2VyUHJvZmlsZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVNeVVzZXJQcm9maWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBwZXJtaXNzaW9ucyBmb3IgYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlUGVybWlzc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZSBhbiBpbnN0YW5jZSdzIFJBSUQgYXJyYXlzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSYWlkQXJyYXlzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmFpZEFycmF5cygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVSYWlkQXJyYXlzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEFtYXpvbiBSRFMgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZHNEYkluc3RhbmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJkc0RiSW5zdGFuY2VzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVJkc0RiSW5zdGFuY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEFXUyBPcHNXb3JrcyBzZXJ2aWNlIGVycm9yc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2VydmljZUVycm9ycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVNlcnZpY2VFcnJvcnMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlU2VydmljZUVycm9ycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzdGFjaydzIHByb3Zpc2lvbmluZyBwYXJhbWV0ZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja1Byb3Zpc2lvbmluZ1BhcmFtZXRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTdGFja1Byb3Zpc2lvbmluZ1BhcmFtZXRlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlU3RhY2tQcm92aXNpb25pbmdQYXJhbWV0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBudW1iZXIgb2YgbGF5ZXJzIGFuZCBhcHBzIGluIGEgc3BlY2lmaWVkIHN0YWNrLCBhbmQgdGhlIG51bWJlciBvZiBpbnN0YW5jZXMgaW4gZWFjaCBzdGF0ZSwgc3VjaCBhcyBydW5uaW5nX3NldHVwIG9yIG9ubGluZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3RhY2tTdW1tYXJ5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU3RhY2tTdW1tYXJ5KCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVN0YWNrU3VtbWFyeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2Ygb25lIG9yIG1vcmUgc3RhY2tzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTdGFja3MoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlU3RhY2tzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRpbWUtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb25zIGZvciBzcGVjaWZpZWQgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUaW1lQmFzZWRBdXRvU2NhbGluZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgc3BlY2lmaWVkIHVzZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVVc2VyUHJvZmlsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2VyUHJvZmlsZXMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlVXNlclByb2ZpbGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGFuIGluc3RhbmNlJ3MgQW1hem9uIEVCUyB2b2x1bWVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVWb2x1bWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVm9sdW1lcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVWb2x1bWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGV0YWNoZXMgYSBzcGVjaWZpZWQgRWxhc3RpYyBMb2FkIEJhbGFuY2luZyBpbnN0YW5jZSBmcm9tIGl0cyBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXRhY2hFbGFzdGljTG9hZEJhbGFuY2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEaXNhc3NvY2lhdGVzIGFuIEVsYXN0aWMgSVAgYWRkcmVzcyBmcm9tIGl0cyBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVFbGFzdGljSXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlRWxhc3RpY0lwKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEaXNhc3NvY2lhdGVFbGFzdGljSXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIGEgZ2VuZXJhdGVkIGhvc3QgbmFtZSBmb3IgdGhlIHNwZWNpZmllZCBsYXllciwgYmFzZWQgb24gdGhlIGN1cnJlbnQgaG9zdCBuYW1lIHRoZW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SG9zdG5hbWVTdWdnZXN0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEhvc3RuYW1lU3VnZ2VzdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6R2V0SG9zdG5hbWVTdWdnZXN0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIFJEUCBhY2Nlc3MgdG8gYSBXaW5kb3dzIGluc3RhbmNlIGZvciBhIHNwZWNpZmllZCB0aW1lIHBlcmlvZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWJvb3RJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HcmFudEFjY2VzcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6R3JhbnRBY2Nlc3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiB0YWdzIHRoYXQgYXJlIGFwcGxpZWQgdG8gdGhlIHNwZWNpZmllZCBzdGFjayBvciBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpMaXN0VGFncycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlYm9vdHMgYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVib290SW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVib290SW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlJlYm9vdEluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGEgc3BlY2lmaWVkIEFtYXpvbiBFQ1MgY2x1c3RlciB3aXRoIGEgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJFY3NDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVyRWNzQ2x1c3RlcigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJFY3NDbHVzdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIEVsYXN0aWMgSVAgYWRkcmVzcyB3aXRoIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyRWxhc3RpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVyRWxhc3RpY0lwKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpSZWdpc3RlckVsYXN0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBpbnN0YW5jZXMgd2l0aCBhIHNwZWNpZmllZCBzdGFjayB0aGF0IHdlcmUgY3JlYXRlZCBvdXRzaWRlIG9mIEFXUyBPcHNXb3Jrc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3Rlckluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVySW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlJlZ2lzdGVySW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlcnMgYW4gQW1hem9uIFJEUyBpbnN0YW5jZSB3aXRoIGEgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJSZHNEYkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVyUmRzRGJJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJSZHNEYkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIEFtYXpvbiBFQlMgdm9sdW1lIHdpdGggYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJWb2x1bWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVnaXN0ZXJWb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlJlZ2lzdGVyVm9sdW1lJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3BlY2lmeSB0aGUgbG9hZC1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpZWQgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0TG9hZEJhc2VkQXV0b1NjYWxpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0TG9hZEJhc2VkQXV0b1NjYWxpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlNldExvYWRCYXNlZEF1dG9TY2FsaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3BlY2lmaWVzIGEgdXNlcidzIHBlcm1pc3Npb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0UGVybWlzc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRQZXJtaXNzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpTZXRQZXJtaXNzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3BlY2lmeSB0aGUgdGltZS1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0VGltZUJhc2VkQXV0b1NjYWxpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0VGltZUJhc2VkQXV0b1NjYWxpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlNldFRpbWVCYXNlZEF1dG9TY2FsaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnRzIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0SW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6U3RhcnRJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXJ0cyBhIHN0YWNrJ3MgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0U3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6U3RhcnRTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wSW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlN0b3BJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wU3RhY2soKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlN0b3BTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFwcGx5IHRhZ3MgdG8gYSBzcGVjaWZpZWQgc3RhY2sgb3IgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVW5hc3NpZ25zIGEgcmVnaXN0ZXJlZCBpbnN0YW5jZSBmcm9tIGFsbCBvZiBpdCdzIGxheWVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbmFzc2lnbkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VuYXNzaWduSW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVuYXNzaWduSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVbmFzc2lnbnMgYW4gYXNzaWduZWQgQW1hem9uIEVCUyB2b2x1bWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW5hc3NpZ25Wb2x1bWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW5hc3NpZ25Wb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVuYXNzaWduVm9sdW1lJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyB0YWdzIGZyb20gYSBzcGVjaWZpZWQgc3RhY2sgb3IgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHNwZWNpZmllZCBhcHBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQXBwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFwcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6VXBkYXRlQXBwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHJlZ2lzdGVyZWQgRWxhc3RpYyBJUCBhZGRyZXNzJ3MgbmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVFbGFzdGljSXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRWxhc3RpY0lwKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVFbGFzdGljSXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBzcGVjaWZpZWQgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTGF5ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTGF5ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZUxheWVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHVzZXIncyBTU0ggcHVibGljIGtleVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVNeVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU15VXNlclByb2ZpbGUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZU15VXNlclByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlUmRzRGJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZHNEYkluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVSZHNEYkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6VXBkYXRlU3RhY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgc3BlY2lmaWVkIHVzZXIgcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVVzZXJQcm9maWxlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVVc2VyUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYW4gQW1hem9uIEVCUyB2b2x1bWUncyBuYW1lIG9yIG1vdW50IHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVZvbHVtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVWb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZVZvbHVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHN0YWNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC91c2VyZ3VpZGUvd29ya2luZ3N0YWNrcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzdGFja0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YWNrSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25TdGFjayhzdGFja0lkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259Om9wc3dvcmtzOiR7UmVnaW9ufToke0FjY291bnR9OnN0YWNrLyR7U3RhY2tJZH0vJztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtTdGFja0lkfScsIHN0YWNrSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=