"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ram = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ram extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ram](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ram';
        this.actionList = {
            "AcceptResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html",
                "description": "Accept the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "AssociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html",
                "description": "Associates resource(s) and/or principal(s) to a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "AssociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html",
                "description": "Associate a Permission with a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "CreateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html",
                "description": "Create resource share with provided resource(s) and/or principal(s)",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn",
                    "ram:RequestedAllowsExternalPrincipals",
                    "ram:Principal"
                ]
            },
            "DeleteResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html",
                "description": "Delete resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "DisassociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html",
                "description": "Disassociates resource(s) and/or principal(s) from a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "DisassociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html",
                "description": "Disassociate a Permission from a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "EnableSharingWithAwsOrganization": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html",
                "description": "Grants permission to access customer's organization and create a SLR in the customer's account",
                "accessLevel": "Write"
            },
            "GetPermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html",
                "description": "Gets the contents of an AWS RAM permission",
                "accessLevel": "Read",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:PermissionArn"
                ]
            },
            "GetResourcePolicies": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html",
                "description": "Gets the policies for the specified resources that you own and have shared",
                "accessLevel": "Read"
            },
            "GetResourceShareAssociations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html",
                "description": "Get a set of resource share associations from a provided list or with a specified status of the specified type",
                "accessLevel": "Read"
            },
            "GetResourceShareInvitations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html",
                "description": "Get resource share invitations by the specified invitation arn or those for the resource share",
                "accessLevel": "Read"
            },
            "GetResourceShares": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html",
                "description": "Get a set of resource shares from a provided list or with a specified status",
                "accessLevel": "Read"
            },
            "ListPendingInvitationResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html",
                "description": "Lists the resources in a resource share that is shared with you but that the invitation is still pending for",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                }
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the AWS RAM permissions",
                "accessLevel": "List"
            },
            "ListPrincipals": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html",
                "description": "Lists the principals that you have shared resources with or that have shared resources with you",
                "accessLevel": "List"
            },
            "ListResourceSharePermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html",
                "description": "List the Permissions associated with a Resource Share",
                "accessLevel": "List",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "ListResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html",
                "description": "Lists the resources that you added to a resource shares or the resources that are shared with you",
                "accessLevel": "List"
            },
            "RejectResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html",
                "description": "Reject the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html",
                "description": "Tag the specified resources share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html",
                "description": "Untag the specified resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html",
                "description": "Update attributes of the resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:RequestedAllowsExternalPrincipals"
                ]
            }
        };
        this.resourceTypes = {
            "resource-share": {
                "name": "resource-share",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName"
                ]
            },
            "resource-share-invitation": {
                "name": "resource-share-invitation",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}",
                "conditionKeys": []
            },
            "permission": {
                "name": "permission",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html",
                "arn": "arn:${Partition}:ram::${Account}:permission/${ResourcePath}",
                "conditionKeys": [
                    "ram:PermissionArn"
                ]
            }
        };
    }
    /**
     * Accept the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html
     */
    toAcceptResourceShareInvitation() {
        this.add('ram:AcceptResourceShareInvitation');
        return this;
    }
    /**
     * Associates resource(s) and/or principal(s) to a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html
     */
    toAssociateResourceShare() {
        this.add('ram:AssociateResourceShare');
        return this;
    }
    /**
     * Associate a Permission with a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html
     */
    toAssociateResourceSharePermission() {
        this.add('ram:AssociateResourceSharePermission');
        return this;
    }
    /**
     * Create resource share with provided resource(s) and/or principal(s)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     * - .ifRequestedAllowsExternalPrincipals()
     * - .ifPrincipal()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html
     */
    toCreateResourceShare() {
        this.add('ram:CreateResourceShare');
        return this;
    }
    /**
     * Delete resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html
     */
    toDeleteResourceShare() {
        this.add('ram:DeleteResourceShare');
        return this;
    }
    /**
     * Disassociates resource(s) and/or principal(s) from a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html
     */
    toDisassociateResourceShare() {
        this.add('ram:DisassociateResourceShare');
        return this;
    }
    /**
     * Disassociate a Permission from a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html
     */
    toDisassociateResourceSharePermission() {
        this.add('ram:DisassociateResourceSharePermission');
        return this;
    }
    /**
     * Grants permission to access customer's organization and create a SLR in the customer's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html
     */
    toEnableSharingWithAwsOrganization() {
        this.add('ram:EnableSharingWithAwsOrganization');
        return this;
    }
    /**
     * Gets the contents of an AWS RAM permission
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html
     */
    toGetPermission() {
        this.add('ram:GetPermission');
        return this;
    }
    /**
     * Gets the policies for the specified resources that you own and have shared
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html
     */
    toGetResourcePolicies() {
        this.add('ram:GetResourcePolicies');
        return this;
    }
    /**
     * Get a set of resource share associations from a provided list or with a specified status of the specified type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html
     */
    toGetResourceShareAssociations() {
        this.add('ram:GetResourceShareAssociations');
        return this;
    }
    /**
     * Get resource share invitations by the specified invitation arn or those for the resource share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html
     */
    toGetResourceShareInvitations() {
        this.add('ram:GetResourceShareInvitations');
        return this;
    }
    /**
     * Get a set of resource shares from a provided list or with a specified status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html
     */
    toGetResourceShares() {
        this.add('ram:GetResourceShares');
        return this;
    }
    /**
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html
     */
    toListPendingInvitationResources() {
        this.add('ram:ListPendingInvitationResources');
        return this;
    }
    /**
     * Lists the AWS RAM permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        this.add('ram:ListPermissions');
        return this;
    }
    /**
     * Lists the principals that you have shared resources with or that have shared resources with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html
     */
    toListPrincipals() {
        this.add('ram:ListPrincipals');
        return this;
    }
    /**
     * List the Permissions associated with a Resource Share
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html
     */
    toListResourceSharePermissions() {
        this.add('ram:ListResourceSharePermissions');
        return this;
    }
    /**
     * Lists the resources that you added to a resource shares or the resources that are shared with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        this.add('ram:ListResources');
        return this;
    }
    /**
     * Reject the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html
     */
    toRejectResourceShareInvitation() {
        this.add('ram:RejectResourceShareInvitation');
        return this;
    }
    /**
     * Tag the specified resources share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('ram:TagResource');
        return this;
    }
    /**
     * Untag the specified resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('ram:UntagResource');
        return this;
    }
    /**
     * Update attributes of the resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifRequestedAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html
     */
    toUpdateResourceShare() {
        this.add('ram:UpdateResourceShare');
        return this;
    }
    /**
     * Adds a resource of type resource-share to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     */
    onResourceShare(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resource-share-invitation to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceShareInvitation(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifPermissionArn()
     */
    onPermission(resourcePath, account, partition) {
        var arn = 'arn:${Partition}:ram::${Account}:permission/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Specifies a tag key and value pair that must be used when creating or tagging a resource share. If users don't pass these specific tags, or if they don't specify tags at all, the request fails.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateResourceShare()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resources that have the specified tag key and value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toAssociateResourceSharePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toDisassociateResourceSharePermission()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Specifies the tag keys that can be used when creating or tagging a resource share
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateResourceShare()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resource shares that allow or deny sharing with external principals. For example, specify true if the action can only be performed on resource shares that allow sharing with external principals. External principals are AWS accounts that are outside of its AWS organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toAssociateResourceSharePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toDisassociateResourceSharePermission()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowsExternalPrincipals(value) {
        return this.if(`ram:AllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on a resource using the specified Permission ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceSharePermission()
     * - .toDisassociateResourceSharePermission()
     * - .toGetPermission()
     *
     * Applies to resource types:
     * - permission
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPermissionArn(value, operator) {
        return this.if(`ram:PermissionArn`, value, operator || 'ArnEquals');
    }
    /**
     * Principals with the specified format can be associated to or disassociated from a resource share
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`ram:Principal`, value, operator || 'StringLike');
    }
    /**
     * The request must have the specified value for 'allowExternalPrincipals'. External principals are AWS accounts that are outside of its AWS Organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toCreateResourceShare()
     * - .toUpdateResourceShare()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestedAllowsExternalPrincipals(value) {
        return this.if(`ram:RequestedAllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on the specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedResourceType(value, operator) {
        return this.if(`ram:RequestedResourceType`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on a resource with the specified ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResourceArn(value, operator) {
        return this.if(`ram:ResourceArn`, value, operator || 'ArnEquals');
    }
    /**
     * Indicates that the action can only be performed on a resource share with the specified name.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toAssociateResourceSharePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toDisassociateResourceSharePermission()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceShareName(value, operator) {
        return this.if(`ram:ResourceShareName`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resource shares owned by a specific account. For example, you can use this condition key to specify which resource share invitations can be accepted or rejected based on the resource share owner’s account ID.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAcceptResourceShareInvitation()
     * - .toRejectResourceShareInvitation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareOwnerAccountId(value, operator) {
        return this.if(`ram:ShareOwnerAccountId`, value, operator || 'StringLike');
    }
}
exports.Ram = Ram;
//# sourceMappingURL=data:application/json;base64,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