"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Snowball = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [snowball](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssnowball.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Snowball extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [snowball](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssnowball.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'snowball';
        this.actionList = {
            "CancelCluster": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelCluster.html",
                "description": "Cancels a cluster job.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelJob.html",
                "description": "Cancels the specified job.",
                "accessLevel": "Write"
            },
            "CreateAddress": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateAddress.html",
                "description": "Creates an address for a Snowball to be shipped to.",
                "accessLevel": "Write"
            },
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateCluster.html",
                "description": "Creates an empty cluster.",
                "accessLevel": "Write"
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateJob.html",
                "description": "Creates a job to import or export data between Amazon S3 and your on-premises data center.",
                "accessLevel": "Write"
            },
            "DescribeAddress": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddress.html",
                "description": "Takes an AddressId and returns specific details about that address in the form of an Address object.",
                "accessLevel": "Read"
            },
            "DescribeAddresses": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddresses.html",
                "description": "Returns a specified number of ADDRESS objects.",
                "accessLevel": "List"
            },
            "DescribeCluster": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeCluster.html",
                "description": "Returns information about a specific cluster including shipping information, cluster status, and other important metadata.",
                "accessLevel": "Read"
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeJob.html",
                "description": "Returns information about a specific job including shipping information, job status, and other important metadata.",
                "accessLevel": "Read"
            },
            "GetJobManifest": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobManifest.html",
                "description": "Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified JobId value.",
                "accessLevel": "Read"
            },
            "GetJobUnlockCode": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobUnlockCode.html",
                "description": "Returns the UnlockCode code value for the specified job.",
                "accessLevel": "Read"
            },
            "GetSnowballUsage": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetSnowballUsage.html",
                "description": "Returns information about the Snowball service limit for your account, and also the number of Snowballs your account has in use.",
                "accessLevel": "Read"
            },
            "ListClusterJobs": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusterJobs.html",
                "description": "Returns an array of JobListEntry objects of the specified length.",
                "accessLevel": "List"
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusters.html",
                "description": "Returns an array of ClusterListEntry objects of the specified length.",
                "accessLevel": "List"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListJobs.html",
                "description": "Returns an array of JobListEntry objects of the specified length.",
                "accessLevel": "List"
            },
            "UpdateCluster": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateCluster.html",
                "description": "While a cluster's ClusterState value is in the AwaitingQuorum state, you can update some of the information associated with a cluster.",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateJob.html",
                "description": "While a job's JobState value is New, you can update some of the information associated with a job.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Cancels a cluster job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelCluster.html
     */
    toCancelCluster() {
        this.add('snowball:CancelCluster');
        return this;
    }
    /**
     * Cancels the specified job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CancelJob.html
     */
    toCancelJob() {
        this.add('snowball:CancelJob');
        return this;
    }
    /**
     * Creates an address for a Snowball to be shipped to.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateAddress.html
     */
    toCreateAddress() {
        this.add('snowball:CreateAddress');
        return this;
    }
    /**
     * Creates an empty cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateCluster.html
     */
    toCreateCluster() {
        this.add('snowball:CreateCluster');
        return this;
    }
    /**
     * Creates a job to import or export data between Amazon S3 and your on-premises data center.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_CreateJob.html
     */
    toCreateJob() {
        this.add('snowball:CreateJob');
        return this;
    }
    /**
     * Takes an AddressId and returns specific details about that address in the form of an Address object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddress.html
     */
    toDescribeAddress() {
        this.add('snowball:DescribeAddress');
        return this;
    }
    /**
     * Returns a specified number of ADDRESS objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeAddresses.html
     */
    toDescribeAddresses() {
        this.add('snowball:DescribeAddresses');
        return this;
    }
    /**
     * Returns information about a specific cluster including shipping information, cluster status, and other important metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        this.add('snowball:DescribeCluster');
        return this;
    }
    /**
     * Returns information about a specific job including shipping information, job status, and other important metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_DescribeJob.html
     */
    toDescribeJob() {
        this.add('snowball:DescribeJob');
        return this;
    }
    /**
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified JobId value.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobManifest.html
     */
    toGetJobManifest() {
        this.add('snowball:GetJobManifest');
        return this;
    }
    /**
     * Returns the UnlockCode code value for the specified job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetJobUnlockCode.html
     */
    toGetJobUnlockCode() {
        this.add('snowball:GetJobUnlockCode');
        return this;
    }
    /**
     * Returns information about the Snowball service limit for your account, and also the number of Snowballs your account has in use.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_GetSnowballUsage.html
     */
    toGetSnowballUsage() {
        this.add('snowball:GetSnowballUsage');
        return this;
    }
    /**
     * Returns an array of JobListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusterJobs.html
     */
    toListClusterJobs() {
        this.add('snowball:ListClusterJobs');
        return this;
    }
    /**
     * Returns an array of ClusterListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListClusters.html
     */
    toListClusters() {
        this.add('snowball:ListClusters');
        return this;
    }
    /**
     * Returns an array of JobListEntry objects of the specified length.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_ListJobs.html
     */
    toListJobs() {
        this.add('snowball:ListJobs');
        return this;
    }
    /**
     * While a cluster's ClusterState value is in the AwaitingQuorum state, you can update some of the information associated with a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        this.add('snowball:UpdateCluster');
        return this;
    }
    /**
     * While a job's JobState value is New, you can update some of the information associated with a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/snowball/latest/api-reference/API_UpdateJob.html
     */
    toUpdateJob() {
        this.add('snowball:UpdateJob');
        return this;
    }
}
exports.Snowball = Snowball;
//# sourceMappingURL=data:application/json;base64,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