"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wafv2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [wafv2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswafv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wafv2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [wafv2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswafv2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'wafv2';
        this.actionList = {
            "AssociateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_AssociateWebACL.html",
                "description": "Grants permission to associate a WebACL with a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    },
                    "apigateway": {
                        "required": false
                    },
                    "loadbalancer/app/": {
                        "required": false
                    }
                }
            },
            "CheckCapacity": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_CheckCapacity.html",
                "description": "Grants permission to calculate web ACL capacity unit (WCU) requirements for a specified scope and set of rules.",
                "accessLevel": "Read"
            },
            "CreateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateIPSet.html",
                "description": "Grants permission to create an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRegexPatternSet.html",
                "description": "Grants permission to create a RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRuleGroup.html",
                "description": "Grants permission to create a RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateWebACL.html",
                "description": "Grants permission to create a WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteFirewallManagerRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteFirewallManagerRuleGroups.html",
                "description": "Grants permission to delete specified FirewallManagedRulesGroups from the specified WebACL if not managed by Firewall Manager anymore.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeleteIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteIPSet.html",
                "description": "Grants permission to delete the specified IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "DeleteLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteLoggingConfiguration.html",
                "description": "Grants permission to delete the LoggingConfiguration from the specified WebACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeletePermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeletePermissionPolicy.html",
                "description": "Grants permission to delete the PermissionPolicy on the specified RuleGroup.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRegexPatternSet.html",
                "description": "Grants permission to delete the specified RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "DeleteRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRuleGroup.html",
                "description": "Grants permission to delete the specified RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteWebACL.html",
                "description": "Grants permission to delete the specified WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DescribeManagedRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedRuleGroup.html",
                "description": "Grants permission to view high-level information for a managed rule group.",
                "accessLevel": "List"
            },
            "DisassociateFirewallManager": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateFirewallManager.html",
                "description": "Grants permission to disassociate Firewall Manager from the specified WebACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DisassociateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateWebACL.html",
                "description": "Grants permission disassociate a WebACL from an application resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apigateway": {
                        "required": false
                    },
                    "loadbalancer/app/": {
                        "required": false
                    }
                }
            },
            "GetIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetIPSet.html",
                "description": "Grants permission to view details about the specified IPSet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetLoggingConfiguration.html",
                "description": "Grants permission to view LoggingConfiguration about the specified WebACL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetPermissionPolicy.html",
                "description": "Grants permission to view PermissionPolicy on the specified RuleGroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "GetRateBasedStatementManagedKeys": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRateBasedStatementManagedKeys.html",
                "description": "Grants permission to view the keys that are currently blocked by a rate-based rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRegexPatternSet.html",
                "description": "Grants permission to view details about the specified RegexPatternSet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRuleGroup.html",
                "description": "Grants permission to view details about the specified RuleGroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetSampledRequests": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetSampledRequests.html",
                "description": "Grants permission to view detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "GetWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACL.html",
                "description": "Grants permission to view details about the specified GetWebACL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetWebACLForResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACLForResource.html",
                "description": "Grants permission to view the WebACL for the specified resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apigateway": {
                        "required": false
                    },
                    "loadbalancer/app/": {
                        "required": false
                    }
                }
            },
            "ListAvailableManagedRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroups.html",
                "description": "Grants permission to view an array of managed rule groups that are available for you to use.",
                "accessLevel": "List"
            },
            "ListIPSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListIPSets.html",
                "description": "Grants permission to view an array of IPSetSummary objects for the IP sets that you manage.",
                "accessLevel": "List"
            },
            "ListLoggingConfigurations": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListLoggingConfigurations.html",
                "description": "Grants permission to view an array of your LoggingConfiguration objects.",
                "accessLevel": "List"
            },
            "ListRegexPatternSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRegexPatternSets.html",
                "description": "Grants permission to view an array of RegexPatternSetSummary objects for the regex pattern sets that you manage.",
                "accessLevel": "List"
            },
            "ListResourcesForWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListResourcesForWebACL.html",
                "description": "Grants permission to view an array of the Amazon Resource Names (ARNs) for the resources that are associated with the specified web ACL.",
                "accessLevel": "List",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "ListRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRuleGroups.html",
                "description": "Grants permission to view an array of RuleGroupSummary objects for the rule groups that you manage.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to lists tag for the specified resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ipset": {
                        "required": false
                    },
                    "regexpatternset": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ListWebACLs": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_ListWebACLs.html",
                "description": "Grants permission to view an array of WebACLSummary objects for the web ACLs that you manage.",
                "accessLevel": "List"
            },
            "PutFirewallManagerRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_PutFirewallManagerRuleGroups.html",
                "description": "Grants permission to create FirewallManagedRulesGroups in the specified WebACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "PutLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_PutLoggingConfiguration.html",
                "description": "Grants permission to enables the specified LoggingConfiguration, to start logging from a web ACL.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "PutPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_PutPermissionPolicy.html",
                "description": "Grants permission to attach the specified IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to associates tags with the specified AWS resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ipset": {
                        "required": false
                    },
                    "regexpatternset": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to disassociates tags from an AWS resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ipset": {
                        "required": false
                    },
                    "regexpatternset": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateIPSet.html",
                "description": "Grants permission to update the specified IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRegexPatternSet.html",
                "description": "Grants permission to update the specified RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRuleGroup.html",
                "description": "Grants permission to update the specified RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateWebACL.html",
                "description": "Grants permission to update the specified WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "webacl": {
                "name": "webacl",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html",
                "arn": "arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/webacl/${Name}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ipset": {
                "name": "ipset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_IPSet.html",
                "arn": "arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/ipset/${Name}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rulegroup": {
                "name": "rulegroup",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_RuleGroup.html",
                "arn": "arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/rulegroup/${Name}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "regexpatternset": {
                "name": "regexpatternset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_RegexPatternSet.html",
                "arn": "arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/regexpatternset/${Name}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "loadbalancer/app/": {
                "name": "loadbalancer/app/",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html",
                "arn": "arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}",
                "conditionKeys": []
            },
            "apigateway": {
                "name": "apigateway",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html",
                "arn": "arn:${Partition}:apigateway:${Region}:${Account}:/restapis/${ApiId}/stages/${StageName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate a WebACL with a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_AssociateWebACL.html
     */
    toAssociateWebACL() {
        this.add('wafv2:AssociateWebACL');
        return this;
    }
    /**
     * Grants permission to calculate web ACL capacity unit (WCU) requirements for a specified scope and set of rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CheckCapacity.html
     */
    toCheckCapacity() {
        this.add('wafv2:CheckCapacity');
        return this;
    }
    /**
     * Grants permission to create an IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        this.add('wafv2:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create a RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRegexPatternSet.html
     */
    toCreateRegexPatternSet() {
        this.add('wafv2:CreateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to create a RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        this.add('wafv2:CreateRuleGroup');
        return this;
    }
    /**
     * Grants permission to create a WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateWebACL.html
     */
    toCreateWebACL() {
        this.add('wafv2:CreateWebACL');
        return this;
    }
    /**
     * Grants permission to delete specified FirewallManagedRulesGroups from the specified WebACL if not managed by Firewall Manager anymore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteFirewallManagerRuleGroups.html
     */
    toDeleteFirewallManagerRuleGroups() {
        this.add('wafv2:DeleteFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.add('wafv2:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete the LoggingConfiguration from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        this.add('wafv2:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeletePermissionPolicy.html
     */
    toDeletePermissionPolicy() {
        this.add('wafv2:DeletePermissionPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRegexPatternSet.html
     */
    toDeleteRegexPatternSet() {
        this.add('wafv2:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to delete the specified RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        this.add('wafv2:DeleteRuleGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteWebACL.html
     */
    toDeleteWebACL() {
        this.add('wafv2:DeleteWebACL');
        return this;
    }
    /**
     * Grants permission to view high-level information for a managed rule group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedRuleGroup.html
     */
    toDescribeManagedRuleGroup() {
        this.add('wafv2:DescribeManagedRuleGroup');
        return this;
    }
    /**
     * Grants permission to disassociate Firewall Manager from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateFirewallManager.html
     */
    toDisassociateFirewallManager() {
        this.add('wafv2:DisassociateFirewallManager');
        return this;
    }
    /**
     * Grants permission disassociate a WebACL from an application resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateWebACL.html
     */
    toDisassociateWebACL() {
        this.add('wafv2:DisassociateWebACL');
        return this;
    }
    /**
     * Grants permission to view details about the specified IPSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        this.add('wafv2:GetIPSet');
        return this;
    }
    /**
     * Grants permission to view LoggingConfiguration about the specified WebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetLoggingConfiguration.html
     */
    toGetLoggingConfiguration() {
        this.add('wafv2:GetLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to view PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetPermissionPolicy.html
     */
    toGetPermissionPolicy() {
        this.add('wafv2:GetPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to view the keys that are currently blocked by a rate-based rule.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRateBasedStatementManagedKeys.html
     */
    toGetRateBasedStatementManagedKeys() {
        this.add('wafv2:GetRateBasedStatementManagedKeys');
        return this;
    }
    /**
     * Grants permission to view details about the specified RegexPatternSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRegexPatternSet.html
     */
    toGetRegexPatternSet() {
        this.add('wafv2:GetRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to view details about the specified RuleGroup.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRuleGroup.html
     */
    toGetRuleGroup() {
        this.add('wafv2:GetRuleGroup');
        return this;
    }
    /**
     * Grants permission to view detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetSampledRequests.html
     */
    toGetSampledRequests() {
        this.add('wafv2:GetSampledRequests');
        return this;
    }
    /**
     * Grants permission to view details about the specified GetWebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACL.html
     */
    toGetWebACL() {
        this.add('wafv2:GetWebACL');
        return this;
    }
    /**
     * Grants permission to view the WebACL for the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACLForResource.html
     */
    toGetWebACLForResource() {
        this.add('wafv2:GetWebACLForResource');
        return this;
    }
    /**
     * Grants permission to view an array of managed rule groups that are available for you to use.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroups.html
     */
    toListAvailableManagedRuleGroups() {
        this.add('wafv2:ListAvailableManagedRuleGroups');
        return this;
    }
    /**
     * Grants permission to view an array of IPSetSummary objects for the IP sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        this.add('wafv2:ListIPSets');
        return this;
    }
    /**
     * Grants permission to view an array of your LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListLoggingConfigurations.html
     */
    toListLoggingConfigurations() {
        this.add('wafv2:ListLoggingConfigurations');
        return this;
    }
    /**
     * Grants permission to view an array of RegexPatternSetSummary objects for the regex pattern sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRegexPatternSets.html
     */
    toListRegexPatternSets() {
        this.add('wafv2:ListRegexPatternSets');
        return this;
    }
    /**
     * Grants permission to view an array of the Amazon Resource Names (ARNs) for the resources that are associated with the specified web ACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListResourcesForWebACL.html
     */
    toListResourcesForWebACL() {
        this.add('wafv2:ListResourcesForWebACL');
        return this;
    }
    /**
     * Grants permission to view an array of RuleGroupSummary objects for the rule groups that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        this.add('wafv2:ListRuleGroups');
        return this;
    }
    /**
     * Grants permission to lists tag for the specified resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('wafv2:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to view an array of WebACLSummary objects for the web ACLs that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListWebACLs.html
     */
    toListWebACLs() {
        this.add('wafv2:ListWebACLs');
        return this;
    }
    /**
     * Grants permission to create FirewallManagedRulesGroups in the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutFirewallManagerRuleGroups.html
     */
    toPutFirewallManagerRuleGroups() {
        this.add('wafv2:PutFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to enables the specified LoggingConfiguration, to start logging from a web ACL.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutLoggingConfiguration.html
     */
    toPutLoggingConfiguration() {
        this.add('wafv2:PutLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to attach the specified IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutPermissionPolicy.html
     */
    toPutPermissionPolicy() {
        this.add('wafv2:PutPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to associates tags with the specified AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('wafv2:TagResource');
        return this;
    }
    /**
     * Grants permission to disassociates tags from an AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('wafv2:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the specified IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.add('wafv2:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRegexPatternSet.html
     */
    toUpdateRegexPatternSet() {
        this.add('wafv2:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to update the specified RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        this.add('wafv2:UpdateRuleGroup');
        return this;
    }
    /**
     * Grants permission to update the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateWebACL.html
     */
    toUpdateWebACL() {
        this.add('wafv2:UpdateWebACL');
        return this;
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/webacl/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_IPSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/ipset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RuleGroup.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/rulegroup/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RegexPatternSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegexpatternset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/regexpatternset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type apigateway to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param apiId - Identifier for the apiId.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApigateway(apiId, stageName, account, region, partition) {
        var arn = 'arn:${Partition}:apigateway:${Region}:${Account}:/restapis/${ApiId}/stages/${StageName}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${StageName}', stageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateIPSet()
     * - .toCreateRegexPatternSet()
     * - .toCreateRuleGroup()
     * - .toCreateWebACL()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetIPSet()
     * - .toGetLoggingConfiguration()
     * - .toGetRateBasedStatementManagedKeys()
     * - .toGetRegexPatternSet()
     * - .toGetRuleGroup()
     * - .toGetWebACL()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUpdateIPSet()
     * - .toUpdateRegexPatternSet()
     * - .toUpdateRuleGroup()
     * - .toUpdateWebACL()
     *
     * Applies to resource types:
     * - webacl
     * - ipset
     * - rulegroup
     * - regexpatternset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateIPSet()
     * - .toCreateRegexPatternSet()
     * - .toCreateRuleGroup()
     * - .toCreateWebACL()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Wafv2 = Wafv2;
//# sourceMappingURL=data:application/json;base64,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