import { Actions, PolicyStatement, PolicyStatementWithCondition, ResourceTypes } from "../shared";
/**
 * Statement provider for service [elemental-activations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_elementalactivations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class ElementalActivations extends PolicyStatement {
    servicePrefix: string;
    protected actionList: Actions;
    protected resourceTypes: ResourceTypes;
    /**
     * Statement provider for service [elemental-activations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_elementalactivations.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Download the Software files for AWS Elemental Appliances and Software Purchases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toDownloadSoftware(): this;
    /**
     * Generate Software Licenses for AWS Elemental Appliances and Software Purchases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toGenerateLicenses(): this;
    /**
     * Describe an activation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toGetActivation(): this;
    /**
     * This action lists tags for an AWS Elemental Activations resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toListTagsForResource(): this;
    /**
     * This action adds a tag for an AWS Elemental Activations resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toTagResource(): this;
    /**
     * This action removes a tag from an AWS Elemental Activations resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     */
    toUntagResource(): this;
    /**
     * Adds a resource of type activation to the statement
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActivation(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: string): PolicyStatementWithCondition;
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - activation
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: string): PolicyStatementWithCondition;
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: string): PolicyStatementWithCondition;
}
