"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicequotas = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicequotas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_servicequotas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicequotas extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicequotas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_servicequotas.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicequotas';
        this.actionList = {
            "AssociateServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html",
                "description": "Grants permission to associate the Service Quotas template with your organization",
                "accessLevel": "Write"
            },
            "DeleteServiceQuotaIncreaseRequestFromTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html",
                "description": "Grants permission to remove the specified service quota from the service quota template",
                "accessLevel": "Write"
            },
            "DisassociateServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html",
                "description": "Grants permission to disassociate the Service Quotas template from your organization",
                "accessLevel": "Write"
            },
            "GetAWSDefaultServiceQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html",
                "description": "Grants permission to return the details for the specified service quota, including the AWS default value",
                "accessLevel": "Read"
            },
            "GetAssociationForServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html",
                "description": "Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization",
                "accessLevel": "Read"
            },
            "GetRequestedServiceQuotaChange": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html",
                "description": "Grants permission to retrieve the details for a particular service quota increase request",
                "accessLevel": "Read"
            },
            "GetServiceQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html",
                "description": "Grants permission to return the details for the specified service quota, including the applied value",
                "accessLevel": "Read"
            },
            "GetServiceQuotaIncreaseRequestFromTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html",
                "description": "Grants permission to retrieve the details for a service quota increase request from the service quota template",
                "accessLevel": "Read"
            },
            "ListAWSDefaultServiceQuotas": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html",
                "description": "Grants permission to list all default service quotas for the specified AWS service",
                "accessLevel": "Read"
            },
            "ListRequestedServiceQuotaChangeHistory": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html",
                "description": "Grants permission to request a list of the changes to quotas for a service",
                "accessLevel": "Read"
            },
            "ListRequestedServiceQuotaChangeHistoryByQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html",
                "description": "Grants permission to request a list of the changes to specific service quotas",
                "accessLevel": "Read"
            },
            "ListServiceQuotaIncreaseRequestsInTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate",
                "description": "Grants permission to return a list of the service quota increase requests from the service quota template",
                "accessLevel": "Read"
            },
            "ListServiceQuotas": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html",
                "description": "Grants permission to list all service quotas for the specified AWS service, in that account, in that Region",
                "accessLevel": "Read"
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html",
                "description": "Grants permission to list the AWS services available in Service Quotas",
                "accessLevel": "Read"
            },
            "PutServiceQuotaIncreaseRequestIntoTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html",
                "description": "Grants permission to define and add a quota to the service quota template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "quota": {
                        "required": false
                    }
                },
                "conditions": [
                    "servicequotas:service"
                ]
            },
            "RequestServiceQuotaIncrease": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html",
                "description": "Grants permission to submit the request for a service quota increase",
                "accessLevel": "Write",
                "resourceTypes": {
                    "quota": {
                        "required": false
                    }
                },
                "conditions": [
                    "servicequotas:service"
                ]
            }
        };
        this.resourceTypes = {
            "quota": {
                "name": "quota",
                "url": "https://docs.aws.amazon.com/servicequotas/latest/userguide/quota-as-resource.html",
                "arn": "arn:${Partition}:servicequotas:${Region}:${Account}:${ServiceCode}/${QuotaCode}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate the Service Quotas template with your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html
     */
    toAssociateServiceQuotaTemplate() {
        this.add('servicequotas:AssociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to remove the specified service quota from the service quota template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html
     */
    toDeleteServiceQuotaIncreaseRequestFromTemplate() {
        this.add('servicequotas:DeleteServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to disassociate the Service Quotas template from your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html
     */
    toDisassociateServiceQuotaTemplate() {
        this.add('servicequotas:DisassociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the AWS default value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html
     */
    toGetAWSDefaultServiceQuota() {
        this.add('servicequotas:GetAWSDefaultServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html
     */
    toGetAssociationForServiceQuotaTemplate() {
        this.add('servicequotas:GetAssociationForServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a particular service quota increase request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html
     */
    toGetRequestedServiceQuotaChange() {
        this.add('servicequotas:GetRequestedServiceQuotaChange');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the applied value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html
     */
    toGetServiceQuota() {
        this.add('servicequotas:GetServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a service quota increase request from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html
     */
    toGetServiceQuotaIncreaseRequestFromTemplate() {
        this.add('servicequotas:GetServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to list all default service quotas for the specified AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html
     */
    toListAWSDefaultServiceQuotas() {
        this.add('servicequotas:ListAWSDefaultServiceQuotas');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to quotas for a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html
     */
    toListRequestedServiceQuotaChangeHistory() {
        this.add('servicequotas:ListRequestedServiceQuotaChangeHistory');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to specific service quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html
     */
    toListRequestedServiceQuotaChangeHistoryByQuota() {
        this.add('servicequotas:ListRequestedServiceQuotaChangeHistoryByQuota');
        return this;
    }
    /**
     * Grants permission to return a list of the service quota increase requests from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate
     */
    toListServiceQuotaIncreaseRequestsInTemplate() {
        this.add('servicequotas:ListServiceQuotaIncreaseRequestsInTemplate');
        return this;
    }
    /**
     * Grants permission to list all service quotas for the specified AWS service, in that account, in that Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html
     */
    toListServiceQuotas() {
        this.add('servicequotas:ListServiceQuotas');
        return this;
    }
    /**
     * Grants permission to list the AWS services available in Service Quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html
     */
    toListServices() {
        this.add('servicequotas:ListServices');
        return this;
    }
    /**
     * Grants permission to define and add a quota to the service quota template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html
     */
    toPutServiceQuotaIncreaseRequestIntoTemplate() {
        this.add('servicequotas:PutServiceQuotaIncreaseRequestIntoTemplate');
        return this;
    }
    /**
     * Grants permission to submit the request for a service quota increase
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html
     */
    toRequestServiceQuotaIncrease() {
        this.add('servicequotas:RequestServiceQuotaIncrease');
        return this;
    }
    /**
     * Adds a resource of type quota to the statement
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/quota-as-resource.html
     *
     * @param serviceCode - Identifier for the serviceCode.
     * @param quotaCode - Identifier for the quotaCode.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQuota(serviceCode, quotaCode, account, region, partition) {
        var arn = 'arn:${Partition}:servicequotas:${Region}:${Account}:${ServiceCode}/${QuotaCode}';
        arn = arn.replace('${ServiceCode}', serviceCode);
        arn = arn.replace('${QuotaCode}', quotaCode);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters or restricts access to a specified AWS service
     *
     * Applies to actions:
     * - .toPutServiceQuotaIncreaseRequestIntoTemplate()
     * - .toRequestServiceQuotaIncrease()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`servicequotas:service`, value, operator || 'StringLike');
    }
}
exports.Servicequotas = Servicequotas;
//# sourceMappingURL=data:application/json;base64,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