"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.conditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.conditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.conditions).forEach((operator) => {
                Object.keys(this.conditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.conditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.conditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (typeof operator === 'undefined') {
            operator = 'StringLike';
        }
        if (!(operator in this.conditions)) {
            this.conditions[operator] = {};
        }
        this.conditions[operator][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || 'ForAnyValue:StringEquals');
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || 'NumericLessThan');
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if('aws:MultiFactorAuthPresent', value, 'Bool');
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnEquals`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || 'ArnEquals');
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if('aws:SecureTransport', value, 'Bool');
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnEquals`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || 'ArnEquals');
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || 'IpAddress');
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator || 'DateGreaterThanEquals');
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if('aws:ViaAWSService', value, 'Bool');
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || 'IpAddress');
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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