"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.actionList = {};
        this.useNotActions = false;
        this.actions = [];
        this.cdkActionsApplied = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            statement[mode] = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            const uniqueActions = this.actions.filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * @param action Actions that will be added to the statement.
     */
    add(action) {
        this.actions.push(action);
        return this;
    }
    /**
     * Adds actions to the statement, matching an `AccessLevel` or regular expression.
     *
     * When no value is passed, all actions of the service will be added.
     */
    allActions(...rules) {
        if (rules.length) {
            rules.forEach((rule) => {
                for (const [name, action] of Object.entries(this.actionList)) {
                    if (Object.values(access_level_1.AccessLevel).includes(rule)) {
                        if (rule == action.accessLevel) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                    else {
                        // assume it's a RegEx
                        if (name.match(RegexParser(rule))) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                }
            });
        }
        else {
            this.add(`${this.servicePrefix}:*`);
        }
        return this;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
//# sourceMappingURL=data:application/json;base64,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