# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['svg_ultralight',
 'svg_ultralight.bounding_boxes',
 'svg_ultralight.constructors',
 'svg_ultralight.strings']

package_data = \
{'': ['*']}

install_requires = \
['lxml>=4.9.2,<5.0.0', 'paragraphs>=0.2.0,<0.3.0']

setup_kwargs = {
    'name': 'svg-ultralight',
    'version': '0.15.0',
    'description': 'a sensible way to create svg files with Python',
    'long_description': '# svg_ultralight\n\nThe most straightforward way to create SVG files with Python.\n\n## Four principal functions:\n\n    from svg_ultralight import new_svg_root, write_svg, write_png_from_svg, write_png\n\n## One convenience:\n\n    from svg_ultralight import NSMAP\n\n### new_svg_root\n    x_: Optional[float],\n    y_: Optional[float],\n    width_: Optional[float],\n    height_: Optional[float],\n    pad_: float = 0\n    dpu_: float = 1\n    nsmap: Optional[Dict[str, str]] = None (svg_ultralight.NSMAP if None)\n    **attributes: Union[float, str],\n    -> etree.Element\n\nCreate an svg root element from viewBox style arguments and provide the necessary svg-specific attributes and namespaces. This is your window onto the scene.\n\nThree ways to call:\n\n1. The trailing-underscore arguments are the same you\'d use to create a `rect` element (plus `pad_` and `dpu_`).  `new_svg_root` will infer `viewBox`, `width`, and `height` svg attributes from these values.\n2. Use the svg attributes you already know: `viewBox`, `width`, `height`, etc. These will be written to the xml file.\n3. Of course, you can combine 1. and 2. if you know what you\'re doing.\n\nSee `namespaces` below.\n\n* `x_`: x value in upper-left corner\n* `y_`: y value in upper-left corner\n* `width_`: width of viewBox\n* `height_`: height of viewBox\n* `pad_`: the one small convenience I\'ve provided. Optionally increase viewBox by `pad` in all directions.\n* `dpu_`: pixels per viewBox unit for output png images.\n* `nsmap`: namespaces. (defaults to svg_ultralight.NSMAP). Available as an argument should you wish to add additional namespaces. To do this, add items to NSMAP then call with `nsmap=NSMAP`.\n* `**attributes`: the trailing-underscore arguments are an *optional* shortcut for creating a scene. The entire svg interface is available to you through kwargs. See `A few helpers` below for details on attribute-name translation between Python and xml (the short version: `this_name` becomes `this-name` and `this_` becomes `this`)\n\n### namespaces (svg_ultralight.NSMAP)\n\n`new_svg_root` will create a root with several available namespaces.\n\n* `"dc": "http://purl.org/dc/elements/1.1/"`\n* `"cc": "http://creativecommons.org/ns#"`\n* `"rdf": "http://www.w3.org/1999/02/22-rdf-syntax-ns#"`\n* `"svg": "http://www.w3.org/2000/svg"`\n* `"xlink": "http://www.w3.org/1999/xlink"`\n* `"sodipodi": "http://sodipodi.sourceforge.net/DTD/sodipodi-0.dtd"`\n* `"inkscape": "http://www.inkscape.org/namespaces/inkscape"`\n\nI have made these available to you as `svg_ultralight.NSMAP`\n\n### write_svg\n    svg: str,\n    xml: etree.Element,\n    stylesheet: Optional[str] = None,\n    do_link_css: bool = False,\n    **tostring_kwargs,\n    -> str:\n\nWrite an xml element as an svg file. This will link or inline your css code and insert the necessary declaration, doctype, and processing instructions.\n\n* `svg`: path to output file (include extension .svg)\n* `param xml`: root node of your svg geometry (created by `new_svg_root`)\n* `stylesheet`: optional path to a css stylesheet\n* `do_link_css`: link to stylesheet, else (default) write contents of stylesheet into svg (ignored if `stylesheet` is None). If you have a stylesheet somewhere, the default action is to dump the entire contents into your svg file. Linking to the stylesheet is more elegant, but inlining *always* works.\n* `**tostring_kwargs`: optional kwarg arguments for `lxml.etree.tostring`. Passing `xml_declaration=True` by itself will create an xml declaration with encoding set to UTF-8 and an svg DOCTYPE. These defaults can be overridden with keyword arguments `encoding` and `doctype`. If you don\'t know what this is, you can probably get away without it.\n* `returns`: for convenience, returns svg filename (`svg`)\n* `effects`: creates svg file at `svg`\n\n### write_png_from_svg\n\n    inkscape_exe: str,\n    svg: str\n    png: Optional[str]\n    -> str\n\nConvert an svg file to a png. Python does not have a library for this. That has an upside, as any library would be one more set of svg implementation idiosyncrasies we\'d have to deal with. Inkscape will convert the file. This function provides the necessary command-line arguments.\n\n* `inkscape_exe`: path to inkscape.exe\n* `svg`: path to svg file\n* `png`: optional path to png output (if not given, png name will be inferred from `svg`: `\'name.svg\'` becomes `\'name.png\'`)\n* `return`: png filename\n* `effects`: creates png file at `png` (or infers png path and filename from `svg`)\n\n### write_png\n\n    inkscape_exe: str,\n    png: str,\n    xml: etree.Element,\n    stylesheet: Optional[str] = None\n    -> str\n\nCreate a png without writing an initial svg to your filesystem. This is not faster (it may be slightly slower), but it may be important when writing many images (animation frames) to your filesystem.\n\n* `inkscape_exe`: path to inkscape.exe\n* `png`: path to output file (include extension .png)\n* `param xml`: root node of your svg geometry (created by `new_svg_root`)\n* `stylesheet`: optional path to a css stylesheet\n* `returns`: for convenience, returns png filename (`png`)\n* `effects`: creates png file at `png`\n\n## A few helpers:\n\n    from svg_ultralight.constructors import new_element, new_sub_element\n\nI do want to keep this ultralight and avoid creating some pseudo scripting language between Python and lxml, but here are two very simple, very optional functions to save your having to `str()` every argument to `etree.Element`.\n\n### constructors.new_element\n\n    tag: str\n    **params: Union[str, float]\n    -> etree.Element\n\nPython allows underscores in variable names; xml uses dashes.\n\nPython understands numbers; xml wants strings.\n\nThis is a convenience function to swap `"_"` for `"-"` and `10.2` for `"10.2"` before creating an xml element.\n\nTranslates numbers to strings\n\n    >>> elem = new_element(\'line\', x1=0, y1=0, x2=5, y2=5)\n    >>> etree.tostring(elem)\n    b\'<line x1="0" y1="0" x2="5" y2="5"/>\'\n\nTranslates underscores to hyphens\n\n    >>> elem = new_element(\'line\', stroke_width=1)\n    >>> etree.tostring(elem)\n    b\'<line stroke-width="1"/>\'\n\nRemoves trailing underscores. You\'ll almost certainly want to use reserved names like ``class`` as svg parameters. This\ncan be done by passing the name with a trailing underscore.\n\n    >>> elem = new_element(\'line\', class_=\'thick_line\')\n    >>> etree.tostring(elem)\n    b\'<line class="thick_line"/>\'\n\nSpecial handling for a \'text\' argument. Places value between element tags.\n\n    >>> elem = new_element(\'text\', text=\'please star my project\')\n    >>> etree.tostring(elem)\n    b\'<text>please star my project</text>\'\n\n### constructors.new_sub_element\n\n    parent: etree.Element\n    tag: str\n    **params: Union[str, float]\n    -> etree.Element\n\nAs above, but creates a subelement.\n\n    >>> parent = etree.Element(\'g\')\n    >>> _ = new_sub_element(\'rect\')\n    >>> etree.tostring(parent)\n    b\'<g><rect/></g>\'\n\n### update_element\n\nAnother way to add params through the new_element name / float translator. Again unnecessary, but potentially helpful. Easily understood from the code or docstrings.\n\n## Extras:\n\n### query.map_ids_to_bounding_boxes\n\nPython cannot parse an svg file. Python can *create* an svg file, and Inkscape can parse (and inspect) it. Inkscape has a command-line interface capable of reading an svg file and returning some limited information. This is the only way I know for a Python program to:\n\n1. create an svg file (optionally without writing to filesystem)\n2. query the svg file for bounding-box information\n3. create an adjusted svg file.\n\nThis would be necessary for, e.g., algorithmically fitting text in a box.\n\n    from svg_ultralight.queries import map_ids_to_bounding_boxes\n\nYou can get a tiny bit more sophisticated with Inkscape bounding-box queries, but not much. This will give you pretty much all you can get out of it.\n\n### animate.write_gif\n\nCreate an animated gif from a sequence of png filenames. This is a Pillow one-liner, but it\'s convenient for me to have it, so it might be convenient for you. Requires pillow, which is not a project dependency.\n\n    from svg_ultralight.animate import write_gif\n\n[Full Documentation and Tutorial](https://shayallenhill.com/svg-with-css-in-python/)\n',
    'author': 'Shay Hill',
    'author_email': 'shay_public@hotmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
