"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureLambda = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
const lambdaTimeout = cdk.Duration.seconds(10);
function ensureLambda(scope) {
    const stack = cdk.Stack.of(scope);
    const lambdaName = 'AthenaManager';
    const ID = 'CFN::Resource::Custom::Athena';
    const existing = stack.node.tryFindChild(lambdaName);
    if (existing) {
        return existing;
    }
    const policy = new iam.ManagedPolicy(stack, 'Athena-Manager-Policy', {
        managedPolicyName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        statements: [
            new iam.PolicyStatement({
                actions: ['athena:GetWorkGroup'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['athena:CreateWorkGroup', 'athena:TagResource'],
                resources: ['*'],
                conditions: {
                    StringLike: {
                        'aws:RequestTag/CreatedByCfnCustomResource': `${ID}-WorkGroup`,
                    },
                },
            }),
            new iam.PolicyStatement({
                actions: [
                    'athena:DeleteWorkGroup',
                    'athena:TagResource',
                    'athena:UntagResource',
                    'athena:UpdateWorkGroup',
                ],
                resources: ['*'],
                conditions: {
                    StringLike: {
                        'aws:ResourceTag/CreatedByCfnCustomResource': `${ID}-WorkGroup`,
                    },
                },
            }),
            new iam.PolicyStatement({
                actions: [
                    'athena:CreateNamedQuery',
                    'athena:DeleteNamedQuery',
                    'athena:GetNamedQuery',
                    'athena:ListNamedQueries',
                ],
                resources: ['*'],
            }),
        ],
    });
    const role = new iam.Role(stack, 'Athena-Manager-Role', {
        roleName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        managedPolicies: [
            policy,
            iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
        ],
    });
    const fn = new lambda.Function(stack, lambdaName, {
        functionName: `${stack.stackName}-${lambdaName}`,
        role: role,
        description: 'Custom CFN resource: Manage Athena resources',
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
        timeout: lambdaTimeout,
    });
    return fn;
}
exports.ensureLambda = ensureLambda;
//# sourceMappingURL=data:application/json;base64,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