# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['zum', 'zum.configs', 'zum.requests']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.16.1,<0.17.0', 'tomlkit>=0.7.0,<0.8.0']

entry_points = \
{'console_scripts': ['zum = zum.cli:dispatcher']}

setup_kwargs = {
    'name': 'zum',
    'version': '0.2.0',
    'description': 'Stop writing scripts to test your APIs. Call them as CLIs instead.',
    'long_description': '<p align="center">\n    <a href="https://github.com/daleal/zum">\n        <img src="https://zum.daleal.dev/assets/zum-250x250.png">\n    </a>\n</p>\n\n<h1 align="center">Zum</h1>\n\n<p align="center">\n    <em>\n        Stop writing scripts to test your APIs. Call them as CLIs instead.\n    </em>\n</p>\n\n<p align="center">\n<a href="https://pypi.org/project/zum" target="_blank">\n    <img src="https://img.shields.io/pypi/v/zum?label=version&logo=python&logoColor=%23fff&color=306998" alt="PyPI - Version">\n</a>\n\n<a href="https://github.com/daleal/zum/actions?query=workflow%3Atests" target="_blank">\n    <img src="https://img.shields.io/github/workflow/status/daleal/zum/tests?label=tests&logo=python&logoColor=%23fff" alt="Tests">\n</a>\n\n<a href="https://codecov.io/gh/daleal/zum" target="_blank">\n    <img src="https://img.shields.io/codecov/c/gh/daleal/zum?label=coverage&logo=codecov&logoColor=ffffff" alt="Coverage">\n</a>\n\n<a href="https://github.com/daleal/zum/actions?query=workflow%3Alinters" target="_blank">\n    <img src="https://img.shields.io/github/workflow/status/daleal/zum/linters?label=linters&logo=github" alt="Linters">\n</a>\n</p>\n\n**Zum** (German word roughly meaning "_to the_" or "_to_" depending on the context, pronounced `/tsʊm/`) is a tool that lets you describe an API using a [TOML](https://toml.io/en/) file and then call that API using your command line. This means that **the days of writing custom scripts to help you develop each of your APIs are over**. Just create a `zum.toml`, describe your API and forget about maintaining more code!\n\n## Installing\n\nInstall using pip!\n\n```sh\npip install zum\n```\n\n## Usage\n\n### The config file\n\nThe first thing that you need to do in order to use `zum` is to describe the API that you\'re trying to ping using a config file, named `zum.toml`. This TOML file should contain the following keys:\n\n#### `metadata`\n\nThe `metadata` key contains everything that is not data about an endpoint of the API itself, but that is needed in order to query the API. This key should contain the following values:\n\n- `server`: The base URL where the API is hosted.\n\nAs an example, the first lines of your `zum.toml` file for a development environment should probably look similar to this:\n\n```toml\n[metadata]\nserver = "http://localhost:8000"\n```\n\nThis indicates to `zum` that the API endpoints are located at `http://localhost:8000`. Easy enough, right?\n\n#### `endpoints`\n\nThe `endpoints` key contains every endpoint that you want to be able to access from `zum`. Each endpoint should also have a `route` value, a `method` value and may include a `params` value and a `body` value. Let\'s see an example:\n\n```toml\n[endpoints.my-endpoint-name]\nroute = "/endpoint-name"\nmethod = "post"\n```\n\nNotice that the header of the section consists of `endpoints.{something}`. **That `{something}` will be the name of your endpoint**. That means that, on the example, to query the endpoint, all you need to do is to run:\n\n```sh\nzum my-endpoint-name\n```\n\nWith the existing configuration, `zum` will make a `POST` HTTP request to `http://localhost:8000/endpoint-name`. Just 5 lines on a TOML file!\n\nThe endpoint configuration will be discussed more on a [dedicated section](#endpoints).\n\n### Endpoints\n\nUp until now, the examples shown are **really** simple. Rarely does an API endpoint not include some parameters on the URL or some request body. The idea of `zum` is to keep everything extremely simple, so let\'s see how to use URL parameters and the request body.\n\n#### Parameters\n\nImagine that you have an API with an endpoint `/entity/:id` that returns an entity with the id `:id`. You would probably like to be able to just `zum entity 3` to get the entity with `:id` corresponding to `3`. Well, that\'s **exactly** what `zum` does. Let\'s first define the endpoint:\n\n```toml\n[endpoints.my-entity]\nroute = "/entity/{id}"\nmethod = "get"\nparams = ["id"]\n```\n\nThis configuration tells `zum` that it should replace the route string `{id}` for whatever `id` it receives from the command line as an argument (the command for that configuration should be `zum my-entity 3`). That makes sense. But why is the `params` value an array? Well, let\'s imagine that you want to search for all the appearances of some string on the entity model. The API endpoint will probably receive a `?query=` parameter. So let\'s describe this new endpoint:\n\n```toml\n[endpoints.search]\nroute = "/entity/{id}?query={query}"\nmethod = "get"\nparams = ["id", "query"]\n```\n\nNow, you can run something like:\n\n```sh\nzum search 3 mystring\n```\n\nThis will search `"mystring"` on the entity with id `3`. **But order matters**. Let\'s imagine that to you, it makes more sense to write the query string first. Then, your definition should be:\n\n```toml\n[endpoints.search]\nroute = "/entity/{id}?query={query}"\nmethod = "get"\nparams = ["query", "id"]\n```\n\nNow, you can run something like:\n\n```sh\nzum search mystring 3\n```\n\nThe query will be exactly the same. **This means that the array tells `zum` in which order to interpret the CLI parameters**.\n\n#### Headers\n\nJust like the parameters, the headers get defined as an array:\n\n```toml\n[endpoints.restricted]\nroute = "/secret"\nmethod = "get"\nheaders = ["Authorization"]\n```\n\nTo run this endpoint, you just need to run:\n\n```sh\nzum restricted "Bearer super-secret-token"\n```\n\nThis will send a `GET` request to `http://localhost:8000/entity` with the following headers:\n\n```json\n{\n    "Authorization": "Bearer super-secret-token"\n}\n```\n\n⚠ **WARNING**: Notice that, for the first time, we surrounded something with quotes. The reason we did this is that, without the quotes, the console has no way of knowing if you want to pass a parameter with a space in the middle or if you want to pass multiple parameters, so it defaults to receiving the words as multiple parameters. To stop this from happening, you can surround the string in quotes, and now the whole string will be interpreted as only one parameter with the space in the middle of the string. This will be handy on future examples, so **keep it in mind**.\n\n#### Request body\n\nJust like the parameters and headers, the request body gets defined as an array:\n\n```toml\n[endpoints.create-entity]\nroute = "/entity"\nmethod = "post"\nbody = ["name", "city"]\n```\n\nTo run this endpoint, you just need to run:\n\n```sh\nzum create-entity dani Santiago\n```\n\nThis will send a `POST` request to `http://localhost:8000/entity` with the following request body:\n\n```json\n{\n    "name": "dani",\n    "city": "Santiago"\n}\n```\n\nAs always, order matters.\n\n```toml\n[endpoints.create-entity]\nroute = "/entity"\nmethod = "post"\nbody = ["city", "name"]\n```\n\nNow, to get the same result as before, you should run:\n\n```sh\nzum create-entity Santiago dani\n```\n\n**But what about types?** On the examples above, all the request body parameters are being sent **as strings**. Of course, you can cast the values to some types. The supported types are:\n\n- `string`\n- `integer`\n- `float`\n- `boolean` (`true` or `false`)\n- `null`\n\nTo declare a type, let\'s imagine you have an API endpoint that receives two numbers `number1` and `number2` and adds them together. To describe this endpoint, you can use the following description:\n\n```toml\n[endpoints.add]\nroute = "/add"\nmethod = "post"\nbody = [\n    { name = "number1", type = "integer" },\n    { name = "number2", type = "integer" }\n]\n```\n\nNow, when you run\n\n```sh\nzum add 5 8\n```\n\nThe request body will be:\n\n```json\n{\n    "number1": 5,\n    "number2": 8\n}\n```\n\nKeep in mind that you can _mix and match_ the definitions. You can even define the parameter with the object notation and not include its type. You could define, for example:\n\n```toml\n[endpoints.example]\nroute = "/example"\nmethod = "post"\nbody = [\n    "parameter1",\n    { name = "parameter2", type = "boolean" },\n    { name = "parameter3" }\n]\n```\n\nNow, `parameter1` will be sent as a string, `parameter2` will be casted as a boolean and `parameter3` will be sent as a string.\n\n#### What about using all of the above?\n\nOf course, sometimes you need to use parameters, headers **and** request bodies. For example, if you wanted to create a nested entity, you would need to use the parent\'s id as a parameter, the authorization headers and the new entity data as a request body. Let\'s describe this situation!\n\n```toml\n[endpoints.create-nested]\nroute = "/entity/{id}"\nmethod = "post"\nparams = ["id"]\nheaders = ["Authorization"]\nbody = ["name", "city"]\n```\n\nNow, you can call the endpoint using:\n\n```sh\nzum create-nested 69 "Bearer super-secret-token" dani Santiago\n```\n\nThis will call `POST /entity/69` with the following headers:\n\n```json\n{\n    "Authorization": "Bearer super-secret-token"\n}\n```\n\nAnd the following request body:\n\n```json\n{\n    "name": "dani",\n    "city": "Santiago"\n}\n```\n\nAs you can probably tell, `zum` receives the `params` first on the CLI, then the `headers` and then the `body`. In _pythonic_ terms, what `zum` does is kind of _unpacks_ the three arrays consecutively, something like the following:\n\n```py\narguments = [*params, *headers, *body]\nzum(arguments)\n```\n\n## `zum.toml` example\n\nHere\'s a simple `zum.toml` file example:\n\n```toml\n[metadata]\nserver = "http://localhost:8000"\n\n[endpoints.my-entity]\nroute = "/entity/{id}"\nmethod = "get"\nparams = ["id"]\n\n[endpoints.search]\nroute = "/entity/{id}?query={query}"\nmethod = "get"\nparams = ["id", "query"]\n\n[endpoints.create-entity]\nroute = "/entity"\nmethod = "post"\nheaders = ["Authorization"]\nbody = ["name", "city"]\n\n[endpoints.create-nested]\nroute = "/entity/{id}"\nmethod = "post"\nparams = ["id"]\nheaders = ["Authorization"]\nbody = ["name", "city"]\n```\n\nWith that config file (using a hypothetical existing API), you could `GET /entity/420` to get the entity with id `420`, `GET /entity/420?query=nice` to search for the appearances of the word `nice` on the model of the entity with id `420`, `POST /entity` with an authorization header and some request body to create a new entity and `POST /entity/69` with an authorization header and some request body to create a new nested entity, child of the entity with id `69`.\n\n## Developing\n\nClone the repository:\n\n```sh\ngit clone https://github.com/daleal/zum.git\n\ncd zum\n```\n\nRecreate environment:\n\n```sh\nmake get-poetry\nmake venv-with-dependencies\n```\n\nRun the linters:\n\n```sh\nmake black flake8 isort mypy pylint\n```\n\nRun the tests:\n\n```sh\nmake tests\n```\n\n## Resources\n\n- [Issue Tracker](https://github.com/daleal/zum/issues/)\n',
    'author': 'Daniel Leal',
    'author_email': 'dlleal@uc.cl',
    'maintainer': 'Daniel Leal',
    'maintainer_email': 'dlleal@uc.cl',
    'url': 'https://github.com/daleal/zum',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
