#!/usr/bin/env python

import pprint

from click.testing import CliRunner
import pytest
import six
import wandb
import numpy as np
from wandb import tensorflow as wandb_tensorflow
from wandb.keras import WandbCallback

# Tests which rely on row history in memory should set `History.keep_rows = True`
from wandb.history import History
History.keep_rows = True

import tensorflow as tf


@pytest.fixture
def history():
    with CliRunner().isolated_filesystem():
        wandb.run = wandb.wandb_run.Run.from_environment_or_defaults()
        wandb.util.mkdir_exists_ok(wandb.run.dir)
        yield wandb.run.history
        wandb.run = None


def test_tf_summary():
    summary = wandb_tensorflow.tf_summary_to_dict(SUMMARY_PB)
    images = summary['input_reshape_input_image']
    del summary['input_reshape_input_image']
    histo_keys = ['layer1/activations', 'layer1/biases/summaries/histogram',
                  'layer1/weights/summaries/histogram', 'layer2/Wx_plus_b/pre_activations',
                  'layer2/activations', 'layer2/biases/summaries/histogram',
                  'layer2/weights/summaries/histogram', 'layer1/Wx_plus_b/pre_activations']
    histos = [summary[k] for k in histo_keys]
    for k in histo_keys:
        del summary[k]

    pprint.pprint(summary)

    assert len(images) == 10
    assert all(isinstance(img, wandb.Image) for img in images)
    assert all(isinstance(hist, wandb.Histogram) for hist in histos)
    assert summary == {
        'accuracy_1': 0.8799999952316284,
        'cross_entropy_1': 0.37727174162864685,
        'dropout/dropout_keep_probability': 0.8999999761581421,
        'layer1/biases/summaries/max': 0.12949132919311523,
        'layer1/biases/summaries/mean': 0.10085226595401764,
        'layer1/biases/summaries/min': 0.0768924281001091,
        'layer1/biases/summaries/stddev_1': 0.01017912570387125,
        'layer1/weights/summaries/max': 0.22247056663036346,
        'layer1/weights/summaries/mean': 0.00014527945313602686,
        'layer1/weights/summaries/min': -0.22323597967624664,
        'layer1/weights/summaries/stddev_1': 0.08832632750272751,
        'layer2/biases/summaries/max': 0.11211398988962173,
        'layer2/biases/summaries/mean': 0.09975100308656693,
        'layer2/biases/summaries/min': 0.0904880091547966,
        'layer2/biases/summaries/stddev_1': 0.006791393272578716,
        'layer2/weights/summaries/max': 0.21537037193775177,
        'layer2/weights/summaries/mean': -0.0023455708287656307,
        'layer2/weights/summaries/min': -0.22206202149391174,
        'layer2/weights/summaries/stddev_1': 0.08973880857229233
    }


def test_tf_summary_in_history(history):
    history.log_tf_summary(SUMMARY_PB)
    summary = history.rows[0]
    pprint.pprint(summary)

    # We don't bother checking anything here because the auto-added
    # History keys get in the way. And the images wind up being a
    # dict (which may be a bug)

@pytest.mark.skip(reason="No one is using the hook")
def test_hook(history):
    g1 = tf.Graph()
    with g1.as_default():
        c1 = tf.constant(42)
        tf.summary.scalar('c1', c1)
        summary_op = tf.summary.merge_all()

        hook = wandb_tensorflow.WandbHook(summary_op, history=history)
        with tf.train.MonitoredTrainingSession(hooks=[hook]) as sess:
            summary, acc = sess.run([summary_op, c1])

    assert wandb_tensorflow.tf_summary_to_dict(summary) == {'c1': 42.0}
    print(history.rows)
    # TODO(adrian): there is still some kind of bug here where the history
    # is being shared with another test that manages to add rows before this one.
    # we just grab the last row for now
    assert history.rows[-1]['c1'] == 42.0


SUMMARY_PB = six.b('\n\xc3\x02\n\x1binput_reshape/input/image/0"\xa3\x02\x08\x1c\x10\x1c\x18\x01"\x9a\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xe1IDAT(\x91\xdd\x90\xafJC\x01\x14\xc6\xbf]V\xaf\x0b\x0b\xc3\xe2+L0Y4O\x04\x8bhQ\xf0\x11\x06f}\x82%Al\x06\x17l"\xc2\xd2\xc0\x95aZ0O\xb0\x98\x07\xb2\xa1\xe9\xf7\xddk\xd0]\xbdp\xee\x0bx\xc2\xf9\xf7\xe3\xe3|\x1c\xe9\xdfD\xad\xe86\xb6\xd48K2IIvq5-\xe0A\xae\xe3v\xba"\xa9\x96\x7f\xa7\xd7\xdd\xe9Rd\x80r\xeaJR\xfd\xef\x8d\xd9g\x92IZ[\xce\xbf\xf0T\x1a\xbcH\xd2\xde]\xd9\xd6y\xaf]\xf4G\xce\xed\x9b\xd8~s\x8ca?f\x0f`\x9e\x9b\x11K\x07\x80\xe7\'\xa1p\x04\xe0\xdb\x90m\xe7\xb6=iD\xac\xf3n f\xe9\x10\xc3\xd3j\xc46\x87`>v\xc2\x83}\xc0\x93\xc3\x90\xddW{\xd1\xba\x01b\xd6z3@y\x99\xfc\xd4\xc5\xa3\xa4\xd9e(\x94\xd4\xbd\xaexwE|\x01\xf6\x1eu\x18\x9b\xb7xP\x00\x00\x00\x00IEND\xaeB`\x82\n\xcc\x02\n\x1binput_reshape/input/image/1"\xac\x02\x08\x1c\x10\x1c\x18\x01"\xa3\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xeaIDAT(\x91c`\xa0\x07\x88\xd8\xfeI\x01\x97\\\xeb\xbf\x7f\xff\xf4\xb1K\t\xf5\xfe\xbb\xb1\xf7\x99\x10v\xc9\xce\x7f\xf7\x85|\xafc\x97\x0b\xfe\xfb/\x85A\xd1\x13\xab\x9c\xe0\xeb\x7f\xdb\x98q9&\xf2\xdf\x1d\x114!&8+\x84a\xe9\x1b\\\x1a\x19>\xdd\xe1B\x17\x82\xeb\xb4\xe7~\xf8\r\xa7$7#\xa6a,h|]\x87\xbf\xe6\xfa?z\xd6\xa0\x8az\xfd\xdb\xcb\xc0\xa0\xbb\xe8\xfb\xbf\x7f\xff>~\xff\xdd\x86!\xc9^\xf3\xe3\xdf\xc7~\x0b\x0by\xa3G\x1fQ%]\xfe\xbe\xbf\xf8\xefs\xaf\x1c\x03\x03\x03\x03\xc3>4I\x865\xff\xfe\x9d\x83H1p\x1c\xde\x88&\xe9\xf6\xef\xdf&&\x06\x06\x06\x06&\xdb\xeb?\xe2\xd1$E\xce\xff\xfb\xd7\x1f\xc0i\x92\xb3\xef\xdf\xbf^\x0c_\x89l\xfc\xfd\xff\xdf\xbf\x7f\xff\xff\x9ev\x87\x8a\xa0x]8\xd0\x85\xff\xff\x9a\xd3\x971\xf4\r\x16\x00\x00X\xb5X\x95I\x94\xe7\xcf\x00\x00\x00\x00IEND\xaeB`\x82\n\xe3\x02\n\x1binput_reshape/input/image/2"\xc3\x02\x08\x1c\x10\x1c\x18\x01"\xba\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x01\x01IDAT(\x91\xbd\xd01H\x02a\x14\xc0\xf1wj\xd8b\xb544\xda\x1e\xba\xd4\x12\rFD\x04\x81\x0eA\xe0\xae\x93\x88\x93C-\x05\x11\xb9\x14\x14MMm79IC-\xea GD\x9b\xc3\xb9\x95\x87\x8b\x97\xd0\xe0\xb9\xfd\x1b\x8e\xa3\xee\xfbn\xf6M\xef}\xbf\xef{\x1f\xef\x89\xcc?\xe2\xa1\xca8|\xf3\xac\xe8\x8bF\xf6\x06&\x17+Q\xb6d\x02\x80[\xd1m\xb5\x0f\x80\xd3\xc4\xddR-\xd9\x01`|\xb0\xdc\xa3\x1dSp\x13\x80aN$\x0f\'\n^\x03\x0c7Dd\x1b\x9e\xfd\xb3D\x80\x19\x11\xb1\xf7\xbeB\x0f\xfew\xff\xd8\r\xdb\x1fZ\xdf\xe6\x8e\x13\x14?\xca\x9f\xf1\xb5 +\xc1\xb1:\x8b\x88$\x8b\xf7\xb6}\xeb0X\xd4m\xbd\xe5\xaf\x08S\xa3\x85\xf2\x04>\xabw\xc0\x83f\x970\xad\xa5\xb2#\xe0E\xb1\xc4\x13x\x05\xc9\x8c\xa0;S1\x05^>}6\x85\xc7X\xe3JG\xf7t\x06\x9c\xabK\xf7\x11\xc0:\x8a01\xf6\xdf\xe1\xb5\x1eE\xf3\x8d_t\x1b\x84\xb4\nt/\xd1\x00\x00\x00\x00IEND\xaeB`\x82\n\xbf\x02\n\x1binput_reshape/input/image/3"\x9f\x02\x08\x1c\x10\x1c\x18\x01"\x96\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xddIDAT(\x91c`\xa0\x0e\xb0y\x12\x86\xcceB\x91\xf4\x92\xfa\x88S\xa3\xe6\xb1\xdd,8%\xf7\xffK\xc3)\x97\xf9w\x1e3N\xc9\xfeO\x8a8\xe5\xf8\x1e\xcdB\x13Arm\x9c\xccm\xdc\x92\xea\xdf\xd6\xe3\x96\xe4_v\x87\x81\x81A\xb4\xd0\x13\x8b\x95\x8a_\n\x18\x18D\x12^\xfc{\xac\x8d)\xd9\xfe\xaf\x80!\xf6\xc5\xbf\x7f\xef\xef\xef\xc64\xf6\x0c\xc3\xb7\xfc\xd9b\x7f\xd6\xea\xaa\xff\x93\x83\n!\x82\xeb\xc6\xcf\xcf\x11l\x7f\x82\xb600\xdc\xd4y\x84n\xac\xc0\xfb\xb3\xff~\xf9000p\xad\x9a\x86i\xe9\x93\xc3\xffV3000W\xff+\xc1\x94\x9c\xfco\xa6"\x03\x03{\xd3\xbfWB\x18v2\xdcc\xf8\xfb\x84\xc1\xb8"\xf8\xb5\xe3;,>]\xf6\xf7\xda\xc6O\xff\xff\xe6c\x91b``\xd0=\xf5\xefJ\x92.v\xb9\xc1\x0f\x00HlJ3\xc5\x1a"\xc0\x00\x00\x00\x00IEND\xaeB`\x82\n\xec\x02\n\x1binput_reshape/input/image/4"\xcc\x02\x08\x1c\x10\x1c\x18\x01"\xc3\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x01\nIDAT(\x91c`\xa0\t`Db\x8b\xea\xfb0\xf8\x1e\x0bg\xb9\xe5\xfc\x14C\x9d\xea\x83\xbf\x7f\xff\xfd\xfd\xfb\xf7\xef\xdff\xa8\x08\x13B\xf2\xe7\xbc|4\xe5,\x08\xe6\xdb\xbb\x7f\x19\x18\x18\xde}~l\xc1\x85\xcd\x01\x93\x8f\x1d\xaf\xe4g`\xc8\xf9\xea\x88\xc7\x95\x13w\xe2\x96\xd3~y\x11\xc3ApP(\x82\xe9Z\x84N\x86{\xb8\x8d=\xfe\xd3\x02\xa7\x9c\xc2\xf3\x16LA;)\x06\x06\x06\x06\xe9\xdaC\xcf\x0c\xa1"\xb0@\x98i\xc9\xa0\xf1\xf2=\x03\xe3\x7fA)\x86\xda\xf3\xa8\xbaN\xfd\xfd{\xfd\xcb\xdf\xbf\x7f\xff\xff\xfd\xfb\xf7\xef\xf9\x00a$)\x8e\xd0w?\xef\x15|\xf8\x0b\x0b\xf8\xbf\xef\x96\x89\xc1%\x95\xfe\xfe\x9d`\xef\x06\x91\xdc\xb2\xe6\xef\xdf\xbf\x7f\xff*\xc2%\xc3\xff\xfe\xfd\xf4\xe9\xe7\xdf\xb3\x7f\xffn\x98$\xce$&\xe6}\xfc\x93<\\\xb2\xf6\xef\xdf\xbf\x7f\'\x7f\xf9\xfb\xf7o\x1b\xa6\'$\x8e\xfc]\xf2\xf8\xe1\xdf\xbf\xcf\x02\xb0\xf9\x9b3\xd0!n\xeb\xcf\x1aq\x9c\x013H\x00\x00u\xf6w\xea=\xf7\xee\x9c\x00\x00\x00\x00IEND\xaeB`\x82\n\xca\x02\n\x1binput_reshape/input/image/5"\xaa\x02\x08\x1c\x10\x1c\x18\x01"\xa1\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xe8IDAT(\x91\xcd\x90\xb1j\x02a\x10\x84\xe7~\x03\x92\x03\xc1^\x0b_ \x04\xd2\x84\x13\x82\xbd\xa4Lc\x15!e\x1aK\xc9\x0b\x84@,\xac\xd2\xa4\xc9+\xa4Hk\x91\\!\xfa\x00i$\xa0\x8d\n\xd7\xa4\x10\x84\xd9\x7f-T\xb8\xf3\xdc.E\xb6Z\xf6cvg\x07\xf8\xa3j\xa8|Y,|\x17\xce\xce\x8f\xb3\xd2+\x85\x9f\x860"\xe5\xb7m\xc0GRZ\x06\x9b\x8a\x88VR\x83\x93T\xaf\n\xdc\xcf\x8f\x0b\xbb+\xb2g,\xbd\x15r}g\xc0g!\xfb\x06+\r\xf2\xe1\xb8}\xd3\xb8\xf2\xaa\x86\xf0lIY4\x81\xe8#\x8e\xe3\xda\x01\x8cHy\xd9\xba"\xbf\x1f\x8a9XAy$$)\xbc\x04\x90\n!\x80\x03\xae/\x9c\x07\xe0|\x90\x81\xc9\xac\xea\x87\x93\x82z\x05\xe0\xa7I\xf6\xe8\x13w\x1bIJ\xfd\xc0Q\xf8\xb6\x87\xeb\x9f\x9b\xd3\xdc7\x9d\x84\xa4\x8c;V\x84\xff\xbd6\x0f\x96w\xf7\xa1\x97\r]\x00\x00\x00\x00IEND\xaeB`\x82\n\xbf\x02\n\x1binput_reshape/input/image/6"\x9f\x02\x08\x1c\x10\x1c\x18\x01"\x96\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xddIDAT(\x91\xd5\xd0/OBQ\x1c\xc6\xf1\x87{\xd9\xd8\xdc@Gc\x06\xe7F\x11#\xbe\x02\xd9|\x05\x8c\xcd\n\xc1F\xb29\x1a\xe56\x9b\x84;3A\xdf\x00\x05f"@\xe0O\xc1N\x91p\x19\x016\xc7W\tw\xdc\x8d\xeb9\xcd\xc2\xd3\xce\xf9\xec9\xbf\xdf\x8e\xf4_\xb9\xfb]\x16\xa3\x83\x13\xc3\x9b\x9ft\xddV\xec\xaeah\xb1\xf2\x066\x15\xb3e\x07@\xd3l\t\x0f\x98\xe5\xcd\x98\x01>/\xcdv\xda\x03\x1a\xb6m\x80\xf7\x0bK\xf1\x038\xb7\x14}\xe0-%\xc9\xc9\xed\xaf\x92\x11\xe6\xa5\x95\xe7\xde\xd6\xe4\\?\xbf\xc6\x9b=xJ\xb5\x01\x18U\r\xd8\xf9&\xcc$\x9c}\xf0\xf1%WR_R\xc1\x80\x92\xbe\xee\xd7\x92\xe6\xc1\xdfg\x81`\x0c\xe0\x19f\x86\xd9N\xfd\x938>,B[\xbe\xc4E\x92\xce\x1e;@\xeb\xcad\xc7\x90\x1d\x9b\xdcn\xbeo\xd21\x84\x00\x00\x00\x00IEND\xaeB`\x82\n\xe0\x02\n\x1binput_reshape/input/image/7"\xc0\x02\x08\x1c\x10\x1c\x18\x01"\xb7\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x00\xfeIDAT(\x91c`\x18J\x80\x11CD\xd1\x8e\x81\xe1\xc0CtI\x0b\xbf\x9d~\xa2\xc1\x0c\xccl\x0c\x0c\xbf\xce\x1d\x9fs\x1dY\x87\xd9\xab\x7f\xc8\xe0 \x03\x03\x03\x0b\\\xf2\x15/\x03\x03\x03\xc3\xb7\x0fw\xf6200\xe8)u\xa2\xd8\xc51\xf7\xff\xbf\x7f{\x0c\xb0\xba\x8c-\xe4\xff\xbf\x7f\xff\xde<x\x90\x87\xe9F\xce:\xb8mk\xa2\xd1%\xeb\xff\xfd\xfb\xff\xe7\xe7\xcf\'O\xfe\xfc\xfb\xf7P\x0cMr\xeb\xbf\'\x8d&\x0c\x0c\x0c\x0c\x963\xfe\xfd\x8bB\x93\x945\x91\x84\xb2\x84>\xff\xbb\x8a\xd5U\x0c\x0c\x0c\x0c\xc6?\xfe\xbd\x822\x990$\x03\xd8\x18\xae\xa2H\n\x17\xda\xc2\xbdd\xcf\xc0p\x02E\xf5\xf6\x7f\'D!,\xfb\x03\xff\xfe\x1d\xd3G\x91\x9c\xf9\xef\xdfrQQQQ\xa7I?\xff\xfd{\x1f\x80j\x8f\xccsD\x88\x9f\xaeBs\x04\xbb\xca\x9b?\xff\xfe\xfd\xfb\xf7\xef\xc7\xc2jN\x840< ]%\x19\x18\x18\x18\x8e\xdd\xc1\xe9E:\x01\x00\xb7\x1at\xec\xdf\x81\xea\xc2\x00\x00\x00\x00IEND\xaeB`\x82\n\x8b\x03\n\x1binput_reshape/input/image/8"\xeb\x02\x08\x1c\x10\x1c\x18\x01"\xe2\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x01)IDAT(\x91\xcd\xd11KBQ\x18\x06\xe0\xf7z\xcb\x96\\*\xe4\xde\x10\x82r\x88 \x9d\x94\x86~A[\xe2\x10\x04ED\x83\xb7\x86~@C\xcb\x8d\xc6\xdc\\\xdc\xb4\xa5?P[AC\x10:\x94\x86V\x06\x97~@E\x91\xa6\xbc\x9c\x06\xaf\xa7s\xefml\xe8\x9b\xbe\xf7<|\x87\xefp\x80\xff]\xe1\xc8\xc1\xb5\x10\xcef$\xe4\x97qs\xff\x9c\x82$\xc9$\x00hr&\xbd\xb1\x14uz\xd0\xce*\xc0\xfc\xeec\xfa\xedg*vH\xbel\x8f\xba\xc9|bN\xb92OrM\xa6\xd4+\x8f\x14\x9cY!\x0b2\x8d5y\x19\x06\x86\xdc\xd8\n\x011\x89#\xc3\xb0\xba\x80\\9\x01\xdcH\\\x9fB\\}G\x91\xcf\xd3\x83~\xa1\xc3\xb2\xae\x98\xf1\xc5\x0b\xd9\xdf\xb2n\xa8\x83\xa6\x10\x034\xefD/\xa1\x1a\xe2\xe4^\xbf\x9b\xac\xb1my\x0cyV\xfb\xcbn\xd5\xf8\xb9\xe35\xfd\x98Y\x00@N\x04\x0csd\x16\x80q\xd2\x15m\xbf\xc1\xa63\x01,^\x91\r\xcbo\xb0\xd9\xc4j\xf5\x83\xac/\x07\x0c6;\xf7\xef$\xcbF\xd0\x90!I\x96fu\xcf\xa9\xfb\xd9Z\xf4\xb4\xd5(=\xf0\x97\xb9\xbf\xafo\xb7hv\xb9\xdc,X\x0e\x00\x00\x00\x00IEND\xaeB`\x82\n\xe5\x02\n\x1binput_reshape/input/image/9"\xc5\x02\x08\x1c\x10\x1c\x18\x01"\xbc\x02\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x1c\x00\x00\x00\x1c\x08\x00\x00\x00\x00Wf\x80H\x00\x00\x01\x03IDAT(\x91\xcd\xd0\xbdKBa\x14\xc7\xf1_/K\x84:$\x04Bwm)\xc8\xa5\x17\x82 \x10\xe7\x86\x06\xc1\x86\x86\x82\x86hhs\xcbjijk\t\xac\xbf \x08"\x84\xa0Q$\xd4E\x87k-%HK\x94\x15u\xa5\xf0KKx\x89{\xda\x1a:\xd3\xf3\x9c\xcfy8\xe7<\xd2?\x8b\x9e\x1f\xb7\xec\xdc\xd8\tG\xee\x8bQ\x18*\xbc\x01p>a\xe0\x0ep\x97\xdb|\'\x1b\xb4I\x8f\x9b\xf5\xb0\xd4\xa2\x12\xb0L\x8b\xf6\xb8\xa4\xe83g\xdf\xa9\xde.>DT\xacJ\x8e\x1bR9\xf02|\xcd\xaa\xa4%\xa8\xc6\x82=OIit\xad\x8e7e\x0c\xbb\xcbU\xf2\t:\xd6"\x9am\x034\xe7-\x93\xf2\x80\x9b\xb2M{\xc0\xf2/6\xed\x01\xdb\xb6%kx\xf1\xfd\xd7\x9c\x89e>\xd2\xd2Vc\xc8\xc2\x06\x17\x92V8\xf4S\xfe\xf7\x1d\xeb^\x92\xd3\x19\xf4\xb1\xbf{*\xd4\x17\xf3\x03\x0b\x89\xbeO\xb3\xe9\xcc-@i\xd8\x9ew\xe4\xe0\xf1r#j\xdb_\xc6\x17s\x13ia\xc6\xbe\xcc\xbd\x00\x00\x00\x00IEND\xaeB`\x82\n$\n\x1dlayer1/weights/summaries/mean\x15(V\x189\n(\n!layer1/weights/summaries/stddev_1\x15o\xe4\xb4=\n#\n\x1clayer1/weights/summaries/max\x15S\xcfc>\n#\n\x1clayer1/weights/summaries/min\x15\xf9\x97d\xbe\n\x9a\x1f\n"layer1/weights/summaries/histogram*\xf3\x1e\t\x00\x00\x00@\xdc\x9f\xcc\xbf\x11\x00\x00\x00 \xdd\x80\xcc?\x19\x00\x00\x00\x00\x00\xed\x17A!\xf0<3\xf7=\xb1K@)\x7f\xf0\x92L\x0f\xe5\xa7@2\xa0\x0f\xedZ\xd3_\xc5\xea\xce\xbf\x05\xc7K\x11?\x1b\xcc\xbf\x90@\xb9\x0f"\x8d\xc9\xbf\xf7QK\x0e|:\xc7\xbf\x0fyD$\xe5\x1d\xc5\xbf\x82?>8s2\xc3\xbf\xd3\x968\x90\xaes\xc1\xbf!\xb5\xac\xd7\x0e\xbb\xbf\xbf\xa9\xa4(!\x99\xd8\xbc\xbf%g\xf6cE9\xba\xbf8/\x9aC\xf9\xd6\xb7\xbf\x04\x88\x00l(\xac\xb5\xbf\xa6{\x00\xa8\xc7\xb3\xb3\xbf\xae\x87]$A\xe9\xb1\xbf\xb5\xef\xf7\xc3iH\xb0\xbfI\x85\xab\xd8\xee\x9a\xad\xbf\xfc\xbe\xe1g\xf0\xe9\xaa\xbf\xcdg\x87\xbb\x94w\xa8\xbf\xd1/\xd8\x1e*>\xa6\xbf`\xce\xdba\x838\xa4\xbf\xcbuS\xcd\xeba\xa2\xbf\xfe\r\x06/\x1c\xb6\xa0\xbf\x87v\xdc\xc9ab\x9e\xbf\xd7\xc8<\xa0A\x9f\x9b\xbf}Y\x944j\x1c\x99\xbf\xce"\x9eu\x03\xd4\x96\xbf^\x08\xa7S\xbd\xc0\x94\xbf\x9b\x07Rc\xc3\xdd\x92\xbf\xea\x06\x1c\xa0\xb1&\x91\xbf\xd7#\xedh\x14/\x8f\xbf\xda7c_XY\x8c\xbf#\xa7+(\x96\xc5\x89\xbfei\x10S\xcem\x87\xbf-\xd4\xb1\x05\x8dL\x85\xbf\xe3\xc0\xfeJ\xdd\\\x83\xbf\xe5\xdd\xfe\x89=\x9a\x81\xbf\x16>\x16\t\x95\x00\x80\xbf\xcb\x13\xb4\xcaT\x18}\xbf\x15o\x8c\xcf5sz\xbf*\x1fQH\xa5\x0bx\xbf&b\xd5\x9e\n\xdcu\xbfh\x13\x1fyO\xdfs\xbfuWy\xe2\xd3\x10r\xbf;8\x11\x88clp\xbf\xf6N\x1f\xe0W\xdcm\xbf\xdf\xa4\xd6(g%k\xbf\x10P}\x99\xa3\xadh\xbfTw\xa0\xe8Nof\xbf5Ucv0ed\xbf\x8d\xaa\xce%\x89\x8ab\xbf\x80l\xeaP\x08\xdb`\xbfE\xc5\xd8{\x83\xa5^\xbf\x84m9\xfcH\xdc[\xbf\xa6\xc0bB\xe5SY\xbf\xdc\x97\x9fSs\x06W\xbf<DKc\x97\xeeT\xbf\x93lDZr\x07S\xbfnK\x9b\x80\x96LQ\xbfk\xa01^\xfasO\xbfIcD\xe1\xfa\x97L\xbf\x88qU\xfb\x86\xfeI\xbf\x1e\n\xc2\xfb\x91\xa1G\xbfa\xc3$\xfc\x9b{E\xbf\x86T\xc4\x13\xa5\x87C\xbf\x05\x07\xe1\xb4!\xc1A\xbf\x04\x92\xcc\x18\xf0#@\xbfd\xac\\D\x9dX=\xbf\xfd\xb3%>\xa6\xad:\xbf\x14u\xdc\xac\xc5@8\xbf\xb5\x81%\x9dV\x0c6\xbf\xeau\x96w7\x0b4\xbf\x1a\x0e\xa0\x0f\xbe82\xbf\x8c\xf5b\xc8\xac\x900\xbf\xb8\xd5VlQ\x1e.\xbf\x04\xab7Kaa+\xbfI\xe1I\xe7)\xe4(\xbf+A\xa0F\xe0\xa0&\xbfU\x984@@\x92$\xbf\xf0[^:\x80\xb3"\xbf\x94S\x84\x06F\x00!\xbfji6\x809\xe9\x1e\xbf\x03\x1a\xbd.\xd7\x19\x1c\xbf\x19\xe9\xab\xe4\xda\x8b\x19\xbf\xd0vV\xa1R9\x17\xbf\xbdT7\xc1\xd6\x1c\x15\xbf\xc35\xd5i}1\x13\xbf\x82\xbcd\x1a\xcfr\x11\xbfx?\x14\x8dx\xb9\x0f\xbf>h\x12\xc6\'\xd7\x0c\xbff\xea\xca\x9c\xf57\n\xbf\xa2\x03\xe7\x02\xc8\xd5\x07\xbf6\xa6]\xeb\x12\xab\x05\xbf1\xf4\x9aa\xcb\xb2\x03\xbf\xe6\x80\x18\xcd[\xe8\x01\xbf\xa2F\x16F\x99G\x00\xbf>\xc6?\xc5s\x99\xfd\xbeO\x11\x97\xca\x97\xe8\xfa\xbe\xea\xc9Zr[v\xf8\xbeI\xa0\xafP\r=\xf6\xbepz\xe5w\x807\xf4\xbe})\xffl\x00a\xf2\xbe\xe6\xdf\x8a4F\xb5\xf0\xbe\x8bh\x9f\xbc\xdc`\xee\xbe8\x19K\xf1\xdf\x9d\xeb\xbea\xe8\xcf\xac(\x1b\xe9\xbe\xa2\x01f\x99\xb3\xbf\xe4\xbe\xd9uE\xd1\xd1\xdc\xe2\xbe\xae\x99\x10\x04\xd6%\xe1\xbe\x99\xd1\xa9\x1e\x85-\xdf\xbeE\xa7\xb1a\xedW\xdc\xbe\x84iD*L\xc4\xd9\xbe\xd6\xe2>M|K\xd5\xbe\xc2\xfc~]\xe5[\xd3\xbejqs&\\\x99\xd1\xbeK+\xe9E\x90\xff\xcf\xbe\x0c;\x81"\xe3q\xca\xbe\n\xaa/eq\n\xc8\xbe\xf1\x0e\x14\xb9\xf2\xda\xc5\xbe\x95\rX\x05Q\xde\xc3\xbe\xfb\x97~\x90\xec\x0f\xc2\xbe\x9e[\x8a=\x91k\xc0\xbe\xb9\x8d\xd8?\xe4\xda\xab>\x19\xcf\x07\x13\xfb\xa3\xae>\xe8\xb1\xaa}0\xda\xb0>\xe6\xf6n\xbd\x9b\x89\xb2>5\xcc"\x9cg\xac\xb8>\x08G&\x92\x0b$\xbb>\xf0\x1a*\x87\xd9\xda\xbd>K+\xe9E\x90\xff\xcf>jqs&\\\x99\xd1>\xc2\xfc~]\xe5[\xd3>\xd6\xe2>M|K\xd5>\xec_\xf8T\xa2l\xd7>\x84iD*L\xc4\xd9>\x99\xd1\xa9\x1e\x85-\xdf>\xae\x99\x10\x04\xd6%\xe1>\xa2\x01f\x99\xb3\xbf\xe4>\xcc\x01\xbd(\xdf\xd2\xe6>a\xe8\xcf\xac(\x1b\xe9>8\x19K\xf1\xdf\x9d\xeb>\x8bh\x9f\xbc\xdc`\xee>\xe6\xdf\x8a4F\xb5\xf0>})\xffl\x00a\xf2>pz\xe5w\x807\xf4>I\xa0\xafP\r=\xf6>\xea\xc9Zr[v\xf8>O\x11\x97\xca\x97\xe8\xfa>>\xc6?\xc5s\x99\xfd>\xa2F\x16F\x99G\x00?\xe6\x80\x18\xcd[\xe8\x01?1\xf4\x9aa\xcb\xb2\x03?6\xa6]\xeb\x12\xab\x05?\xa2\x03\xe7\x02\xc8\xd5\x07?f\xea\xca\x9c\xf57\n?>h\x12\xc6\'\xd7\x0c?x?\x14\x8dx\xb9\x0f?\x82\xbcd\x1a\xcfr\x11?\xc35\xd5i}1\x13?\xbdT7\xc1\xd6\x1c\x15?\xd0vV\xa1R9\x17?\x19\xe9\xab\xe4\xda\x8b\x19?\x03\x1a\xbd.\xd7\x19\x1c?ji6\x809\xe9\x1e?\x94S\x84\x06F\x00!?\xf0[^:\x80\xb3"?U\x984@@\x92$?+A\xa0F\xe0\xa0&?I\xe1I\xe7)\xe4(?\x04\xab7Kaa+?\xb8\xd5VlQ\x1e.?\x8c\xf5b\xc8\xac\x900?\x1a\x0e\xa0\x0f\xbe82?\xeau\x96w7\x0b4?\xb5\x81%\x9dV\x0c6?\x14u\xdc\xac\xc5@8?\xfd\xb3%>\xa6\xad:?d\xac\\D\x9dX=?\x04\x92\xcc\x18\xf0#@?\x05\x07\xe1\xb4!\xc1A?\x86T\xc4\x13\xa5\x87C?a\xc3$\xfc\x9b{E?\x1e\n\xc2\xfb\x91\xa1G?\x88qU\xfb\x86\xfeI?IcD\xe1\xfa\x97L?k\xa01^\xfasO?nK\x9b\x80\x96LQ?\x93lDZr\x07S?<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?\xa6\xc0bB\xe5SY?\x84m9\xfcH\xdc[?E\xc5\xd8{\x83\xa5^?\x80l\xeaP\x08\xdb`?\x8d\xaa\xce%\x89\x8ab?5Ucv0ed?Tw\xa0\xe8Nof?\x10P}\x99\xa3\xadh?\xdf\xa4\xd6(g%k?\xf6N\x1f\xe0W\xdcm?;8\x11\x88clp?uWy\xe2\xd3\x10r?h\x13\x1fyO\xdfs?&b\xd5\x9e\n\xdcu?*\x1fQH\xa5\x0bx?\x15o\x8c\xcf5sz?\xcb\x13\xb4\xcaT\x18}?\x16>\x16\t\x95\x00\x80?\xe5\xdd\xfe\x89=\x9a\x81?\xe3\xc0\xfeJ\xdd\\\x83?-\xd4\xb1\x05\x8dL\x85?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xda7c_XY\x8c?\xd7#\xedh\x14/\x8f?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd7\xc8<\xa0A\x9f\x9b?\x87v\xdc\xc9ab\x9e?\xfe\r\x06/\x1c\xb6\xa0?\xcbuS\xcd\xeba\xa2?`\xce\xdba\x838\xa4?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?\xfc\xbe\xe1g\xf0\xe9\xaa?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\xa6{\x00\xa8\xc7\xb3\xb3?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?!\xb5\xac\xd7\x0e\xbb\xbf?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xedZ\xd3_\xc5\xea\xce?\xff\xff\xff\xff\xff\xff\xef\x7f:\xa0\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00H\x8c@\x00\x00\x00\x00\x00\x83\xb0@\x00\x00\x00\x00\x00f\xb7@\x00\x00\x00\x00\x00U\xbc@\x00\x00\x00\x00\x00\x0b\xc0@\x00\x00\x00\x00\x00\x95\xc0@\x00\x00\x00\x00\x80\xcf\xc1@\x00\x00\x00\x00\x80T\xc2@\x00\x00\x00\x00\x80i\xc2@\x00\x00\x00\x00\x80\xf0\xc1@\x00\x00\x00\x00\x80\xaa\xc1@\x00\x00\x00\x00\x00?\xc1@\x00\x00\x00\x00\x00\x15\xc0@\x00\x00\x00\x00\x00\xea\xbe@\x00\x00\x00\x00\x00R\xbc@\x00\x00\x00\x00\x00-\xbb@\x00\x00\x00\x00\x00\xfd\xb8@\x00\x00\x00\x00\x008\xb7@\x00\x00\x00\x00\x00?\xb5@\x00\x00\x00\x00\x00\xc8\xb3@\x00\x00\x00\x00\x00\xf2\xb1@\x00\x00\x00\x00\x00i\xb0@\x00\x00\x00\x00\x00h\xae@\x00\x00\x00\x00\x00\xc2\xab@\x00\x00\x00\x00\x00\x82\xa9@\x00\x00\x00\x00\x00H\xa7@\x00\x00\x00\x00\x00\x10\xa5@\x00\x00\x00\x00\x00\x9e\xa3@\x00\x00\x00\x00\x00\x14\xa1@\x00\x00\x00\x00\x00\xb0\x9f@\x00\x00\x00\x00\x00`\x9d@\x00\x00\x00\x00\x00\x10\x9a@\x00\x00\x00\x00\x00\xf4\x97@\x00\x00\x00\x00\x00(\x95@\x00\x00\x00\x00\x00<\x93@\x00\x00\x00\x00\x00P\x91@\x00\x00\x00\x00\x00\xb4\x90@\x00\x00\x00\x00\x00(\x8e@\x00\x00\x00\x00\x00\xc8\x8b@\x00\x00\x00\x00\x00\x80\x88@\x00\x00\x00\x00\x00\x88\x84@\x00\x00\x00\x00\x00\x08\x85@\x00\x00\x00\x00\x00p\x82@\x00\x00\x00\x00\x00@\x80@\x00\x00\x00\x00\x00 }@\x00\x00\x00\x00\x00P|@\x00\x00\x00\x00\x00pz@\x00\x00\x00\x00\x00\x90u@\x00\x00\x00\x00\x00 s@\x00\x00\x00\x00\x00\xa0s@\x00\x00\x00\x00\x00 r@\x00\x00\x00\x00\x00\xa0m@\x00\x00\x00\x00\x00@o@\x00\x00\x00\x00\x00\x80j@\x00\x00\x00\x00\x00\xc0i@\x00\x00\x00\x00\x00`e@\x00\x00\x00\x00\x00`e@\x00\x00\x00\x00\x00@_@\x00\x00\x00\x00\x00\x00`@\x00\x00\x00\x00\x00\xc0_@\x00\x00\x00\x00\x00@Y@\x00\x00\x00\x00\x00\x00W@\x00\x00\x00\x00\x00\xc0V@\x00\x00\x00\x00\x00\x80T@\x00\x00\x00\x00\x00@V@\x00\x00\x00\x00\x00\x80Q@\x00\x00\x00\x00\x00\x00I@\x00\x00\x00\x00\x00@R@\x00\x00\x00\x00\x00\x00J@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x00G@\x00\x00\x00\x00\x00\x00G@\x00\x00\x00\x00\x00\x80D@\x00\x00\x00\x00\x00\x00?@\x00\x00\x00\x00\x00\x005@\x00\x00\x00\x00\x00\x00?@\x00\x00\x00\x00\x00\x005@\x00\x00\x00\x00\x00\x004@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x003@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x003@\x00\x00\x00\x00\x00\x008@\x00\x00\x00\x00\x00\x00?@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x80@@\x00\x00\x00\x00\x00\x00B@\x00\x00\x00\x00\x00\x00F@\x00\x00\x00\x00\x00\x80G@\x00\x00\x00\x00\x00\x80@@\x00\x00\x00\x00\x00\x00J@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x00O@\x00\x00\x00\x00\x00\x00Q@\x00\x00\x00\x00\x00\xc0R@\x00\x00\x00\x00\x00\x80N@\x00\x00\x00\x00\x00\xc0R@\x00\x00\x00\x00\x00\xc0T@\x00\x00\x00\x00\x00\x80a@\x00\x00\x00\x00\x00 `@\x00\x00\x00\x00\x00\x00\\@\x00\x00\x00\x00\x00 c@\x00\x00\x00\x00\x00`c@\x00\x00\x00\x00\x00`d@\x00\x00\x00\x00\x00`i@\x00\x00\x00\x00\x00@l@\x00\x00\x00\x00\x00\xc0k@\x00\x00\x00\x00\x00\xa0m@\x00\x00\x00\x00\x00\x10q@\x00\x00\x00\x00\x000s@\x00\x00\x00\x00\x00\x90s@\x00\x00\x00\x00\x00\xd0x@\x00\x00\x00\x00\x00py@\x00\x00\x00\x00\x00\x10|@\x00\x00\x00\x00\x00\xc0\x7f@\x00\x00\x00\x00\x00\x08\x81@\x00\x00\x00\x00\x00\xc8\x81@\x00\x00\x00\x00\x00p\x84@\x00\x00\x00\x00\x00p\x87@\x00\x00\x00\x00\x00 \x89@\x00\x00\x00\x00\x00\xb0\x8b@\x00\x00\x00\x00\x008\x8e@\x00\x00\x00\x00\x00d\x90@\x00\x00\x00\x00\x00<\x92@\x00\x00\x00\x00\x00\xc4\x94@\x00\x00\x00\x00\x00,\x96@\x00\x00\x00\x00\x00\x8c\x98@\x00\x00\x00\x00\x00\x1c\x9a@\x00\x00\x00\x00\x00\x94\x9d@\x00\x00\x00\x00\x00\x16\xa0@\x00\x00\x00\x00\x00\xaa\xa1@\x00\x00\x00\x00\x00Z\xa3@\x00\x00\x00\x00\x00>\xa5@\x00\x00\x00\x00\x00p\xa6@\x00\x00\x00\x00\x00\xca\xa8@\x00\x00\x00\x00\x00\xd2\xab@\x00\x00\x00\x00\x00\xde\xad@\x00\x00\x00\x00\x00<\xb0@\x00\x00\x00\x00\x00\x08\xb2@\x00\x00\x00\x00\x00\xa3\xb3@\x00\x00\x00\x00\x00z\xb5@\x00\x00\x00\x00\x00\xcf\xb6@\x00\x00\x00\x00\x00f\xb9@\x00\x00\x00\x00\x00G\xbb@\x00\x00\x00\x00\x00\x1d\xbd@\x00\x00\x00\x00\x00\xd9\xbe@\x00\x00\x00\x00\x80n\xc0@\x00\x00\x00\x00\x80r\xc1@\x00\x00\x00\x00\x80\xa5\xc1@\x00\x00\x00\x00\x80N\xc2@\x00\x00\x00\x00\x80G\xc2@\x00\x00\x00\x00\x80B\xc2@\x00\x00\x00\x00\x80\xf2\xc1@\x00\x00\x00\x00\x00\'\xc1@\x00\x00\x00\x00\x00\x8c\xbf@\x00\x00\x00\x00\x00\x9b\xbb@\x00\x00\x00\x00\x00\x0f\xb8@\x00\x00\x00\x00\x001\xb0@\x00\x00\x00\x00\x00\x80\x8d@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x00\x00\n#\n\x1clayer1/biases/summaries/mean\x15\xa2\x8b\xce=\n\'\n layer1/biases/summaries/stddev_1\x15Y\xc6&<\n"\n\x1blayer1/biases/summaries/max\x15`\x99\x04>\n"\n\x1blayer1/biases/summaries/min\x15\xc7y\x9d=\n\xe7\x01\n!layer1/biases/summaries/histogram*\xc1\x01\t\x00\x00\x00\xe08\xaf\xb3?\x11\x00\x00\x00\x00,\x93\xc0?\x19\x00\x00\x00\x00\x00@\x7f@!\x00\x00p\x9a\x8b6I@)\xeaH\xc4\xf1\xb1\x8c\x14@2H\xae\x87]$A\xe9\xb1?\xa6{\x00\xa8\xc7\xb3\xb3?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?!\xb5\xac\xd7\x0e\xbb\xbf?\xd3\x968\x90\xaes\xc1?\xff\xff\xff\xff\xff\xff\xef\x7f:H\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00>@\x00\x00\x00\x00\x00\x80V@\x00\x00\x00\x00\x00\x00d@\x00\x00\x00\x00\x00\xe0b@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\n\xa8\x1d\n layer1/Wx_plus_b/pre_activations*\x83\x1d\t\x00\x00\x00`\xaf\xf7\x10\xc0\x11\x00\x00\x00\xe0\xb5\x01\x12@\x19\x00\x00\x00\x00\x00j\xe8@!\x00 \x8a\xab\x9fq\xbd@) V\nZ\xdc\xdc\xe6@2\xa8\x0e!\xd1\x0e\x9av\x8b\x12\xc0{2\xf6.\xe0\xdb\x10\xc0\x81DK\xf8\x0b\xa7\x0e\xc0u\xf8r\xca\xad\xdd\x0b\xc0\r\x9c\xc5\xa0)U\t\xc0h\xbc\x105\x9a\x07\x07\xc0\xea6\x0fv\xa3\xef\x04\xc0w`<\x0ef\x08\x03\xc0\xe0\x11\xf1\x0ctM\x01\xc0\xdc\x94\xcd.\x8du\xff\xbfS\x87F\x13i\x99\xfc\xbfyL\xfa\xe2\xd3\xff\xf9\xbf\x85E\xcc\x9f\xc0\xa2\xf7\xbf3?\xe8\x1c\xaf|\xf5\xbf\xbaP\x8d1\x9f\x88\xf3\xbfcI\xc6\x15\x05\xc2\xf1\xbf+\x14\xfa\xcd\xbe$\xf0\xbf\x08\x99i\x19\x15Z\xed\xbf\x1e\x8b\x8e\xe8\xfb\xae\xea\xbf2g\xc7G\xfcA\xe8\xbf\x16uo\xfbp\r\xe6\xbf+Se*8\x0c\xe4\xbf\x0f\x1d\\l\xa79\xe2\xbf\xb01%\xee\x80\x91\xe0\xbf\xfa\xe5Z%\xd3\x1f\xde\xbfW\xe8i\xf3\xbfb\xdb\xbf\xda^\xbd\xaeh\xe5\xd8\xbf\x97\xca7\x13\x02\xa2\xd6\xbf\xcfCa\xb4G\x93\xd4\xbf_&A\xbbo\xb4\xd2\xbf\x9c\x0b\x81\xc1\x1f\x01\xd1\xbf\xedZ\xd3_\xc5\xea\xce\xbf\x05\xc7K\x11?\x1b\xcc\xbf\x90@\xb9\x0f"\x8d\xc9\xbf\xf7QK\x0e|:\xc7\xbf\x0fyD$\xe5\x1d\xc5\xbf\x82?>8s2\xc3\xbf\xd3\x968\x90\xaes\xc1\xbf!\xb5\xac\xd7\x0e\xbb\xbf\xbf\xa9\xa4(!\x99\xd8\xbc\xbf%g\xf6cE9\xba\xbf8/\x9aC\xf9\xd6\xb7\xbf\x04\x88\x00l(\xac\xb5\xbf\xa6{\x00\xa8\xc7\xb3\xb3\xbf\xae\x87]$A\xe9\xb1\xbf\xb5\xef\xf7\xc3iH\xb0\xbfI\x85\xab\xd8\xee\x9a\xad\xbf\xfc\xbe\xe1g\xf0\xe9\xaa\xbf\xcdg\x87\xbb\x94w\xa8\xbf\xd1/\xd8\x1e*>\xa6\xbf`\xce\xdba\x838\xa4\xbf\xcbuS\xcd\xeba\xa2\xbf\xfe\r\x06/\x1c\xb6\xa0\xbf\x87v\xdc\xc9ab\x9e\xbf\xd7\xc8<\xa0A\x9f\x9b\xbf}Y\x944j\x1c\x99\xbf\xce"\x9eu\x03\xd4\x96\xbf^\x08\xa7S\xbd\xc0\x94\xbf\x9b\x07Rc\xc3\xdd\x92\xbf\xea\x06\x1c\xa0\xb1&\x91\xbf\xd7#\xedh\x14/\x8f\xbf\xda7c_XY\x8c\xbf#\xa7+(\x96\xc5\x89\xbfei\x10S\xcem\x87\xbf-\xd4\xb1\x05\x8dL\x85\xbf\xe3\xc0\xfeJ\xdd\\\x83\xbf\xe5\xdd\xfe\x89=\x9a\x81\xbf\x16>\x16\t\x95\x00\x80\xbf\xcb\x13\xb4\xcaT\x18}\xbf\x15o\x8c\xcf5sz\xbf*\x1fQH\xa5\x0bx\xbf&b\xd5\x9e\n\xdcu\xbfh\x13\x1fyO\xdfs\xbfuWy\xe2\xd3\x10r\xbf;8\x11\x88clp\xbf\xf6N\x1f\xe0W\xdcm\xbf\xdf\xa4\xd6(g%k\xbf\x10P}\x99\xa3\xadh\xbfTw\xa0\xe8Nof\xbf5Ucv0ed\xbf\x8d\xaa\xce%\x89\x8ab\xbf\x80l\xeaP\x08\xdb`\xbfE\xc5\xd8{\x83\xa5^\xbf\x84m9\xfcH\xdc[\xbf\xa6\xc0bB\xe5SY\xbf\xdc\x97\x9fSs\x06W\xbf<DKc\x97\xeeT\xbf\x93lDZr\x07S\xbfnK\x9b\x80\x96LQ\xbfk\xa01^\xfasO\xbfIcD\xe1\xfa\x97L\xbf\x88qU\xfb\x86\xfeI\xbf\x1e\n\xc2\xfb\x91\xa1G\xbfa\xc3$\xfc\x9b{E\xbf\x86T\xc4\x13\xa5\x87C\xbf\x05\x07\xe1\xb4!\xc1A\xbf\x04\x92\xcc\x18\xf0#@\xbfd\xac\\D\x9dX=\xbf\xfd\xb3%>\xa6\xad:\xbf\x14u\xdc\xac\xc5@8\xbf\xb5\x81%\x9dV\x0c6\xbf\x1a\x0e\xa0\x0f\xbe82\xbf\x8c\xf5b\xc8\xac\x900\xbf\xb8\xd5VlQ\x1e.\xbf\x04\xab7Kaa+\xbf\x19\xe9\xab\xe4\xda\x8b\x19\xbf\xd0vV\xa1R9\x17\xbf\xc35\xd5i}1\x13\xbf\x82\xbcd\x1a\xcfr\x11\xbf>h\x12\xc6\'\xd7\x0c\xbff\xea\xca\x9c\xf57\n\xbf\xa2F\x16F\x99G\x00\xbf>\xc6?\xc5s\x99\xfd\xbeO\x11\x97\xca\x97\xe8\xfa>>\xc6?\xc5s\x99\xfd>\xa2F\x16F\x99G\x00?\xe6\x80\x18\xcd[\xe8\x01?6\xa6]\xeb\x12\xab\x05?\xa2\x03\xe7\x02\xc8\xd5\x07?>h\x12\xc6\'\xd7\x0c?x?\x14\x8dx\xb9\x0f?\xc35\xd5i}1\x13?\xbdT7\xc1\xd6\x1c\x15?+A\xa0F\xe0\xa0&?I\xe1I\xe7)\xe4(?\x1a\x0e\xa0\x0f\xbe82?\xeau\x96w7\x0b4?\xb5\x81%\x9dV\x0c6?\x14u\xdc\xac\xc5@8?\xfd\xb3%>\xa6\xad:?d\xac\\D\x9dX=?\x04\x92\xcc\x18\xf0#@?\x05\x07\xe1\xb4!\xc1A?\x86T\xc4\x13\xa5\x87C?a\xc3$\xfc\x9b{E?\x1e\n\xc2\xfb\x91\xa1G?\x88qU\xfb\x86\xfeI?IcD\xe1\xfa\x97L?k\xa01^\xfasO?nK\x9b\x80\x96LQ?\x93lDZr\x07S?<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?\xa6\xc0bB\xe5SY?\x84m9\xfcH\xdc[?E\xc5\xd8{\x83\xa5^?\x80l\xeaP\x08\xdb`?\x8d\xaa\xce%\x89\x8ab?5Ucv0ed?Tw\xa0\xe8Nof?\x10P}\x99\xa3\xadh?\xdf\xa4\xd6(g%k?\xf6N\x1f\xe0W\xdcm?;8\x11\x88clp?uWy\xe2\xd3\x10r?h\x13\x1fyO\xdfs?&b\xd5\x9e\n\xdcu?*\x1fQH\xa5\x0bx?\x15o\x8c\xcf5sz?\xcb\x13\xb4\xcaT\x18}?\x16>\x16\t\x95\x00\x80?\xe5\xdd\xfe\x89=\x9a\x81?\xe3\xc0\xfeJ\xdd\\\x83?-\xd4\xb1\x05\x8dL\x85?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xda7c_XY\x8c?\xd7#\xedh\x14/\x8f?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd7\xc8<\xa0A\x9f\x9b?\x87v\xdc\xc9ab\x9e?\xfe\r\x06/\x1c\xb6\xa0?\xcbuS\xcd\xeba\xa2?`\xce\xdba\x838\xa4?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?\xfc\xbe\xe1g\xf0\xe9\xaa?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\xa6{\x00\xa8\xc7\xb3\xb3?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?!\xb5\xac\xd7\x0e\xbb\xbf?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xedZ\xd3_\xc5\xea\xce?\x9c\x0b\x81\xc1\x1f\x01\xd1?_&A\xbbo\xb4\xd2?\xcfCa\xb4G\x93\xd4?\x97\xca7\x13\x02\xa2\xd6?\xda^\xbd\xaeh\xe5\xd8?W\xe8i\xf3\xbfb\xdb?\xfa\xe5Z%\xd3\x1f\xde?\xb01%\xee\x80\x91\xe0?\x0f\x1d\\l\xa79\xe2?+Se*8\x0c\xe4?\x16uo\xfbp\r\xe6?2g\xc7G\xfcA\xe8?\x1e\x8b\x8e\xe8\xfb\xae\xea?\x08\x99i\x19\x15Z\xed?+\x14\xfa\xcd\xbe$\xf0?cI\xc6\x15\x05\xc2\xf1?\xbaP\x8d1\x9f\x88\xf3?3?\xe8\x1c\xaf|\xf5?\x85E\xcc\x9f\xc0\xa2\xf7?yL\xfa\xe2\xd3\xff\xf9?S\x87F\x13i\x99\xfc?\xdc\x94\xcd.\x8du\xff?\xe0\x11\xf1\x0ctM\x01@w`<\x0ef\x08\x03@\xea6\x0fv\xa3\xef\x04@h\xbc\x105\x9a\x07\x07@\r\x9c\xc5\xa0)U\t@u\xf8r\xca\xad\xdd\x0b@\x81DK\xf8\x0b\xa7\x0e@{2\xf6.\xe0\xdb\x10@!\xd1\x0e\x9av\x8b\x12@\xff\xff\xff\xff\xff\xff\xef\x7f:\xa8\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00M@\x00\x00\x00\x00\x00\xc0V@\x00\x00\x00\x00\x00`c@\x00\x00\x00\x00\x00\xc0i@\x00\x00\x00\x00\x00\xa0q@\x00\x00\x00\x00\x00\x10z@\x00\x00\x00\x00\x00\x10~@\x00\x00\x00\x00\x000\x82@\x00\x00\x00\x00\x00(\x85@\x00\x00\x00\x00\x00\x90\x87@\x00\x00\x00\x00\x00\xd0\x87@\x00\x00\x00\x00\x00\x10\x8b@\x00\x00\x00\x00\x00\xb0\x8c@\x00\x00\x00\x00\x00\xb8\x8c@\x00\x00\x00\x00\x00\xa0\x8d@\x00\x00\x00\x00\x00\xd0\x8c@\x00\x00\x00\x00\x00H\x8c@\x00\x00\x00\x00\x00X\x8b@\x00\x00\x00\x00\x00\xd8\x89@\x00\x00\x00\x00\x00\x00\x88@\x00\x00\x00\x00\x000\x86@\x00\x00\x00\x00\x00\xd8\x84@\x00\x00\x00\x00\x00\xa8\x83@\x00\x00\x00\x00\x00x\x83@\x00\x00\x00\x00\x008\x81@\x00\x00\x00\x00\x00@\x80@\x00\x00\x00\x00\x000~@\x00\x00\x00\x00\x00Pz@\x00\x00\x00\x00\x00\xc0{@\x00\x00\x00\x00\x00 t@\x00\x00\x00\x00\x000w@\x00\x00\x00\x00\x00Pr@\x00\x00\x00\x00\x00pr@\x00\x00\x00\x00\x00\xc0q@\x00\x00\x00\x00\x00\xe0k@\x00\x00\x00\x00\x00\xe0j@\x00\x00\x00\x00\x00\x80e@\x00\x00\x00\x00\x00@f@\x00\x00\x00\x00\x00\x00c@\x00\x00\x00\x00\x00\x00b@\x00\x00\x00\x00\x00 `@\x00\x00\x00\x00\x00\x00\\@\x00\x00\x00\x00\x00@X@\x00\x00\x00\x00\x00@W@\x00\x00\x00\x00\x00\x80R@\x00\x00\x00\x00\x00\x80V@\x00\x00\x00\x00\x00\xc0S@\x00\x00\x00\x00\x00\x80N@\x00\x00\x00\x00\x00\x00K@\x00\x00\x00\x00\x00\x00P@\x00\x00\x00\x00\x00\x00J@\x00\x00\x00\x00\x00\x00J@\x00\x00\x00\x00\x00\x80E@\x00\x00\x00\x00\x00\x80F@\x00\x00\x00\x00\x00\x006@\x00\x00\x00\x00\x00\x00D@\x00\x00\x00\x00\x00\x00;@\x00\x00\x00\x00\x00\x80B@\x00\x00\x00\x00\x00\x006@\x00\x00\x00\x00\x00\x00@@\x00\x00\x00\x00\x00\x00=@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x004@\x00\x00\x00\x00\x00\x006@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x80@@\x00\x00\x00\x00\x00\x80A@\x00\x00\x00\x00\x00\x80C@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x00B@\x00\x00\x00\x00\x00\x80Q@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x00N@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x00T@\x00\x00\x00\x00\x00\x00V@\x00\x00\x00\x00\x00\x00]@\x00\x00\x00\x00\x00\x00^@\x00\x00\x00\x00\x00\x80^@\x00\x00\x00\x00\x00@a@\x00\x00\x00\x00\x00`d@\x00\x00\x00\x00\x00@d@\x00\x00\x00\x00\x00`l@\x00\x00\x00\x00\x00\xe0i@\x00\x00\x00\x00\x00`m@\x00\x00\x00\x00\x00`l@\x00\x00\x00\x00\x00`n@\x00\x00\x00\x00\x00\xa0s@\x00\x00\x00\x00\x00\xb0s@\x00\x00\x00\x00\x00\xc0v@\x00\x00\x00\x00\x00\xb0x@\x00\x00\x00\x00\x00\xd0{@\x00\x00\x00\x00\x00\x10~@\x00\x00\x00\x00\x00\x00\x80@\x00\x00\x00\x00\x00\xa0\x81@\x00\x00\x00\x00\x00P\x83@\x00\x00\x00\x00\x00\xa0\x85@\x00\x00\x00\x00\x00P\x86@\x00\x00\x00\x00\x00P\x89@\x00\x00\x00\x00\x00\xf0\x8a@\x00\x00\x00\x00\x00\x00\x8e@\x00\x00\x00\x00\x00\xf4\x90@\x00\x00\x00\x00\x00\x98\x91@\x00\x00\x00\x00\x00l\x91@\x00\x00\x00\x00\x00\xd0\x92@\x00\x00\x00\x00\x00\xb4\x93@\x00\x00\x00\x00\x00\\\x94@\x00\x00\x00\x00\x00@\x95@\x00\x00\x00\x00\x00\xa4\x94@\x00\x00\x00\x00\x00\xec\x93@\x00\x00\x00\x00\x00,\x93@\x00\x00\x00\x00\x00\x90\x91@\x00\x00\x00\x00\x00D\x90@\x00\x00\x00\x00\x00\x90\x8b@\x00\x00\x00\x00\x00`\x85@\x00\x00\x00\x00\x00\x88\x80@\x00\x00\x00\x00\x00\xd0u@\x00\x00\x00\x00\x00\xa0k@\x00\x00\x00\x00\x00\xc0b@\x00\x00\x00\x00\x00@U@\x00\x00\x00\x00\x00\x80A@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\n\xa1\x0f\n\x12layer1/activations*\x8a\x0f\x11\x00\x00\x00\xe0\xb5\x01\x12@\x19\x00\x00\x00\x00\x00j\xe8@!\x00h\x89(jQ\xd6@)\xc1\x8f\xde\xefK\x17\xdc@2\xb0\x07\x00\x00\x00\x00\x00\x00\x00\x00\x11\xea-\x81\x99\x97q=O\x11\x97\xca\x97\xe8\xfa>>\xc6?\xc5s\x99\xfd>\xa2F\x16F\x99G\x00?\xe6\x80\x18\xcd[\xe8\x01?6\xa6]\xeb\x12\xab\x05?\xa2\x03\xe7\x02\xc8\xd5\x07?>h\x12\xc6\'\xd7\x0c?x?\x14\x8dx\xb9\x0f?\xc35\xd5i}1\x13?\xbdT7\xc1\xd6\x1c\x15?+A\xa0F\xe0\xa0&?I\xe1I\xe7)\xe4(?\x1a\x0e\xa0\x0f\xbe82?\xeau\x96w7\x0b4?\xb5\x81%\x9dV\x0c6?\x14u\xdc\xac\xc5@8?\xfd\xb3%>\xa6\xad:?d\xac\\D\x9dX=?\x04\x92\xcc\x18\xf0#@?\x05\x07\xe1\xb4!\xc1A?\x86T\xc4\x13\xa5\x87C?a\xc3$\xfc\x9b{E?\x1e\n\xc2\xfb\x91\xa1G?\x88qU\xfb\x86\xfeI?IcD\xe1\xfa\x97L?k\xa01^\xfasO?nK\x9b\x80\x96LQ?\x93lDZr\x07S?<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?\xa6\xc0bB\xe5SY?\x84m9\xfcH\xdc[?E\xc5\xd8{\x83\xa5^?\x80l\xeaP\x08\xdb`?\x8d\xaa\xce%\x89\x8ab?5Ucv0ed?Tw\xa0\xe8Nof?\x10P}\x99\xa3\xadh?\xdf\xa4\xd6(g%k?\xf6N\x1f\xe0W\xdcm?;8\x11\x88clp?uWy\xe2\xd3\x10r?h\x13\x1fyO\xdfs?&b\xd5\x9e\n\xdcu?*\x1fQH\xa5\x0bx?\x15o\x8c\xcf5sz?\xcb\x13\xb4\xcaT\x18}?\x16>\x16\t\x95\x00\x80?\xe5\xdd\xfe\x89=\x9a\x81?\xe3\xc0\xfeJ\xdd\\\x83?-\xd4\xb1\x05\x8dL\x85?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xda7c_XY\x8c?\xd7#\xedh\x14/\x8f?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd7\xc8<\xa0A\x9f\x9b?\x87v\xdc\xc9ab\x9e?\xfe\r\x06/\x1c\xb6\xa0?\xcbuS\xcd\xeba\xa2?`\xce\xdba\x838\xa4?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?\xfc\xbe\xe1g\xf0\xe9\xaa?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\xa6{\x00\xa8\xc7\xb3\xb3?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?!\xb5\xac\xd7\x0e\xbb\xbf?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xedZ\xd3_\xc5\xea\xce?\x9c\x0b\x81\xc1\x1f\x01\xd1?_&A\xbbo\xb4\xd2?\xcfCa\xb4G\x93\xd4?\x97\xca7\x13\x02\xa2\xd6?\xda^\xbd\xaeh\xe5\xd8?W\xe8i\xf3\xbfb\xdb?\xfa\xe5Z%\xd3\x1f\xde?\xb01%\xee\x80\x91\xe0?\x0f\x1d\\l\xa79\xe2?+Se*8\x0c\xe4?\x16uo\xfbp\r\xe6?2g\xc7G\xfcA\xe8?\x1e\x8b\x8e\xe8\xfb\xae\xea?\x08\x99i\x19\x15Z\xed?+\x14\xfa\xcd\xbe$\xf0?cI\xc6\x15\x05\xc2\xf1?\xbaP\x8d1\x9f\x88\xf3?3?\xe8\x1c\xaf|\xf5?\x85E\xcc\x9f\xc0\xa2\xf7?yL\xfa\xe2\xd3\xff\xf9?S\x87F\x13i\x99\xfc?\xdc\x94\xcd.\x8du\xff?\xe0\x11\xf1\x0ctM\x01@w`<\x0ef\x08\x03@\xea6\x0fv\xa3\xef\x04@h\xbc\x105\x9a\x07\x07@\r\x9c\xc5\xa0)U\t@u\xf8r\xca\xad\xdd\x0b@\x81DK\xf8\x0b\xa7\x0e@{2\xf6.\xe0\xdb\x10@!\xd1\x0e\x9av\x8b\x12@\xff\xff\xff\xff\xff\xff\xef\x7f:\xb0\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x15\xd5@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x004@\x00\x00\x00\x00\x00\x006@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x80@@\x00\x00\x00\x00\x00\x80A@\x00\x00\x00\x00\x00\x80C@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x00B@\x00\x00\x00\x00\x00\x80Q@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x00N@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x80P@\x00\x00\x00\x00\x00\x00T@\x00\x00\x00\x00\x00\x00V@\x00\x00\x00\x00\x00\x00]@\x00\x00\x00\x00\x00\x00^@\x00\x00\x00\x00\x00\x80^@\x00\x00\x00\x00\x00@a@\x00\x00\x00\x00\x00`d@\x00\x00\x00\x00\x00@d@\x00\x00\x00\x00\x00`l@\x00\x00\x00\x00\x00\xe0i@\x00\x00\x00\x00\x00`m@\x00\x00\x00\x00\x00`l@\x00\x00\x00\x00\x00`n@\x00\x00\x00\x00\x00\xa0s@\x00\x00\x00\x00\x00\xb0s@\x00\x00\x00\x00\x00\xc0v@\x00\x00\x00\x00\x00\xb0x@\x00\x00\x00\x00\x00\xd0{@\x00\x00\x00\x00\x00\x10~@\x00\x00\x00\x00\x00\x00\x80@\x00\x00\x00\x00\x00\xa0\x81@\x00\x00\x00\x00\x00P\x83@\x00\x00\x00\x00\x00\xa0\x85@\x00\x00\x00\x00\x00P\x86@\x00\x00\x00\x00\x00P\x89@\x00\x00\x00\x00\x00\xf0\x8a@\x00\x00\x00\x00\x00\x00\x8e@\x00\x00\x00\x00\x00\xf4\x90@\x00\x00\x00\x00\x00\x98\x91@\x00\x00\x00\x00\x00l\x91@\x00\x00\x00\x00\x00\xd0\x92@\x00\x00\x00\x00\x00\xb4\x93@\x00\x00\x00\x00\x00\\\x94@\x00\x00\x00\x00\x00@\x95@\x00\x00\x00\x00\x00\xa4\x94@\x00\x00\x00\x00\x00\xec\x93@\x00\x00\x00\x00\x00,\x93@\x00\x00\x00\x00\x00\x90\x91@\x00\x00\x00\x00\x00D\x90@\x00\x00\x00\x00\x00\x90\x8b@\x00\x00\x00\x00\x00`\x85@\x00\x00\x00\x00\x00\x88\x80@\x00\x00\x00\x00\x00\xd0u@\x00\x00\x00\x00\x00\xa0k@\x00\x00\x00\x00\x00\xc0b@\x00\x00\x00\x00\x00@U@\x00\x00\x00\x00\x00\x80A@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\n\'\n dropout/dropout_keep_probability\x15fff?\n$\n\x1dlayer2/weights/summaries/mean\x15&\xb8\x19\xbb\n(\n!layer2/weights/summaries/stddev_1\x15\xfb\xc8\xb7=\n#\n\x1clayer2/weights/summaries/max\x15\r\x8a\\>\n#\n\x1clayer2/weights/summaries/min\x15:dc\xbe\n\xaa\x15\n"layer2/weights/summaries/histogram*\x83\x15\t\x00\x00\x00@\x87l\xcc\xbf\x11\x00\x00\x00\xa0A\x91\xcb?\x19\x00\x00\x00\x00\x00\x88\xb3@!\x00\xa0\xb4&\xaat\'\xc0)\xa6Z\xf0dy%D@2\xa8\n\xedZ\xd3_\xc5\xea\xce\xbf\x05\xc7K\x11?\x1b\xcc\xbf\x90@\xb9\x0f"\x8d\xc9\xbf\xf7QK\x0e|:\xc7\xbf\x0fyD$\xe5\x1d\xc5\xbf\x82?>8s2\xc3\xbf\xd3\x968\x90\xaes\xc1\xbf!\xb5\xac\xd7\x0e\xbb\xbf\xbf\xa9\xa4(!\x99\xd8\xbc\xbf%g\xf6cE9\xba\xbf8/\x9aC\xf9\xd6\xb7\xbf\x04\x88\x00l(\xac\xb5\xbf\xa6{\x00\xa8\xc7\xb3\xb3\xbf\xae\x87]$A\xe9\xb1\xbf\xb5\xef\xf7\xc3iH\xb0\xbfI\x85\xab\xd8\xee\x9a\xad\xbf\xfc\xbe\xe1g\xf0\xe9\xaa\xbf\xcdg\x87\xbb\x94w\xa8\xbf\xd1/\xd8\x1e*>\xa6\xbf`\xce\xdba\x838\xa4\xbf\xcbuS\xcd\xeba\xa2\xbf\xfe\r\x06/\x1c\xb6\xa0\xbf\x87v\xdc\xc9ab\x9e\xbf\xd7\xc8<\xa0A\x9f\x9b\xbf}Y\x944j\x1c\x99\xbf\xce"\x9eu\x03\xd4\x96\xbf^\x08\xa7S\xbd\xc0\x94\xbf\x9b\x07Rc\xc3\xdd\x92\xbf\xea\x06\x1c\xa0\xb1&\x91\xbf\xd7#\xedh\x14/\x8f\xbf\xda7c_XY\x8c\xbf#\xa7+(\x96\xc5\x89\xbfei\x10S\xcem\x87\xbf-\xd4\xb1\x05\x8dL\x85\xbf\xe3\xc0\xfeJ\xdd\\\x83\xbf\xe5\xdd\xfe\x89=\x9a\x81\xbf\x16>\x16\t\x95\x00\x80\xbf\xcb\x13\xb4\xcaT\x18}\xbf\x15o\x8c\xcf5sz\xbf*\x1fQH\xa5\x0bx\xbf&b\xd5\x9e\n\xdcu\xbfh\x13\x1fyO\xdfs\xbfuWy\xe2\xd3\x10r\xbf;8\x11\x88clp\xbf\xf6N\x1f\xe0W\xdcm\xbf\xdf\xa4\xd6(g%k\xbf\x10P}\x99\xa3\xadh\xbfTw\xa0\xe8Nof\xbf5Ucv0ed\xbf\x8d\xaa\xce%\x89\x8ab\xbf\x80l\xeaP\x08\xdb`\xbfE\xc5\xd8{\x83\xa5^\xbf\x84m9\xfcH\xdc[\xbf\xa6\xc0bB\xe5SY\xbf\xdc\x97\x9fSs\x06W\xbf<DKc\x97\xeeT\xbf\x93lDZr\x07S\xbfnK\x9b\x80\x96LQ\xbfk\xa01^\xfasO\xbfIcD\xe1\xfa\x97L\xbf\x1e\n\xc2\xfb\x91\xa1G\xbfa\xc3$\xfc\x9b{E\xbf\x86T\xc4\x13\xa5\x87C\xbf\x05\x07\xe1\xb4!\xc1A\xbf\x04\x92\xcc\x18\xf0#@\xbfd\xac\\D\x9dX=\xbf\xfd\xb3%>\xa6\xad:\xbf\x14u\xdc\xac\xc5@8\xbf\xb5\x81%\x9dV\x0c6\xbf\xeau\x96w7\x0b4\xbf\x1a\x0e\xa0\x0f\xbe82\xbf\x8c\xf5b\xc8\xac\x900\xbf\xb8\xd5VlQ\x1e.\xbf\x04\xab7Kaa+\xbfI\xe1I\xe7)\xe4(\xbf+A\xa0F\xe0\xa0&\xbfU\x984@@\x92$\xbf\xf0[^:\x80\xb3"\xbf\x94S\x84\x06F\x00!\xbf\x82\xbcd\x1a\xcfr\x11\xbfx?\x14\x8dx\xb9\x0f\xbf\x8bh\x9f\xbc\xdc`\xee\xbe8\x19K\xf1\xdf\x9d\xeb\xbe\xcc\x01\xbd(\xdf\xd2\xe6\xbe\xa2\x01f\x99\xb3\xbf\xe4\xbe\x82\xbcd\x1a\xcfr\x11?\xc35\xd5i}1\x13?ji6\x809\xe9\x1e?\x94S\x84\x06F\x00!?\xf0[^:\x80\xb3"?U\x984@@\x92$?+A\xa0F\xe0\xa0&?I\xe1I\xe7)\xe4(?\x1a\x0e\xa0\x0f\xbe82?\xeau\x96w7\x0b4?\xb5\x81%\x9dV\x0c6?\x14u\xdc\xac\xc5@8?\xfd\xb3%>\xa6\xad:?d\xac\\D\x9dX=?\x04\x92\xcc\x18\xf0#@?\x05\x07\xe1\xb4!\xc1A?\x86T\xc4\x13\xa5\x87C?a\xc3$\xfc\x9b{E?\x1e\n\xc2\xfb\x91\xa1G?\x88qU\xfb\x86\xfeI?IcD\xe1\xfa\x97L?k\xa01^\xfasO?nK\x9b\x80\x96LQ?\x93lDZr\x07S?<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?\xa6\xc0bB\xe5SY?\x84m9\xfcH\xdc[?E\xc5\xd8{\x83\xa5^?\x80l\xeaP\x08\xdb`?\x8d\xaa\xce%\x89\x8ab?5Ucv0ed?Tw\xa0\xe8Nof?\x10P}\x99\xa3\xadh?\xdf\xa4\xd6(g%k?\xf6N\x1f\xe0W\xdcm?;8\x11\x88clp?uWy\xe2\xd3\x10r?h\x13\x1fyO\xdfs?&b\xd5\x9e\n\xdcu?*\x1fQH\xa5\x0bx?\x15o\x8c\xcf5sz?\xcb\x13\xb4\xcaT\x18}?\x16>\x16\t\x95\x00\x80?\xe5\xdd\xfe\x89=\x9a\x81?\xe3\xc0\xfeJ\xdd\\\x83?-\xd4\xb1\x05\x8dL\x85?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xda7c_XY\x8c?\xd7#\xedh\x14/\x8f?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd7\xc8<\xa0A\x9f\x9b?\x87v\xdc\xc9ab\x9e?\xfe\r\x06/\x1c\xb6\xa0?\xcbuS\xcd\xeba\xa2?`\xce\xdba\x838\xa4?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?\xfc\xbe\xe1g\xf0\xe9\xaa?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\xa6{\x00\xa8\xc7\xb3\xb3?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?!\xb5\xac\xd7\x0e\xbb\xbf?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xff\xff\xff\xff\xff\xff\xef\x7f:\xa8\n\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x004@\x00\x00\x00\x00\x00\x00M@\x00\x00\x00\x00\x00\xc0S@\x00\x00\x00\x00\x00\x00Y@\x00\x00\x00\x00\x00\x80[@\x00\x00\x00\x00\x00\xc0\\@\x00\x00\x00\x00\x00\xc0_@\x00\x00\x00\x00\x00\x80_@\x00\x00\x00\x00\x00@`@\x00\x00\x00\x00\x00\xc0\\@\x00\x00\x00\x00\x00\x80_@\x00\x00\x00\x00\x00\x00_@\x00\x00\x00\x00\x00\x80Z@\x00\x00\x00\x00\x00\x00X@\x00\x00\x00\x00\x00@V@\x00\x00\x00\x00\x00\x80W@\x00\x00\x00\x00\x00\xc0S@\x00\x00\x00\x00\x00\x00M@\x00\x00\x00\x00\x00@S@\x00\x00\x00\x00\x00\x00N@\x00\x00\x00\x00\x00\x00L@\x00\x00\x00\x00\x00\x00I@\x00\x00\x00\x00\x00\x00F@\x00\x00\x00\x00\x00\x00D@\x00\x00\x00\x00\x00\x80E@\x00\x00\x00\x00\x00\x80C@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x80B@\x00\x00\x00\x00\x00\x00?@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x00D@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00<@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x003@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x006@\x00\x00\x00\x00\x00\x00=@\x00\x00\x00\x00\x00\x00C@\x00\x00\x00\x00\x00\x00A@\x00\x00\x00\x00\x00\x00E@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x80H@\x00\x00\x00\x00\x00\x80D@\x00\x00\x00\x00\x00\x00K@\x00\x00\x00\x00\x00\x00E@\x00\x00\x00\x00\x00\x00Q@\x00\x00\x00\x00\x00\x00I@\x00\x00\x00\x00\x00\x00R@\x00\x00\x00\x00\x00\x80U@\x00\x00\x00\x00\x00\x80T@\x00\x00\x00\x00\x00@[@\x00\x00\x00\x00\x00\x80X@\x00\x00\x00\x00\x00\x80\\@\x00\x00\x00\x00\x00\x80Z@\x00\x00\x00\x00\x00\x00^@\x00\x00\x00\x00\x00\xc0X@\x00\x00\x00\x00\x00\x80\\@\x00\x00\x00\x00\x00\xc0\\@\x00\x00\x00\x00\x00\xe0`@\x00\x00\x00\x00\x00\x00X@\x00\x00\x00\x00\x00\x00W@\x00\x00\x00\x00\x00\x00T@\x00\x00\x00\x00\x00\x00X@\x00\x00\x00\x00\x00\x80I@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00\x00\x00\n#\n\x1clayer2/biases/summaries/mean\x15AJ\xcc=\n\'\n layer2/biases/summaries/stddev_1\x15V\x8a\xde;\n"\n\x1blayer2/biases/summaries/max\x15\x05\x9c\xe5=\n"\n\x1blayer2/biases/summaries/min\x15\xc7Q\xb9=\n\xa7\x01\n!layer2/biases/summaries/histogram*\x81\x01\t\x00\x00\x00\xe08*\xb7?\x11\x00\x00\x00\xa0\x80\xb3\xbc?\x19\x00\x00\x00\x00\x00\x00$@!\x00\x00\x008\x9a\xeb\xef?)\xb0\xd9\xa7O;\x97\xb9?2(\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?\xff\xff\xff\xff\xff\xff\xef\x7f:(\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x00\x00\n\xe8\x11\n layer2/Wx_plus_b/pre_activations*\xc3\x11\t\x00\x00\x00\xe0\xbe\xb6"\xc0\x11\x00\x00\x00\x80\xebW,@\x19\x00\x00\x00\x00\x00@\x8f@!\x00\x00p\x1b\xd1Th\xc0)\xfe\x83\x96\x118\xa1\xc5@2\xc8\x08\xae\xd5\xa0\xf9M\xe0#\xc0\xb5\x93\x06@\xbb\x11"\xc0v@\x1d\xdd5m \xc0a/5L\xd6\xdd\x1d\xc0)\x88\xea\xd0\xc2&\x1b\xc0S\xaa\xa6\xa6\xdf\xae\x18\xc0\xee\xf7h:np\x16\xc0\xd8\xb2v\xa95f\x14\xc0!\xd1\x0e\x9av\x8b\x12\xc0{2\xf6.\xe0\xdb\x10\xc0\x81DK\xf8\x0b\xa7\x0e\xc0u\xf8r\xca\xad\xdd\x0b\xc0\r\x9c\xc5\xa0)U\t\xc0h\xbc\x105\x9a\x07\x07\xc0\xea6\x0fv\xa3\xef\x04\xc0w`<\x0ef\x08\x03\xc0\xe0\x11\xf1\x0ctM\x01\xc0\xdc\x94\xcd.\x8du\xff\xbfS\x87F\x13i\x99\xfc\xbfyL\xfa\xe2\xd3\xff\xf9\xbf\x85E\xcc\x9f\xc0\xa2\xf7\xbf3?\xe8\x1c\xaf|\xf5\xbf\xbaP\x8d1\x9f\x88\xf3\xbfcI\xc6\x15\x05\xc2\xf1\xbf+\x14\xfa\xcd\xbe$\xf0\xbf\x08\x99i\x19\x15Z\xed\xbf\x1e\x8b\x8e\xe8\xfb\xae\xea\xbf2g\xc7G\xfcA\xe8\xbf\x16uo\xfbp\r\xe6\xbf+Se*8\x0c\xe4\xbf\x0f\x1d\\l\xa79\xe2\xbf\xb01%\xee\x80\x91\xe0\xbf\xfa\xe5Z%\xd3\x1f\xde\xbfW\xe8i\xf3\xbfb\xdb\xbf\xda^\xbd\xaeh\xe5\xd8\xbf\x97\xca7\x13\x02\xa2\xd6\xbf\xcfCa\xb4G\x93\xd4\xbf_&A\xbbo\xb4\xd2\xbf\x9c\x0b\x81\xc1\x1f\x01\xd1\xbf\xedZ\xd3_\xc5\xea\xce\xbf\x05\xc7K\x11?\x1b\xcc\xbf\x90@\xb9\x0f"\x8d\xc9\xbf\xf7QK\x0e|:\xc7\xbf\x0fyD$\xe5\x1d\xc5\xbf\x82?>8s2\xc3\xbf\xd3\x968\x90\xaes\xc1\xbf!\xb5\xac\xd7\x0e\xbb\xbf\xbf\xa9\xa4(!\x99\xd8\xbc\xbf%g\xf6cE9\xba\xbf8/\x9aC\xf9\xd6\xb7\xbf\x04\x88\x00l(\xac\xb5\xbf\xa6{\x00\xa8\xc7\xb3\xb3\xbf\xae\x87]$A\xe9\xb1\xbf\xb5\xef\xf7\xc3iH\xb0\xbfI\x85\xab\xd8\xee\x9a\xad\xbf\xfc\xbe\xe1g\xf0\xe9\xaa\xbf\xcdg\x87\xbb\x94w\xa8\xbf\xd1/\xd8\x1e*>\xa6\xbf\x87v\xdc\xc9ab\x9e\xbf\xd7\xc8<\xa0A\x9f\x9b\xbf\xce"\x9eu\x03\xd4\x96\xbf^\x08\xa7S\xbd\xc0\x94\xbf-\xd4\xb1\x05\x8dL\x85\xbf\xe3\xc0\xfeJ\xdd\\\x83\xbf\x15o\x8c\xcf5sz\xbf*\x1fQH\xa5\x0bx\xbf\x19\xe9\xab\xe4\xda\x8b\x19\xbf\xd0vV\xa1R9\x17\xbf<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xedZ\xd3_\xc5\xea\xce?\x9c\x0b\x81\xc1\x1f\x01\xd1?_&A\xbbo\xb4\xd2?\xcfCa\xb4G\x93\xd4?\x97\xca7\x13\x02\xa2\xd6?\xda^\xbd\xaeh\xe5\xd8?W\xe8i\xf3\xbfb\xdb?\xfa\xe5Z%\xd3\x1f\xde?\xb01%\xee\x80\x91\xe0?\x0f\x1d\\l\xa79\xe2?+Se*8\x0c\xe4?\x16uo\xfbp\r\xe6?2g\xc7G\xfcA\xe8?\x1e\x8b\x8e\xe8\xfb\xae\xea?\x08\x99i\x19\x15Z\xed?+\x14\xfa\xcd\xbe$\xf0?cI\xc6\x15\x05\xc2\xf1?\xbaP\x8d1\x9f\x88\xf3?3?\xe8\x1c\xaf|\xf5?\x85E\xcc\x9f\xc0\xa2\xf7?yL\xfa\xe2\xd3\xff\xf9?S\x87F\x13i\x99\xfc?\xdc\x94\xcd.\x8du\xff?\xe0\x11\xf1\x0ctM\x01@w`<\x0ef\x08\x03@\xea6\x0fv\xa3\xef\x04@h\xbc\x105\x9a\x07\x07@\r\x9c\xc5\xa0)U\t@u\xf8r\xca\xad\xdd\x0b@\x81DK\xf8\x0b\xa7\x0e@{2\xf6.\xe0\xdb\x10@!\xd1\x0e\x9av\x8b\x12@\xd8\xb2v\xa95f\x14@\xee\xf7h:np\x16@S\xaa\xa6\xa6\xdf\xae\x18@)\x88\xea\xd0\xc2&\x1b@a/5L\xd6\xdd\x1d@v@\x1d\xdd5m @\xb5\x93\x06@\xbb\x11"@\xae\xd5\xa0\xf9M\xe0#@sQ\x97\x92"\xdd%@e\xa6\xd9:\xd9\x0c(@\xd6\x03\x89\x8d\x88t*@\xd2\x1d}h\xc9\x19-@\xff\xff\xff\xff\xff\xff\xef\x7f:\xc8\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00=@\x00\x00\x00\x00\x00\x00<@\x00\x00\x00\x00\x00\x00<@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00;@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x001@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x003@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\n\xda\x11\n\x12layer2/activations*\xc3\x11\t\x00\x00\x00\xe0\xbe\xb6"\xc0\x11\x00\x00\x00\x80\xebW,@\x19\x00\x00\x00\x00\x00@\x8f@!\x00\x00p\x1b\xd1Th\xc0)\xfe\x83\x96\x118\xa1\xc5@2\xc8\x08\xae\xd5\xa0\xf9M\xe0#\xc0\xb5\x93\x06@\xbb\x11"\xc0v@\x1d\xdd5m \xc0a/5L\xd6\xdd\x1d\xc0)\x88\xea\xd0\xc2&\x1b\xc0S\xaa\xa6\xa6\xdf\xae\x18\xc0\xee\xf7h:np\x16\xc0\xd8\xb2v\xa95f\x14\xc0!\xd1\x0e\x9av\x8b\x12\xc0{2\xf6.\xe0\xdb\x10\xc0\x81DK\xf8\x0b\xa7\x0e\xc0u\xf8r\xca\xad\xdd\x0b\xc0\r\x9c\xc5\xa0)U\t\xc0h\xbc\x105\x9a\x07\x07\xc0\xea6\x0fv\xa3\xef\x04\xc0w`<\x0ef\x08\x03\xc0\xe0\x11\xf1\x0ctM\x01\xc0\xdc\x94\xcd.\x8du\xff\xbfS\x87F\x13i\x99\xfc\xbfyL\xfa\xe2\xd3\xff\xf9\xbf\x85E\xcc\x9f\xc0\xa2\xf7\xbf3?\xe8\x1c\xaf|\xf5\xbf\xbaP\x8d1\x9f\x88\xf3\xbfcI\xc6\x15\x05\xc2\xf1\xbf+\x14\xfa\xcd\xbe$\xf0\xbf\x08\x99i\x19\x15Z\xed\xbf\x1e\x8b\x8e\xe8\xfb\xae\xea\xbf2g\xc7G\xfcA\xe8\xbf\x16uo\xfbp\r\xe6\xbf+Se*8\x0c\xe4\xbf\x0f\x1d\\l\xa79\xe2\xbf\xb01%\xee\x80\x91\xe0\xbf\xfa\xe5Z%\xd3\x1f\xde\xbfW\xe8i\xf3\xbfb\xdb\xbf\xda^\xbd\xaeh\xe5\xd8\xbf\x97\xca7\x13\x02\xa2\xd6\xbf\xcfCa\xb4G\x93\xd4\xbf_&A\xbbo\xb4\xd2\xbf\x9c\x0b\x81\xc1\x1f\x01\xd1\xbf\xedZ\xd3_\xc5\xea\xce\xbf\x05\xc7K\x11?\x1b\xcc\xbf\x90@\xb9\x0f"\x8d\xc9\xbf\xf7QK\x0e|:\xc7\xbf\x0fyD$\xe5\x1d\xc5\xbf\x82?>8s2\xc3\xbf\xd3\x968\x90\xaes\xc1\xbf!\xb5\xac\xd7\x0e\xbb\xbf\xbf\xa9\xa4(!\x99\xd8\xbc\xbf%g\xf6cE9\xba\xbf8/\x9aC\xf9\xd6\xb7\xbf\x04\x88\x00l(\xac\xb5\xbf\xa6{\x00\xa8\xc7\xb3\xb3\xbf\xae\x87]$A\xe9\xb1\xbf\xb5\xef\xf7\xc3iH\xb0\xbfI\x85\xab\xd8\xee\x9a\xad\xbf\xfc\xbe\xe1g\xf0\xe9\xaa\xbf\xcdg\x87\xbb\x94w\xa8\xbf\xd1/\xd8\x1e*>\xa6\xbf\x87v\xdc\xc9ab\x9e\xbf\xd7\xc8<\xa0A\x9f\x9b\xbf\xce"\x9eu\x03\xd4\x96\xbf^\x08\xa7S\xbd\xc0\x94\xbf-\xd4\xb1\x05\x8dL\x85\xbf\xe3\xc0\xfeJ\xdd\\\x83\xbf\x15o\x8c\xcf5sz\xbf*\x1fQH\xa5\x0bx\xbf\x19\xe9\xab\xe4\xda\x8b\x19\xbf\xd0vV\xa1R9\x17\xbf<DKc\x97\xeeT?\xdc\x97\x9fSs\x06W?ei\x10S\xcem\x87?#\xa7+(\x96\xc5\x89?\xea\x06\x1c\xa0\xb1&\x91?\x9b\x07Rc\xc3\xdd\x92?^\x08\xa7S\xbd\xc0\x94?\xce"\x9eu\x03\xd4\x96?}Y\x944j\x1c\x99?\xd1/\xd8\x1e*>\xa6?\xcdg\x87\xbb\x94w\xa8?I\x85\xab\xd8\xee\x9a\xad?\xb5\xef\xf7\xc3iH\xb0?\xae\x87]$A\xe9\xb1?\x04\x88\x00l(\xac\xb5?8/\x9aC\xf9\xd6\xb7?%g\xf6cE9\xba?\xa9\xa4(!\x99\xd8\xbc?\xd3\x968\x90\xaes\xc1?\x82?>8s2\xc3?\x0fyD$\xe5\x1d\xc5?\xf7QK\x0e|:\xc7?\x90@\xb9\x0f"\x8d\xc9?\x05\xc7K\x11?\x1b\xcc?\xedZ\xd3_\xc5\xea\xce?\x9c\x0b\x81\xc1\x1f\x01\xd1?_&A\xbbo\xb4\xd2?\xcfCa\xb4G\x93\xd4?\x97\xca7\x13\x02\xa2\xd6?\xda^\xbd\xaeh\xe5\xd8?W\xe8i\xf3\xbfb\xdb?\xfa\xe5Z%\xd3\x1f\xde?\xb01%\xee\x80\x91\xe0?\x0f\x1d\\l\xa79\xe2?+Se*8\x0c\xe4?\x16uo\xfbp\r\xe6?2g\xc7G\xfcA\xe8?\x1e\x8b\x8e\xe8\xfb\xae\xea?\x08\x99i\x19\x15Z\xed?+\x14\xfa\xcd\xbe$\xf0?cI\xc6\x15\x05\xc2\xf1?\xbaP\x8d1\x9f\x88\xf3?3?\xe8\x1c\xaf|\xf5?\x85E\xcc\x9f\xc0\xa2\xf7?yL\xfa\xe2\xd3\xff\xf9?S\x87F\x13i\x99\xfc?\xdc\x94\xcd.\x8du\xff?\xe0\x11\xf1\x0ctM\x01@w`<\x0ef\x08\x03@\xea6\x0fv\xa3\xef\x04@h\xbc\x105\x9a\x07\x07@\r\x9c\xc5\xa0)U\t@u\xf8r\xca\xad\xdd\x0b@\x81DK\xf8\x0b\xa7\x0e@{2\xf6.\xe0\xdb\x10@!\xd1\x0e\x9av\x8b\x12@\xd8\xb2v\xa95f\x14@\xee\xf7h:np\x16@S\xaa\xa6\xa6\xdf\xae\x18@)\x88\xea\xd0\xc2&\x1b@a/5L\xd6\xdd\x1d@v@\x1d\xdd5m @\xb5\x93\x06@\xbb\x11"@\xae\xd5\xa0\xf9M\xe0#@sQ\x97\x92"\xdd%@e\xa6\xd9:\xd9\x0c(@\xd6\x03\x89\x8d\x88t*@\xd2\x1d}h\xc9\x19-@\xff\xff\xff\xff\xff\xff\xef\x7f:\xc8\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00=@\x00\x00\x00\x00\x00\x00<@\x00\x00\x00\x00\x00\x00<@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x00;@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x001@\x00\x00\x00\x00\x00\x00:@\x00\x00\x00\x00\x00\x002@\x00\x00\x00\x00\x00\x007@\x00\x00\x00\x00\x00\x003@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x10@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00\x14@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00\x1c@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x000@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00"@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00.@\x00\x00\x00\x00\x00\x00&@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00,@\x00\x00\x00\x00\x00\x00(@\x00\x00\x00\x00\x00\x00*@\x00\x00\x00\x00\x00\x00$@\x00\x00\x00\x00\x00\x00 @\x00\x00\x00\x00\x00\x00\x00@\x00\x00\x00\x00\x00\x00\x18@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\x08@\x00\x00\x00\x00\x00\x00\xf0?\x00\x00\x00\x00\x00\x00\x00\x00\n\x16\n\x0fcross_entropy_1\x15\xc3)\xc1>\n\x11\n\naccuracy_1\x15\xaeGa?')
