"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const nested_stack_1 = require("../lib/nested-stack");
module.exports = {
    'fails if defined as a root'(test) {
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(undefined, 'boom'), /Nested stacks cannot be defined as a root construct/);
        test.done();
    },
    'fails if defined without a parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const group = new core_1.Construct(app, 'group');
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(app, 'boom'), /must be defined within scope of another non-nested stack/);
        test.throws(() => new nested_stack_1.NestedStack(group, 'bam'), /must be defined within scope of another non-nested stack/);
        test.done();
    },
    'can be defined as a direct child or an indirect child of a Stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        // THEN
        new nested_stack_1.NestedStack(parent, 'direct');
        new nested_stack_1.NestedStack(new core_1.Construct(parent, 'group'), 'indirect');
        test.done();
    },
    'nested stack is not synthesized as a stack artifact into the assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent-stack');
        new nested_stack_1.NestedStack(parentStack, 'nested-stack');
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.artifacts.length, 2);
        test.done();
    },
    'the template of the nested stack is synthesized into the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        const template = JSON.parse(fs.readFileSync(path.join(assembly.directory, `${nested.node.uniqueId}.nested.template.json`), 'utf-8'));
        test.deepEqual(template, {
            Resources: {
                ResourceInNestedStack: {
                    Type: 'AWS::Resource::Nested',
                },
            },
        });
        test.done();
    },
    'file asset metadata is associated with the parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.getStackByName(parent.stackName).assets, [{
                path: 'parentstacknestedstack844892C0.nested.template.json',
                id: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                packaging: 'file',
                sourceHash: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                s3BucketParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345',
                s3KeyParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                artifactHashParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7',
            }]);
        test.done();
    },
    'aws::cloudformation::stack is synthesized in the parent scope'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // THEN
        const assembly = app.synth();
        // assembly has one stack (the parent)
        test.deepEqual(assembly.stacks.length, 1);
        // but this stack has an asset that points to the synthesized template
        test.deepEqual(assembly.stacks[0].assets[0].path, 'parentstacknestedstack844892C0.nested.template.json');
        // the template includes our resource
        const filePath = path.join(assembly.directory, assembly.stacks[0].assets[0].path);
        test.deepEqual(JSON.parse(fs.readFileSync(filePath).toString('utf-8')), {
            Resources: { ResourceInNestedStack: { Type: 'AWS::Resource::Nested' } },
        });
        // the parent template includes the parameters and the nested stack resource which points to the s3 url
        assert_1.expect(parent).toMatch({
            Resources: {
                nestedstackNestedStacknestedstackNestedStackResource71CDD241: {
                    Type: 'AWS::CloudFormation::Stack',
                    Properties: {
                        TemplateURL: {
                            'Fn::Join': [
                                '',
                                [
                                    'https://s3.',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                    '/',
                                    {
                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345',
                                    },
                                    '/',
                                    {
                                        'Fn::Select': [
                                            0,
                                            {
                                                'Fn::Split': [
                                                    '||',
                                                    {
                                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '||',
                                                    {
                                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            ],
                        },
                    },
                },
            },
            Parameters: {
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345: {
                    Type: 'String',
                    Description: 'S3 bucket for asset "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6: {
                    Type: 'String',
                    Description: 'S3 key for asset version "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7: {
                    Type: 'String',
                    Description: 'Artifact hash for asset "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
            },
        });
        test.done();
    },
    'Stack.of()'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                this.stackOfChild = core_1.Stack.of(param);
            }
        }
        const parent = new core_1.Stack();
        const nested = new MyNestedStack(parent, 'nested');
        test.ok(nested.stackOfChild === nested);
        test.ok(core_1.Stack.of(nested) === nested);
        test.done();
    },
    'references within the nested stack are not reported as cross stack references'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                new core_1.CfnResource(this, 'resource', {
                    type: 'My::Resource',
                    properties: {
                        SomeProp: param.valueAsString,
                    },
                });
            }
        }
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent');
        new MyNestedStack(parent, 'nested');
        // references are added during "prepare"
        const assembly = app.synth();
        test.deepEqual(assembly.stacks.length, 1);
        test.deepEqual(assembly.stacks[0].dependencies, []);
        test.done();
    },
    'references to a resource from the parent stack in a nested stack is translated into a cfn parameter'(test) {
        // WHEN
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id, resourceFromParent) {
                super(scope, id);
                new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                    properties: {
                        ReferenceToResourceInParentStack: resourceFromParent.ref,
                    },
                });
                new core_1.CfnResource(this, 'resource2', {
                    type: 'My::Resource::2',
                    properties: {
                        Prop1: resourceFromParent.getAtt('Attr'),
                        Prop2: resourceFromParent.ref,
                    },
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const resource = new core_1.CfnResource(parentStack, 'parent-resource', { type: 'AWS::Parent::Resource' });
        const nested = new MyNestedStack(parentStack, 'nested', resource);
        // THEN
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: {
                    Type: 'AWS::Child::Resource',
                    Properties: { ReferenceToResourceInParentStack: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' } },
                },
                resource2: {
                    Type: 'My::Resource::2',
                    Properties: {
                        Prop1: { Ref: 'referencetoparentparentresourceD56EA8F7Attr' },
                        Prop2: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' },
                    },
                },
            },
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: { Type: 'String' },
                referencetoparentparentresourceD56EA8F7Attr: { Type: 'String' },
            },
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: {
                    Ref: 'parentresource',
                },
                referencetoparentparentresourceD56EA8F7Attr: {
                    'Fn::GetAtt': [
                        'parentresource',
                        'Attr',
                    ],
                },
            },
        }));
        test.done();
    },
    'references to a resource in the nested stack in the parent is translated into a cfn output'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                this.resourceFromChild = new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const nested = new MyNestedStack(parentStack, 'nested');
        new core_1.CfnResource(parentStack, 'another-parent-resource', {
            type: 'AWS::Parent::Resource',
            properties: {
                RefToResourceInNestedStack: nested.resourceFromChild.ref,
            },
        });
        // references are added during "prepare"
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: { Type: 'AWS::Child::Resource' },
            },
            Outputs: {
                parentnestedresource4D680677Ref: { Value: { Ref: 'resource' } },
            },
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::Parent::Resource', {
            RefToResourceInNestedStack: {
                'Fn::GetAtt': [
                    'nestedNestedStacknestedNestedStackResource3DD143BF',
                    'Outputs.parentnestedresource4D680677Ref',
                ],
            },
        }));
        test.done();
    },
    'nested stack references a resource from another non-nested stack (not the parent)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInStack2 = new core_1.CfnResource(stack2, 'ResourceInStack2', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack1', {
            type: 'Nested::Resource',
            properties: {
                RefToSibling: resourceInStack2.getAtt('MyAttribute'),
            },
        });
        // THEN
        const assembly = app.synth();
        // producing stack should have an export
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: { Type: 'MyResource' },
            },
            Outputs: {
                ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009: {
                    Value: { 'Fn::GetAtt': ['ResourceInStack2', 'MyAttribute'] },
                    Export: { Name: 'Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009' },
                },
            },
        });
        // nested stack uses Fn::ImportValue like normal
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack1: {
                    Type: 'Nested::Resource',
                    Properties: {
                        RefToSibling: {
                            'Fn::ImportValue': 'Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009',
                        },
                    },
                },
            },
        });
        // verify a depedency was established between the parents
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 1);
        test.deepEqual(stack2Artifact.dependencies.length, 0);
        test.same(stack1Artifact.dependencies[0], stack2Artifact);
        test.done();
    },
    'nested stack within a nested stack references a resource in a sibling top-level stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const consumerTopLevel = new core_1.Stack(app, 'ConsumerTopLevel');
        const consumerNested1 = new nested_stack_1.NestedStack(consumerTopLevel, 'ConsumerNested1');
        const consumerNested2 = new nested_stack_1.NestedStack(consumerNested1, 'ConsumerNested2');
        const producerTopLevel = new core_1.Stack(app, 'ProducerTopLevel');
        const producer = new core_1.CfnResource(producerTopLevel, 'Producer', { type: 'Producer' });
        // WHEN
        new core_1.CfnResource(consumerNested2, 'Consumer', {
            type: 'Consumer',
            properties: {
                Ref: producer.ref,
            },
        });
        // THEN
        const manifest = app.synth();
        const consumerDeps = manifest.getStackArtifact(consumerTopLevel.artifactId).dependencies.map(d => d.id);
        test.deepEqual(consumerDeps, ['ProducerTopLevel']);
        test.done();
    },
    'another non-nested stack takes a reference on a resource within the nested stack (the parent exports)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInNestedStack = new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(stack2, 'ResourceInStack2', {
            type: 'JustResource',
            properties: {
                RefToSibling: resourceInNestedStack.getAtt('MyAttribute'),
            },
        });
        // THEN
        const assembly = app.synth();
        // nested stack should output this value as if it was referenced by the parent (without the export)
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack: {
                    Type: 'MyResource',
                },
            },
            Outputs: {
                Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute: {
                    Value: {
                        'Fn::GetAtt': [
                            'ResourceInNestedStack',
                            'MyAttribute',
                        ],
                    },
                },
            },
        });
        // parent stack (stack1) should export this value
        test.deepEqual(assembly.getStackByName(stack1.stackName).template.Outputs, {
            ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3: {
                Value: { 'Fn::GetAtt': ['NestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305B', 'Outputs.Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute'] },
                Export: { Name: 'Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3' },
            },
        });
        // consuming stack should use ImportValue to import the value from the parent stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: {
                    Type: 'JustResource',
                    Properties: {
                        RefToSibling: {
                            'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3',
                        },
                    },
                },
            },
        });
        test.deepEqual(assembly.stacks.length, 2);
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 0);
        test.deepEqual(stack2Artifact.dependencies.length, 1);
        test.same(stack2Artifact.dependencies[0], stack1Artifact);
        test.done();
    },
    'references between sibling nested stacks should output from one and getAtt from the other'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'Parent');
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(parent, 'Nested2');
        const resource1 = new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource1' });
        // WHEN
        new core_1.CfnResource(nested2, 'Resource2', {
            type: 'Resource2',
            properties: {
                RefToResource1: resource1.ref,
            },
        });
        // THEN
        app.synth();
        // producing nested stack
        assert_1.expect(nested1).toMatch({
            Resources: {
                Resource1: {
                    Type: 'Resource1',
                },
            },
            Outputs: {
                ParentNested1Resource15F3F0657Ref: {
                    Value: {
                        Ref: 'Resource1',
                    },
                },
            },
        });
        // consuming nested stack
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: {
                    Type: 'Resource2',
                    Properties: {
                        RefToResource1: {
                            Ref: 'referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref',
                        },
                    },
                },
            },
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    Type: 'String',
                },
            },
        });
        // parent
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    'Fn::GetAtt': [
                        'Nested1NestedStackNested1NestedStackResourceCD0AD36B',
                        'Outputs.ParentNested1Resource15F3F0657Ref',
                    ],
                },
            },
        }));
        test.done();
    },
    'stackId returns AWS::StackId when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackId: nested.stackId },
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackId: { Ref: 'AWS::StackId' },
        }));
        test.done();
    },
    'stackId returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackId: nested.stackId },
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackId: { Ref: 'NestedStackNestedStackNestedStackNestedStackResourceB70834FD' },
        }));
        test.done();
    },
    'stackName returns AWS::StackName when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackName: nested.stackName },
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackName: { Ref: 'AWS::StackName' },
        }));
        test.done();
    },
    'stackName returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackName: nested.stackName },
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackName: {
                'Fn::Select': [
                    1,
                    {
                        'Fn::Split': [
                            '/',
                            {
                                Ref: 'NestedStackNestedStackNestedStackNestedStackResourceB70834FD',
                            },
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    '"account", "region" and "environment" are all derived from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // THEN
        test.deepEqual(nested.environment, parent.environment);
        test.deepEqual(nested.account, parent.account);
        test.deepEqual(nested.region, parent.region);
        test.done();
    },
    'double-nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'stack');
        // WHEN
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(nested1, 'Nested2');
        new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource::1' });
        new core_1.CfnResource(nested2, 'Resource2', { type: 'Resource::2' });
        // THEN
        const assembly = app.synth();
        // nested2 is a "leaf", so it's just the resource
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: { Type: 'Resource::2' },
            },
        });
        // nested1 wires the nested2 template through parameters, so we expect those
        assert_1.expect(nested1).to(assert_1.haveResource('Resource::1'));
        const nested2Template = assert_1.SynthUtils.toCloudFormation(nested1);
        test.deepEqual(nested2Template.Parameters, {
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Type: 'String' },
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Type: 'String' },
        });
        // parent stack should have two sets of parameters. one for the first nested stack and the second
        // for the second nested stack, passed in as parameters to the first
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        test.deepEqual(template.Parameters, {
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6: { Type: 'String', Description: 'S3 bucket for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA: { Type: 'String', Description: 'S3 key for asset version "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cArtifactHash7DC546E0: { Type: 'String', Description: 'Artifact hash for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3Bucket76ACFB38: { Type: 'String', Description: 'S3 bucket for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3VersionKey04162EF1: { Type: 'String', Description: 'S3 key for asset version "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfArtifactHashF227ADD3: { Type: 'String', Description: 'Artifact hash for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
        });
        // proxy asset params to nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Ref: 'AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6' },
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Ref: 'AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA' },
            },
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'assets within nested stacks are proxied from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack');
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        const asset = new s3_assets.Asset(nested, 'asset', {
            path: path.join(__dirname, 'asset-fixture.txt'),
        });
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: {
                AssetBucket: asset.s3BucketName,
                AssetKey: asset.s3ObjectKey,
            },
        });
        // THEN
        const assembly = app.synth();
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        // two sets of asset parameters: one for the nested stack itself and one as a proxy for the asset within the stack
        test.deepEqual(template.Parameters, {
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637: { Type: 'String', Description: 'S3 bucket for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2: { Type: 'String', Description: 'S3 key for asset version "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281ArtifactHash373B14D2: { Type: 'String', Description: 'Artifact hash for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3Bucket3C4265E9: { Type: 'String', Description: 'S3 bucket for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3VersionKey8E981535: { Type: 'String', Description: 'S3 key for asset version "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71ArtifactHash45A28583: { Type: 'String', Description: 'Artifact hash for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
        });
        // asset proxy parameters are passed to the nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3Bucket82C55B96Ref: { Ref: 'AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637' },
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyA43C3CC6Ref: { Ref: 'AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2' },
            },
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'docker image assets are wired through the top-level stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'my-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        // WHEN
        const location = nested.addDockerImageAsset({
            directoryName: 'my-image',
            dockerBuildArgs: { key: 'value', boom: 'bam' },
            dockerBuildTarget: 'buildTarget',
            sourceHash: 'hash-of-source',
        });
        // use the asset, so the parameters will be wired.
        new sns.Topic(nested, 'MyTopic', {
            displayName: `image location is ${location.imageUri}`,
        });
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackArtifact(parent.artifactId).assets, [
            {
                repositoryName: 'aws-cdk/assets',
                imageTag: 'hash-of-source',
                id: 'hash-of-source',
                packaging: 'container-image',
                path: 'my-image',
                sourceHash: 'hash-of-source',
                buildArgs: { key: 'value', boom: 'bam' },
                target: 'buildTarget',
            },
            {
                path: 'mystacknestedstackFAE12FB5.nested.template.json',
                id: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                packaging: 'file',
                sourceHash: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                s3BucketParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3Bucket67A749F8',
                s3KeyParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3VersionKeyE1E6A8D4',
                artifactHashParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5ArtifactHash0AEDBE8A',
            },
        ]);
        test.done();
    },
    'metadata defined in nested stacks is reported at the parent stack level in the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App({ stackTraces: false });
        const parent = new core_1.Stack(app, 'parent');
        const child = new core_1.Stack(parent, 'child');
        const nested = new nested_stack_1.NestedStack(child, 'nested');
        const resource = new core_1.CfnResource(nested, 'resource', { type: 'foo' });
        // WHEN
        resource.node.addMetadata('foo', 'bar');
        // THEN: the first non-nested stack records the assembly metadata
        const asm = app.synth();
        test.deepEqual(asm.stacks.length, 2); // only one stack is defined as an artifact
        test.deepEqual(asm.getStackByName(parent.stackName).findMetadataByType('foo'), []);
        test.deepEqual(asm.getStackByName(child.stackName).findMetadataByType('foo'), [
            {
                path: '/parent/child/nested/resource',
                type: 'foo',
                data: 'bar',
            },
        ]);
        test.done();
    },
    'referencing attributes with period across stacks'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'nested');
        const consumed = new core_1.CfnResource(nested, 'resource-in-nested', { type: 'CONSUMED' });
        // WHEN
        new core_1.CfnResource(parent, 'resource-in-parent', {
            type: 'CONSUMER',
            properties: {
                ConsumedAttribute: consumed.getAtt('Consumed.Attribute'),
            },
        });
        // THEN
        assert_1.expect(nested).toMatch({
            Resources: {
                resourceinnested: {
                    Type: 'CONSUMED',
                },
            },
            Outputs: {
                nestedresourceinnested59B1F01CConsumedAttribute: {
                    Value: {
                        'Fn::GetAtt': [
                            'resourceinnested',
                            'Consumed.Attribute',
                        ],
                    },
                },
            },
        });
        assert_1.expect(parent).to(assert_1.haveResource('CONSUMER', {
            ConsumedAttribute: {
                'Fn::GetAtt': [
                    'nestedNestedStacknestedNestedStackResource3DD143BF',
                    'Outputs.nestedresourceinnested59B1F01CConsumedAttribute',
                ],
            },
        }));
        test.done();
    },
    'missing context in nested stack is reported if the context is not available'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        const nestedStack = new nested_stack_1.NestedStack(stack, 'nested');
        const provider = 'availability-zones';
        const expectedKey = core_1.ContextProvider.getKey(nestedStack, {
            provider,
        }).key;
        // WHEN
        core_1.ContextProvider.getValue(nestedStack, {
            provider,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        });
        // THEN: missing context is reported in the cloud assembly
        const asm = app.synth();
        const missing = asm.manifest.missing;
        test.ok(missing && missing.find(m => {
            return (m.key === expectedKey);
        }));
        test.done();
    },
    'references to a resource from a deeply nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const top = new core_1.Stack(app, 'stack');
        const topLevel = new core_1.CfnResource(top, 'toplevel', { type: 'TopLevel' });
        const nested1 = new nested_stack_1.NestedStack(top, 'nested1');
        const nested2 = new nested_stack_1.NestedStack(nested1, 'nested2');
        // WHEN
        new core_1.CfnResource(nested2, 'refToTopLevel', {
            type: 'BottomLevel',
            properties: { RefToTopLevel: topLevel.ref },
        });
        // THEN
        assert_1.expect(top).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostackAssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3Bucket5DA5D2E7Ref: {
                    Ref: 'AssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3BucketDD4D96B5',
                },
                referencetostackAssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3VersionKey8FBE5C12Ref: {
                    Ref: 'AssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3VersionKey83E381F3',
                },
                referencetostacktoplevelBB16BF13Ref: {
                    Ref: 'toplevel',
                },
            },
        }));
        assert_1.expect(nested1).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostacktoplevelBB16BF13Ref: {
                    Ref: 'referencetostacktoplevelBB16BF13Ref',
                },
            },
        }));
        assert_1.expect(nested2).to(assert_1.matchTemplate({
            Resources: {
                refToTopLevel: {
                    Type: 'BottomLevel',
                    Properties: {
                        RefToTopLevel: {
                            Ref: 'referencetostacktoplevelBB16BF13Ref',
                        },
                    },
                },
            },
            Parameters: {
                referencetostacktoplevelBB16BF13Ref: {
                    Type: 'String',
                },
            },
        }));
        test.done();
    },
    'bottom nested stack consumes value from a top-level stack through a parameter in a middle nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const top = new core_1.Stack(app, 'Grandparent');
        const middle = new nested_stack_1.NestedStack(top, 'Parent');
        const bottom = new nested_stack_1.NestedStack(middle, 'Child');
        const resourceInGrandparent = new core_1.CfnResource(top, 'ResourceInGrandparent', { type: 'ResourceInGrandparent' });
        // WHEN
        new core_1.CfnResource(bottom, 'ResourceInChild', {
            type: 'ResourceInChild',
            properties: {
                RefToGrandparent: resourceInGrandparent.ref,
            },
        });
        // THEN
        // this is the name allocated for the parameter that's propagated through
        // the hierarchy.
        const paramName = 'referencetoGrandparentResourceInGrandparent010E997ARef';
        // child (bottom) references through a parameter.
        assert_1.expect(bottom).toMatch({
            Resources: {
                ResourceInChild: {
                    Type: 'ResourceInChild',
                    Properties: {
                        RefToGrandparent: { Ref: paramName },
                    },
                },
            },
            Parameters: {
                [paramName]: { Type: 'String' },
            },
        });
        // the parent (middle) sets the value of this parameter to be a reference to another parameter
        assert_1.expect(middle).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                [paramName]: { Ref: paramName },
            },
        }));
        // grandparent (top) assigns the actual value to the parameter
        assert_1.expect(top).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                [paramName]: { Ref: 'ResourceInGrandparent' },
                // these are for the asset of the bottom nested stack
                referencetoGrandparentAssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3Bucket06EEE58DRef: {
                    Ref: 'AssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3Bucket01877C2E',
                },
                referencetoGrandparentAssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3VersionKeyD3B04909Ref: {
                    Ref: 'AssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3VersionKey5765F084',
                },
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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