"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
class TestCustomResource extends cdk.Construct {
    constructor(scope, id, opts = {}) {
        super(scope, id);
        const singletonLambda = new lambda.SingletonFunction(this, 'Lambda', {
            uuid: 'TestCustomResourceProvider',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        this.resource = new lib_1.CustomResource(this, 'Resource', {
            ...opts,
            provider: lib_1.CustomResourceProvider.fromLambda(singletonLambda),
        });
    }
}
module.exports = nodeunit_1.testCase({
    'custom resources honor removalPolicy': {
        'unspecified (aka .Destroy)'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Destroy'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.DESTROY });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Retain'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.RETAIN });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
    },
    'custom resource is added twice, lambda is added once'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        // WHEN
        new TestCustomResource(stack, 'Custom1');
        new TestCustomResource(stack, 'Custom2');
        // THEN
        assert_1.expect(stack).toMatch({
            'Resources': {
                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'Service': 'lambda.amazonaws.com',
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'ManagedPolicyArns': [
                            { 'Fn::Join': ['', [
                                        'arn:', { 'Ref': 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole'
                                    ]] },
                        ],
                    },
                },
                'SingletonLambdaTestCustomResourceProviderA9255269': {
                    'Type': 'AWS::Lambda::Function',
                    'Properties': {
                        'Code': {
                            'ZipFile': 'def hello(): pass',
                        },
                        'Handler': 'index.hello',
                        'Role': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                                'Arn',
                            ],
                        },
                        'Runtime': 'python2.7',
                        'Timeout': 300,
                    },
                    'DependsOn': [
                        'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                    ],
                },
                'Custom1D319B237': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
                'Custom2DD5FB44D': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
            },
        });
        test.done();
    },
    'custom resources can specify a resource type that starts with Custom::'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            resourceType: 'Custom::MyCustomResourceType',
            provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::MyCustomResourceType'));
        test.done();
    },
    'fails if custom resource type is invalid': {
        'does not start with "Custom::"'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'NoCustom::MyCustomResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type must begin with "Custom::"/);
            test.done();
        },
        'has invalid characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::My Custom?ResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type name can only include alphanumeric characters and/);
            test.done();
        },
        'is longer than 60 characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::0123456789012345678901234567890123456789012345678901234567891',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type length > 60/);
            test.done();
        },
    },
    '.ref returns the intrinsic reference (physical name)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const res = new TestCustomResource(stack, 'myResource');
        // THEN
        test.deepEqual(stack.resolve(res.resource.ref), { Ref: 'myResourceC6A188A9' });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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