# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['__init__']
install_requires = \
['numpy>=1.19.0,<2.0.0']

setup_kwargs = {
    'name': 'polyagamma',
    'version': '1.3.1',
    'description': 'Efficiently generate samples from the Polya-Gamma distribution using a NumPy/SciPy compatible interface.',
    'long_description': '# Polya-Gamma\n[![PyPI - Wheel][4]](https://pypi.org/project/polyagamma/#files)\n[![PyPI][5]](https://pypi.org/project/polyagamma/#history)\n[![PyPI - License][6]](https://github.com/zoj613/polyagamma/blob/main/LICENSE)\n[![CircleCI][7]](https://circleci.com/gh/zoj613/polyagamma/)\n[![Codecov][8]](https://codecov.io/gh/zoj613/polyagamma/)\n\n\nEfficiently generate samples from the Polya-Gamma distribution using a NumPy/SciPy compatible interface.\n\n## Why?\n\nIf you are reading this, you probably have already used the [pypolyagamma][9] package before. It is\na great package that I have also used in the past, however I encountered several issues:\n- Generating an array of samples is awkward because it requires using a list comprehension\n  if parameter values are scalars or have pre-allocated arrays of a known size to pass for both\n  the parameters and the output array. Moreover, broadcasting of input is not supported and thus\n  requiring the user to write another layer to support it.\n- It requires extra effort to be used in multiprocessing because pickling of the\n  sampler is not supported.\n- There is no parameter validation supported meaning it is easy to get the wrong samples if\n  you do not check the inputs manually.\n- The sampling API is very different from the ones used by popular packages like numpy/scipy,\n  making it harder to just "plug-n-play" in existing code bases.\n- It does not allow passing in an instance of a `np.random.RandomState` or `np.random.Generator`\n  for seeding, requiring extra effort when changing the seed if used in a larger code base.\n- The C++ code wrapped by the package is GPLv3 licensed, making it difficult to\n  use the source code in a project that prefers licenses like MIT/Apache/BSD.\n\nThe above issues are the reason why this package exists. And the aim of `polyagamma` is to "fix" them.\n\n\n## Features\n- Input parameters can be scalars, arrays or both; allowing for easy generation\nof multi-dimensional samples without specifying the size.\n- Input validation is done internally with clear error messages upon failure.\n- It is flexible and allows the user to sample using one of 4 available algorithms.\n- Implements functions to compute the CDF and density of the distribution as well\n  as their logarithms.\n- Random number generation is thread safe.\n- The functional API resembles that of common numpy/scipy functions, therefore making it easy to plugin to\nexisting libraries.\n- `polyagamma` is optimized for performance and tests show that it is faster\n  than other implementations.\n- Pre-built wheels are provided for easy installation on `x86_64` linux systems.\n\n\n## Examples\n\n### Python\n\n```python\nimport array\nimport numpy as np\nfrom polyagamma import random_polyagamma\n\n# generate a PG(1, 0) sample\no = random_polyagamma()\n\n# Get a 5 by 10 array of PG(1, 2) variates.\no = random_polyagamma(z=2, size=(5, 10))\n\n# Pass sequences as input. Numpy\'s broadcasting rules apply here.\nz = [[1.5, 2, -0.75, 4, 5],\n     [9.5, -8, 7, 6, -0.9]]\no = random_polyagamma(1, z)\n\n# We can pass an output array using the `out` parameter. It does not have to be\n# a numpy array. it can be any object that implements the array or buffer protocols.\n# As long as its type is 64bit float, contiguous in memory and aligned (e.g. Python\'s array object).\nnumpy_out = np.empty(5)\narray_out = array.array(\'d\', [0] * 5)\nrandom_polyagamma(out=numpy_out)\nprint(numpy_out)\nrandom_polyagamma(out=array_out)\nprint(array_out)\n\n# one can choose a sampling method from {devroye, alternate, gamma, saddle}.\n# If not given, the default behaviour is a hybrid sampler that picks the most\n# efficient method based on the input values.\no = random_polyagamma(method="saddle")\n\n# one can also use an existing instance of `numpy.random.Generator` as a parameter.\n# This is useful to reproduce samples generated via a given seed.\nrng = np.random.default_rng(12345)\no = random_polyagamma(random_state=rng)\n\n# If one is using a `numpy.random.RandomState` instance instead of the `Generator`\n# class, the object\'s underlying bitgenerator can be passed as the value of random_state\nbit_gen = np.random.RandomState(12345)._bit_generator\no = random_polyagamma(random_state=bit_gen)\n\n# When passing a large input array for the shape parameter `h`, parameter value\n# validation checks can be disabled if the values are guaranteed to be positive\n# to avoid some overhead, which may boost performance.\nlarge_h = np.ones(1000000)\no = random_polyagamma(large_h, disable_checks=True)\n```\nFunctions to compute the density and CDF are available. Broadcasting of input is supported.\n```python\nfrom polyagamma import polyagamma_pdf, polyagamma_cdf\n\n>>> polyagamma_pdf(0.1)\n# 3.613955566329298\n>>> polyagamma_cdf([1, 2], h=2, z=1)\n# array([0.95637847, 0.99963397])\n>>> polyagamma_pdf([2, 0.1], h=[[1, 2], [3, 4]], return_log=True)\n# array([[   -8.03172733,  -489.17101125]\n#        [   -3.82023942, -1987.09156971]])\n>>> polyagamma_cdf(4, z=[-100, 0, 2], return_log=True)\n# array([ 3.72007598e-44, -3.40628215e-09, -1.25463528e-12])\n```\n\n### Cython\nThe package also provides functions that can be imported in cython modules. They are:\n- `random_polyagamma`\n- `random_polyagamma_fill`\n- `random_polyagamma_fill2`\n\nRefer to the [pgm_random.h](./include/pgm_random.h) header file for more info about the\nfunction signatures. Below is an example of how these functions can be used.\n\n```cython\nfrom cpython.pycapsule cimport PyCapsule_GetPointer\nfrom polyagamma cimport random_polyagamma_fill, DEVROYE\nfrom numpy.random cimport bitgen_t\nimport numpy as np\n\n# assuming there exists an instance of the Generator class called `rng`.\nbitgenerator = rng._bit_generator\n# get pointer to the underlying bitgenerator struct\ncdef bitgen_t* bitgen = <bitgen_t*>PyCapsule_GetPointer(bitgenerator.capsule, "BitGenerator")\n# set distribution parameters\ncdef double h = 1, z = 0\n# get a memory view of the array to store samples in\ncdef double[:] out = np.empty(300)\nwith bitgenerator.lock, nogil:\n    random_polyagamma_fill(bitgen, h, z, DEVROYE, <size_t>out.shape[0], &out[0])\nprint(out.base)\n...\n```\n\n### C\nFor an example of how to use `polyagamma` in a C program, see [here][1].\n\n\n## Dependencies\n- Numpy >= 1.19.0\n\n\n## Installation\nTo get the latest version of the package, one can install it by downloading the wheel/source distribution \nfrom the [releases][3] page, or using `pip` with the following shell command:\n```shell\n$ pip install polyagamma\n```\nTo install the latest pre-release version, use:\n```shell\n$ pip install --pre -U polyagamma\n```\nAlternatively, once can install from source by cloning the repo. This requires an installation of [poetry][2]\nand the following shell commands:\n```shell\n$ git clone https://github.com/zoj613/polya-gamma.git\n$ cd polya-gamma/\n$ poetry install\n# add package to python\'s path\n$ export PYTHONPATH=$PWD:$PYTHONPATH \n```\n\n\n## Benchmarks\n\nBelow are runtime plots of 20000 samples generated for various values of `h` \nand `z`, using each method. We restrict `h` to integer values to accomodate the \n`devroye` method, which cannot be used for non-integer `h`. The version of the\npackage used to generate them is `v1.3.1`.\n\n![](./scripts/img/perf_methods_0.0.svg) | ![](./scripts/img/perf_methods_2.5.svg)\n| --- | --- |\n\n![](./scripts/img/perf_methods_5.0.svg) | ![](./scripts/img/perf_methods_10.0.svg)\n| --- | --- |\n\nGenerally:\n- The `gamma` method is slowest and should be avoided in cases where speed is paramount.\n- For `h >= 8`, the `saddle` method is the fastest for any value of `z`.\n- For `0 <= z <= 1` and integer `h <= 4`, the `devroye` method should be preferred.\n- For `z > 1` and `1 < h < 8`, the `alternate` method is the most efficient.\n- For `h > 50` (or any value large enough), the normal approximation to the distribution is \nfastest (not reported in the above plot but it is around 10 times faster than the `saddle` \nmethod and also equally accurate).\n\nTherefore, we devise a "hybrid/default" sampler that picks a sampler based on the above guidelines.\n\nWe also benchmark the hybrid sampler runtime with the sampler found in the `pypolyagamma` \npackage (version `1.2.3`). The version of NumPy we use is `1.18.5`. We compare our\nsampler to the `pgdrawv` functions provided by the package. Below are runtime plots of 20000\nsamples for each value of `h` and `z`. Values of `h` range from 0.1 to 50, while `z` is set\nto 0, 2.5, 5, and 10.\n\n![](./scripts/img/perf_samplers_0.0.svg) | ![](./scripts/img/perf_samplers_2.5.svg)\n| --- | --- |\n\n![](./scripts/img/perf_samplers_5.0.svg) | ![](./scripts/img/perf_samplers_10.0.svg)\n| --- | --- |\n\nIt can be seen that when generating many samples at once for any given combination of \nparameters, `polyagamma` outperforms the `pypolyagamma` package by a large margin.\nThe exception is when the scale parameter is very small (e.g `h < 1`). It is also worth\nnoting that the `pypolygamma` package is on average faster than ours at generating exactly 1\nsample value from the distribution. This is mainly due to the overhead introduced by creating\nthe bitgenerator + acquiring/releasing the thread lock + doing parameter validation checks at\nevery call to the function. This overhead can somewhat be mitigated by passing in a random\ngenerator instance at every call to the `polyagamma` function. To eliminate this overhead,\nit is best to use the Cython functions directly. Below is a timing example to demonstrate\nthe benefit of passing a generator explicitly:\n```ipynb\nIn [3]: rng = np.random.SFC64(1)\n\nIn [4]: %timeit random_polyagamma()\n90 µs ± 1.65 µs per loop (mean ± std. dev. of 7 runs, 10000 loops each)\n\nIn [5]: %timeit random_polyagamma(random_state=rng)\n1.69 µs ± 6.96 ns per loop (mean ± std. dev. of 7 runs, 1000000 loops each)\n```\n\nTo generate the above plots locally, run `python scripts/benchmark.py --size=<some size> --z=<z value>`.\nNote that the runtimes may differ  than the ones reported here, depending on the machine this script \nis ran on.\n\n\n## Distribution Plots\nBelow is a visualization of the Cumulative distribution and density functions for\nvarious values of the parameters.\n![](./scripts/img/pdf.svg) | ![](./scripts/img/cdf.svg)\n| --- | --- |\n\nWe can compare these plots to the Kernel density estimate and empirical CDF plots\ngenerated from 20000 random samples using each of the available methods.\n![](./scripts/img/kde.svg) | ![](./scripts/img/ecdf.svg)\n| --- | --- |\n\n\n## Contributing\nAll contributions, bug reports, bug fixes, documentation improvements, enhancements, and ideas are welcome.\n\nTo submit a PR, follow the steps below:\n1) Fork the repo.\n2) Setup the dev environment with `poetry install`. All dependencies will be installed.\n3) Start writing your changes, including unittests.\n3) Once finished, run `make install` to build the project with the new changes.\n4) Once build is successful, run tests to make sure they all pass with `make test`.\n5) Once finished, you can submit a PR for review.\n\n\n## References\n- Luc Devroye. "On exact simulation algorithms for some distributions related to Jacobi theta functions." Statistics & Probability Letters, Volume 79, Issue 21, (2009): 2251-2259.\n- Polson, Nicholas G., James G. Scott, and Jesse Windle. "Bayesian inference for logistic models using Pólya–Gamma latent variables." Journal of the American statistical Association 108.504 (2013): 1339-1349.\n- J. Windle, N. G. Polson, and J. G. Scott. "Improved Polya-gamma sampling". Technical Report, University of Texas at Austin, 2013b.\n- Windle, Jesse, Nicholas G. Polson, and James G. Scott. "Sampling Polya-Gamma random variates: alternate and approximate techniques." arXiv preprint arXiv:1405.0506 (2014)\n- Windle, J. (2013). Forecasting high-dimensional, time-varying variance-covariance matrices with high-frequency data and sampling Pólya-Gamma random variates for posterior distributions derived from logistic likelihoods.(PhD thesis). Retrieved from http://hdl.handle.net/2152/21842 .\n\n\n[1]: ./examples/c_polyagamma.c\n[2]: https://python-poetry.org/docs/pyproject/\n[3]: https://github.com/zoj613/polyagamma/releases\n[4]: https://img.shields.io/pypi/wheel/polyagamma?style=flat-square\n[5]: https://img.shields.io/github/v/release/zoj613/polyagamma?include_prereleases&style=flat-square\n[6]: https://img.shields.io/pypi/l/polyagamma?style=flat-square\n[7]: https://img.shields.io/circleci/build/github/zoj613/polyagamma/main?style=flat-square\n[8]: https://img.shields.io/codecov/c/github/zoj613/polyagamma?style=flat-square\n[9]: https://github.com/slinderman/pypolyagamma\n',
    'author': 'Zolisa Bleki',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/zoj613/polyagamma/',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
