import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Schedule } from './schedule';
import { BasicStepScalingPolicyProps, StepScalingPolicy } from './step-scaling-policy';
import { BasicTargetTrackingScalingPolicyProps, TargetTrackingScalingPolicy } from './target-tracking-scaling-policy';
/**
 * @stability stable
 */
export interface IScalableTarget extends IResource {
    /**
     * @stability stable
     * @attribute true
     */
    readonly scalableTargetId: string;
}
/**
 * Properties for a scalable target.
 *
 * @stability stable
 */
export interface ScalableTargetProps {
    /**
     * The minimum value that Application Auto Scaling can use to scale a target during a scaling activity.
     *
     * @stability stable
     */
    readonly minCapacity: number;
    /**
     * The maximum value that Application Auto Scaling can use to scale a target during a scaling activity.
     *
     * @stability stable
     */
    readonly maxCapacity: number;
    /**
     * Role that allows Application Auto Scaling to modify your scalable target.
     *
     * @default A role is automatically created
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The resource identifier to associate with this scalable target.
     *
     * This string consists of the resource type and unique identifier.
     *
     * Example value: `service/ecsStack-MyECSCluster-AB12CDE3F4GH/ecsStack-MyECSService-AB12CDE3F4GH`
     *
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_RegisterScalableTarget.html
     * @stability stable
     */
    readonly resourceId: string;
    /**
     * The scalable dimension that's associated with the scalable target.
     *
     * Specify the service namespace, resource type, and scaling property.
     *
     * Example value: `ecs:service:DesiredCount`
     *
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_ScalingPolicy.html
     * @stability stable
     */
    readonly scalableDimension: string;
    /**
     * The namespace of the AWS service that provides the resource or custom-resource for a resource provided by your own application or service.
     *
     * For valid AWS service namespace values, see the RegisterScalableTarget
     * action in the Application Auto Scaling API Reference.
     *
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_RegisterScalableTarget.html
     * @stability stable
     */
    readonly serviceNamespace: ServiceNamespace;
}
/**
 * Define a scalable target.
 *
 * @stability stable
 */
export declare class ScalableTarget extends Resource implements IScalableTarget {
    /**
     * @stability stable
     */
    static fromScalableTargetId(scope: Construct, id: string, scalableTargetId: string): IScalableTarget;
    /**
     * ID of the Scalable Target.
     *
     * Example value: `service/ecsStack-MyECSCluster-AB12CDE3F4GH/ecsStack-MyECSService-AB12CDE3F4GH|ecs:service:DesiredCount|ecs`
     *
     * @stability stable
     * @attribute true
     */
    readonly scalableTargetId: string;
    /**
     * The role used to give AutoScaling permissions to your resource.
     *
     * @stability stable
     */
    readonly role: iam.IRole;
    private readonly actions;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ScalableTargetProps);
    /**
     * Add a policy statement to the role's policy.
     *
     * @stability stable
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Scale out or in based on time.
     *
     * @stability stable
     */
    scaleOnSchedule(id: string, action: ScalingSchedule): void;
    /**
     * Scale out or in, in response to a metric.
     *
     * @stability stable
     */
    scaleOnMetric(id: string, props: BasicStepScalingPolicyProps): StepScalingPolicy;
    /**
     * Scale out or in in order to keep a metric around a target value.
     *
     * @stability stable
     */
    scaleToTrackMetric(id: string, props: BasicTargetTrackingScalingPolicyProps): TargetTrackingScalingPolicy;
}
/**
 * A scheduled scaling action.
 *
 * @stability stable
 */
export interface ScalingSchedule {
    /**
     * When to perform this action.
     *
     * @stability stable
     */
    readonly schedule: Schedule;
    /**
     * When this scheduled action becomes active.
     *
     * @default The rule is activate immediately
     * @stability stable
     */
    readonly startTime?: Date;
    /**
     * When this scheduled action expires.
     *
     * @default The rule never expires.
     * @stability stable
     */
    readonly endTime?: Date;
    /**
     * The new minimum capacity.
     *
     * During the scheduled time, if the current capacity is below the minimum
     * capacity, Application Auto Scaling scales out to the minimum capacity.
     *
     * At least one of maxCapacity and minCapacity must be supplied.
     *
     * @default No new minimum capacity
     * @stability stable
     */
    readonly minCapacity?: number;
    /**
     * The new maximum capacity.
     *
     * During the scheduled time, the current capacity is above the maximum
     * capacity, Application Auto Scaling scales in to the maximum capacity.
     *
     * At least one of maxCapacity and minCapacity must be supplied.
     *
     * @default No new maximum capacity
     * @stability stable
     */
    readonly maxCapacity?: number;
}
/**
 * The service that supports Application AutoScaling.
 *
 * @stability stable
 */
export declare enum ServiceNamespace {
    /**
     * Elastic Container Service.
     *
     * @stability stable
     */
    ECS = "ecs",
    /**
     * Elastic Map Reduce.
     *
     * @stability stable
     */
    ELASTIC_MAP_REDUCE = "elasticmapreduce",
    /**
     * Elastic Compute Cloud.
     *
     * @stability stable
     */
    EC2 = "ec2",
    /**
     * App Stream.
     *
     * @stability stable
     */
    APPSTREAM = "appstream",
    /**
     * Dynamo DB.
     *
     * @stability stable
     */
    DYNAMODB = "dynamodb",
    /**
     * Relational Database Service.
     *
     * @stability stable
     */
    RDS = "rds",
    /**
     * SageMaker.
     *
     * @stability stable
     */
    SAGEMAKER = "sagemaker",
    /**
     * Custom Resource.
     *
     * @stability stable
     */
    CUSTOM_RESOURCE = "custom-resource",
    /**
     * Lambda.
     *
     * @stability stable
     */
    LAMBDA = "lambda",
    /**
     * Comprehend.
     *
     * @stability stable
     */
    COMPREHEND = "comprehend",
    /**
     * Kafka.
     *
     * @stability stable
     */
    KAFKA = "kafka",
    /**
     * ElastiCache.
     *
     * @stability stable
     */
    ELASTICACHE = "elasticache"
}
