# jarvis_ai/cli.py
#!/usr/bin/env python3
"""
Основной CLI интерфейс для Jarvis AI
"""

import os
import sys
import argparse
import json
from pathlib import Path
from typing import Optional, List, Dict, Any
import logging
import readline  # Для истории команд и автодополнения

# Добавляем путь к модулям проекта
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Импорты из нашего проекта
from jarvis_ai.core.ai_engine import AIEngine
from jarvis_ai.core.privacy_engine import PrivacyEngine
from jarvis_ai.core.knowledge_base import KnowledgeBase
from jarvis_ai.system.file_manager import FileManager
from jarvis_ai.system.screen_analyzer import ScreenAnalyzer

# Настройка логирования
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler(os.path.expanduser("~/.jarvis/logs/cli.log")),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)

class JarvisCLI:
    """
    Основной класс CLI интерфейса
    """
    
    def __init__(self):
        self.parser = self._create_parser()
        self.knowledge_base = KnowledgeBase()
        self.privacy_engine = PrivacyEngine()
        self.ai_engine = AIEngine()
        self.file_manager = FileManager()
        self.screen_analyzer = ScreenAnalyzer()
        
        # История команд
        self.command_history = []
        self.history_file = os.path.expanduser("~/.jarvis/history.txt")
        self._load_history()
    
    def _create_parser(self) -> argparse.ArgumentParser:
        """
        Создание парсера аргументов командной строки
        """
        parser = argparse.ArgumentParser(
            description="🤖 Jarvis AI Assistant - Умный помощник с локальным ИИ",
            epilog="Примеры использования:\n"
                   "  jarvis start                    # Запуск интерактивного режима\n"
                   "  jarvis analyze 'проблема'      # Анализ проблемы со скриншотом\n"
                   "  jarvis ask 'вопрос'            # Задать вопрос ИИ\n"
                   "  jarvis translate 'текст'       # Переводчик\n"
                   "  jarvis stats                   # Статистика\n"
                   "  jarvis config                  # Настройки\n",
            formatter_class=argparse.RawDescriptionHelpFormatter
        )
        
        subparsers = parser.add_subparsers(
            dest='command',
            title='Команды',
            description='Доступные команды Jarvis AI',
            help='Выберите команду для выполнения'
        )
        
        # Команда: start
        start_parser = subparsers.add_parser(
            'start',
            help='Запуск интерактивного ассистента',
            description='Запуск интерактивного диалогового режима с ИИ'
        )
        
        # Команда: analyze
        analyze_parser = subparsers.add_parser(
            'analyze',
            help='Анализ проблемы со скриншотом',
            description='Создание скриншота и анализ проблемы с помощью ИИ'
        )
        analyze_parser.add_argument(
            'problem',
            type=str,
            help='Описание проблемы (например: "не работает кнопка сохранения")'
        )
        analyze_parser.add_argument(
            '--model',
            type=str,
            default=None,
            help='Модель ИИ для анализа (по умолчанию: автоматический выбор)'
        )
        
        # Команда: ask
        ask_parser = subparsers.add_parser(
            'ask',
            help='Задать вопрос ИИ',
            description='Задать вопрос и получить ответ от ИИ'
        )
        ask_parser.add_argument(
            'question',
            type=str,
            help='Вопрос для ИИ'
        )
        ask_parser.add_argument(
            '--model',
            type=str,
            default=None,
            help='Модель ИИ для ответа'
        )
        
        # Команда: translate
        translate_parser = subparsers.add_parser(
            'translate',
            help='Перевод текста',
            description='Перевод текста между языками'
        )
        translate_parser.add_argument(
            'text',
            type=str,
            help='Текст для перевода'
        )
        translate_parser.add_argument(
            '--from',
            dest='from_lang',
            type=str,
            default='auto',
            help='Исходный язык (по умолчанию: автоопределение)'
        )
        translate_parser.add_argument(
            '--to',
            dest='to_lang',
            type=str,
            default='en',
            help='Целевой язык (по умолчанию: английский)'
        )
        
        # Команда: summarize
        summarize_parser = subparsers.add_parser(
            'summarize',
            help='Суммаризация текста',
            description='Сокращение длинного текста до ключевых моментов'
        )
        summarize_parser.add_argument(
            'text',
            type=str,
            help='Текст для суммаризации'
        )
        summarize_parser.add_argument(
            '--max-length',
            type=int,
            default=150,
            help='Максимальная длина результата'
        )
        
        # Команда: stats
        stats_parser = subparsers.add_parser(
            'stats',
            help='Статистика системы',
            description='Показать статистику использования и эффективности'
        )
        stats_parser.add_argument(
            '--detailed',
            action='store_true',
            help='Подробная статистика'
        )
        
        # Команда: config
        config_parser = subparsers.add_parser(
            'config',
            help='Настройки системы',
            description='Управление настройками Jarvis AI'
        )
        config_subparsers = config_parser.add_subparsers(
            dest='config_command',
            help='Команды настройки'
        )
        
        # Подкоманды config
        config_subparsers.add_parser(
            'show',
            help='Показать текущие настройки'
        )
        
        config_token = config_subparsers.add_parser(
            'set-token',
            help='Установить API токен'
        )
        config_token.add_argument(
            'token',
            type=str,
            help='Токен Hugging Face API'
        )
        
        config_model = config_subparsers.add_parser(
            'set-model',
            help='Установить модель по умолчанию'
        )
        config_model.add_argument(
            'model_type',
            type=str,
            choices=['text', 'qa', 'translation', 'summarization'],
            help='Тип модели'
        )
        config_model.add_argument(
            'model_name',
            type=str,
            help='Имя модели'
        )
        
        # Команда: file
        file_parser = subparsers.add_parser(
            'file',
            help='Файловый менеджер',
            description='Управление файлами и директориями'
        )
        file_subparsers = file_parser.add_subparsers(
            dest='file_command',
            help='Команды файлового менеджера'
        )
        
        # Подкоманды file
        file_subparsers.add_parser('list', help='Список файлов')
        file_subparsers.add_parser('pwd', help='Текущая директория')
        
        file_mkdir = file_subparsers.add_parser('mkdir', help='Создать директорию')
        file_mkdir.add_argument('dirname', type=str, help='Имя директории')
        
        file_touch = file_subparsers.add_parser('touch', help='Создать файл')
        file_touch.add_argument('filename', type=str, help='Имя файла')
        
        # Команда: knowledge
        knowledge_parser = subparsers.add_parser(
            'knowledge',
            help='База знаний',
            description='Управление локальной базой знаний'
        )
        knowledge_subparsers = knowledge_parser.add_subparsers(
            dest='knowledge_command',
            help='Команды базы знаний'
        )
        
        knowledge_search = knowledge_subparsers.add_parser(
            'search',
            help='Поиск в базе знаний'
        )
        knowledge_search.add_argument(
            'query',
            type=str,
            help='Поисковый запрос'
        )
        
        knowledge_subparsers.add_parser(
            'clear',
            help='Очистить базу знаний'
        )
        
        # Команда: screenshots
        screenshots_parser = subparsers.add_parser(
            'screenshots',
            help='Управление скриншотами',
            description='Просмотр и управление скриншотами'
        )
        screenshots_parser.add_argument(
            '--limit',
            type=int,
            default=10,
            help='Количество скриншотов для показа'
        )
        screenshots_parser.add_argument(
            '--cleanup',
            action='store_true',
            help='Очистить старые скриншоты'
        )
        
        # Команда: models
        models_parser = subparsers.add_parser(
            'models',
            help='Управление моделями ИИ',
            description='Просмотр и управление моделями ИИ'
        )
        models_parser.add_argument(
            '--list',
            action='store_true',
            help='Показать доступные модели'
        )
        models_parser.add_argument(
            '--search',
            type=str,
            help='Поиск моделей'
        )
        
        return parser
    
    def _load_history(self):
        """Загрузка истории команд из файла"""
        try:
            history_dir = os.path.dirname(self.history_file)
            if not os.path.exists(history_dir):
                os.makedirs(history_dir, exist_ok=True)
            
            if os.path.exists(self.history_file):
                with open(self.history_file, 'r', encoding='utf-8') as f:
                    self.command_history = [line.strip() for line in f.readlines()]
        except Exception as e:
            logger.warning(f"Не удалось загрузить историю: {e}")
    
    def _save_history(self):
        """Сохранение истории команд в файл"""
        try:
            with open(self.history_file, 'w', encoding='utf-8') as f:
                for cmd in self.command_history[-100:]:  # Сохраняем последние 100 команд
                    f.write(cmd + '\n')
        except Exception as e:
            logger.warning(f"Не удалось сохранить историю: {e}")
    
    def _add_to_history(self, command: str):
        """Добавление команды в историю"""
        if command and command not in self.command_history[-10:]:  # Не сохраняем дубликаты
            self.command_history.append(command)
            self._save_history()
    
    def run(self):
        """Запуск CLI"""
        args = self.parser.parse_args()
        
        if not args.command:
            self.parser.print_help()
            return 0
        
        # Добавляем команду в историю
        self._add_to_history(' '.join(sys.argv[1:]))
        
        # Обработка команд
        try:
            if args.command == 'start':
                return self._handle_start(args)
            elif args.command == 'analyze':
                return self._handle_analyze(args)
            elif args.command == 'ask':
                return self._handle_ask(args)
            elif args.command == 'translate':
                return self._handle_translate(args)
            elif args.command == 'summarize':
                return self._handle_summarize(args)
            elif args.command == 'stats':
                return self._handle_stats(args)
            elif args.command == 'config':
                return self._handle_config(args)
            elif args.command == 'file':
                return self._handle_file(args)
            elif args.command == 'knowledge':
                return self._handle_knowledge(args)
            elif args.command == 'screenshots':
                return self._handle_screenshots(args)
            elif args.command == 'models':
                return self._handle_models(args)
            else:
                print(f"❌ Неизвестная команда: {args.command}")
                self.parser.print_help()
                return 1
                
        except KeyboardInterrupt:
            print("\n\n⚠️  Прервано пользователем")
            return 130
        except Exception as e:
            print(f"❌ Ошибка: {e}")
            logger.error(f"Ошибка выполнения команды: {e}", exc_info=True)
            return 1
    
    def _handle_start(self, args) -> int:
        """
        Запуск интерактивного ассистента
        """
        print("🚀" + "="*50)
        print("🤖 Jarvis AI Assistant v1.0")
        print("="*52)
        
        # Проверка токена
        if not self.ai_engine.api_token:
            print("⚠️  Внимание: HUGGINGFACE_TOKEN не установлен")
            print("📝 Для настройки выполните: jarvis config set-token <ваш_токен>")
            print("   или установите переменную окружения: setx HUGGINGFACE_TOKEN <токен>")
            print("="*52)
        
        print("\nДоступные команды в интерактивном режиме:")
        print("  /help       - Показать это сообщение")
        print("  /stats      - Статистика системы")
        print("  /clear      - Очистить экран")
        print("  /history    - История команд")
        print("  /translate  <текст> - Переводчик")
        print("  /summarize  <текст> - Суммаризация")
        print("  /exit       - Выйти")
        print("\nПросто введите ваш вопрос для получения помощи!")
        print("="*52)
        
        session_count = 0
        
        try:
            while True:
                try:
                    # Чтение ввода с поддержкой истории
                    user_input = input("\n🤔 Вы: ").strip()
                    
                    if not user_input:
                        continue
                    
                    # Обработка специальных команд
                    if user_input.lower() == '/exit':
                        break
                    elif user_input.lower() == '/help':
                        print("\n📚 Доступные команды:")
                        print("  /help       - Показать это сообщение")
                        print("  /stats      - Статистика системы")
                        print("  /clear      - Очистить экран")
                        print("  /history    - История команд")
                        print("  /translate  <текст> - Переводчик")
                        print("  /summarize  <текст> - Суммаризация")
                        print("  /exit       - Выйти")
                        print("\n💡 Примеры вопросов:")
                        print("  Как исправить синий экран смерти?")
                        print("  Почему не запускается Photoshop?")
                        print("  Как ускорить работу Windows?")
                        continue
                    elif user_input.lower() == '/stats':
                        self._show_detailed_stats()
                        continue
                    elif user_input.lower() == '/clear':
                        os.system('cls' if os.name == 'nt' else 'clear')
                        continue
                    elif user_input.lower() == '/history':
                        print("\n📜 История команд:")
                        for i, cmd in enumerate(self.command_history[-10:], 1):
                            print(f"  {i:2}. {cmd}")
                        continue
                    elif user_input.lower().startswith('/translate '):
                        text = user_input[11:].strip()
                        if text:
                            result = self.ai_engine.translate_text(text, "ru", "en")
                            print(f"\n🌍 Перевод: {result}")
                        continue
                    elif user_input.lower().startswith('/summarize '):
                        text = user_input[11:].strip()
                        if text:
                            result = self.ai_engine.summarize_text(text)
                            print(f"\n📝 Суммаризация: {result}")
                        continue
                    
                    session_count += 1
                    print(f"\n🔎 Обработка запроса #{session_count}...")
                    
                    # 1. Очистка приватности
                    print("   🛡️  Очистка конфиденциальных данных...")
                    cleaned_input, privacy_report = self.privacy_engine.clean_text(user_input)
                    
                    # 2. Поиск в локальной базе знаний
                    print("   🧠 Поиск в локальной базе знаний...")
                    local_answer = self.knowledge_base.find_similar_question(cleaned_input)
                    
                    if local_answer:
                        print("   ✅ Найдено локальное решение")
                        # Восстановление контекста
                        final_answer = self.privacy_engine.restore_context(local_answer, user_input)
                        print(f"\n💡 Jarvis: {final_answer}")
                        
                        # Обновление статистики
                        self.knowledge_base.record_local_usage()
                        continue
                    
                    # 3. Запрос к Hugging Face
                    print("   🤖 Обработка через Hugging Face...")
                    ai_response = self.ai_engine.generate_text(
                        prompt=cleaned_input,
                        max_length=300
                    )
                    
                    if ai_response and not ai_response.startswith("Ошибка:"):
                        print("   ✅ Получен ответ от ИИ")
                        
                        # 4. Сохранение в базу знаний
                        self.knowledge_base.add_qa_pair(
                            question=cleaned_input,
                            answer=ai_response,
                            source="huggingface"
                        )
                        
                        # 5. Восстановление контекста
                        final_answer = self.privacy_engine.restore_context(ai_response, user_input)
                        print(f"\n🤖 Jarvis: {final_answer}")
                    else:
                        print(f"\n❌ Ошибка ИИ: {ai_response}")
                        print("\n🔧 Рекомендации:")
                        print("1. Проверьте токен: jarvis config show")
                        print("2. Проверьте подключение к интернету")
                        print("3. Попробуйте другую модель: jarvis models --list")
                        
                except KeyboardInterrupt:
                    print("\n\n📝 Используйте /exit для выхода или продолжайте ввод")
                    continue
                except Exception as e:
                    print(f"\n❌ Ошибка: {e}")
                    continue
                    
        finally:
            print(f"\n📊 Сессия завершена")
            print(f"   Обработано запросов: {session_count}")
            
            stats = self.knowledge_base.get_statistics()
            ai_stats = self.ai_engine.get_statistics()
            
            print(f"   Локальных ответов: {stats.get('local_responses', 0)}")
            print(f"   Hugging Face ответов: {stats.get('huggingface_responses', 0)}")
            print(f"   Всего токенов использовано: {ai_stats.get('tokens_used', 0)}")
            print(f"   Успешных запросов: {ai_stats.get('success_rate', 0):.1f}%")
            
            print("\n👋 До свидания! Возвращайтесь снова!")
        
        return 0
    
    def _handle_analyze(self, args) -> int:
        """
        Анализ проблемы со скриншотом
        """
        problem = args.problem
        print(f"🔍 Анализ проблемы: {problem}")
        
        # Создаем скриншот
        screenshot_info = self.screen_analyzer.capture_problem_screenshot(problem)
        if not screenshot_info:
            print("❌ Не удалось создать скриншот")
            return 1
        
        screenshot_path = screenshot_info.get("screenshot_path")
        print(f"📸 Скриншот сохранен: {screenshot_path}")
        
        # Анализ проблемы с помощью ИИ
        print("🤖 Анализ проблемы с помощью ИИ...")
        
        # Очистка текста проблемы
        cleaned_problem, _ = self.privacy_engine.clean_text(problem)
        
        # Поиск в локальной базе знаний
        local_answer = self.knowledge_base.find_similar_question(cleaned_problem)
        if local_answer:
            print("✅ Найдено локальное решение")
            final_answer = self.privacy_engine.restore_context(local_answer, problem)
            print(f"\n💡 Решение: {final_answer}")
            return 0
        
        # Запрос к Hugging Face
        prompt = f"""
        Пользователь сообщает о проблеме: "{cleaned_problem}"
        
        Проанализируй проблему и предложи решение.
        Будь конкретным и дай пошаговые инструкции.
        
        Ответ должен содержать:
        1. Вероятную причину проблемы
        2. Пошаговое решение
        3. Советы по предотвращению
        
        Ответ:
        """
        
        ai_response = self.ai_engine.generate_text(
            prompt=prompt,
            max_length=500,
            model=args.model
        )
        
        if ai_response and not ai_response.startswith("Ошибка:"):
            print("✅ Получен ответ от ИИ")
            
            # Сохраняем в базу знаний
            self.knowledge_base.add_qa_pair(
                question=cleaned_problem,
                answer=ai_response,
                source="huggingface"
            )
            
            final_answer = self.privacy_engine.restore_context(ai_response, problem)
            print(f"\n🤖 Анализ проблемы:")
            print("="*50)
            print(final_answer)
            print("="*50)
            
            # Предложение дополнительных действий
            print("\n🎯 Дополнительные действия:")
            print("  • Для просмотра скриншотов: jarvis screenshots")
            print("  • Для поиска похожих проблем: jarvis knowledge search \"ключевые слова\"")
            print("  • Для статистики: jarvis stats")
            
        else:
            print(f"❌ Ошибка ИИ: {ai_response}")
            print("\n📸 Скриншот сохранен для дальнейшего анализа.")
            print("🔧 Попробуйте:")
            print("  1. jarvis config set-token <ваш_токен>")
            print("  2. jarvis ask \"{problem}\"")
        
        return 0
    
    def _handle_ask(self, args) -> int:
        """
        Задать вопрос ИИ
        """
        question = args.question
        print(f"❓ Вопрос: {question}")
        
        # Очистка приватности
        cleaned_question, _ = self.privacy_engine.clean_text(question)
        
        # Поиск в локальной базе знаний
        local_answer = self.knowledge_base.find_similar_question(cleaned_question)
        if local_answer:
            print("✅ Найдено локальное решение")
            final_answer = self.privacy_engine.restore_context(local_answer, question)
            print(f"\n💡 Ответ: {final_answer}")
            return 0
        
        # Запрос к Hugging Face
        ai_response = self.ai_engine.generate_text(
            prompt=cleaned_question,
            max_length=400,
            model=args.model
        )
        
        if ai_response and not ai_response.startswith("Ошибка:"):
            print("✅ Получен ответ от ИИ")
            
            # Сохраняем в базу знаний
            self.knowledge_base.add_qa_pair(
                question=cleaned_question,
                answer=ai_response,
                source="huggingface"
            )
            
            final_answer = self.privacy_engine.restore_context(ai_response, question)
            print(f"\n🤖 Ответ: {final_answer}")
            
        else:
            print(f"❌ Ошибка ИИ: {ai_response}")
        
        return 0
    
    def _handle_translate(self, args) -> int:
        """
        Перевод текста
        """
        text = args.text
        
        # Определение языка
        from_lang = args.from_lang
        to_lang = args.to_lang
        
        if from_lang == 'auto':
            # Простая детекция языка
            if any(c in 'абвгдеёжзийклмнопрстуфхцчшщъыьэюя' for c in text.lower()):
                from_lang = 'ru'
            else:
                from_lang = 'en'
        
        print(f"🌍 Перевод с {from_lang.upper()} на {to_lang.upper()}:")
        print(f"📝 Исходный текст: {text}")
        
        # Перевод
        result = self.ai_engine.translate_text(text, from_lang, to_lang)
        
        if result and not result.startswith("Ошибка:"):
            print(f"\n✅ Перевод: {result}")
        else:
            print(f"❌ Ошибка перевода: {result}")
        
        return 0
    
    def _handle_summarize(self, args) -> int:
        """
        Суммаризация текста
        """
        text = args.text
        max_length = args.max_length
        
        if len(text) < 100:
            print("⚠️  Текст слишком короткий для суммаризации")
            return 0
        
        print(f"📝 Суммаризация текста (длина: {len(text)} символов)")
        
        result = self.ai_engine.summarize_text(text, max_length=max_length)
        
        if result and not result.startswith("Ошибка:"):
            print(f"\n✅ Суммаризация ({len(result)} символов):")
            print("="*50)
            print(result)
            print("="*50)
        else:
            print(f"❌ Ошибка суммаризации: {result}")
        
        return 0
    
    def _show_detailed_stats(self):
        """Показать подробную статистику"""
        stats = self.knowledge_base.get_statistics()
        ai_stats = self.ai_engine.get_statistics()
        
        print("\n📊 ДЕТАЛЬНАЯ СТАТИСТИКА СИСТЕМЫ")
        print("="*50)
        
        print("🎯 БАЗА ЗНАНИЙ:")
        print(f"   Всего записей: {stats.get('total_entries', 0)}")
        print(f"   Уникальных вопросов: {stats.get('unique_questions', 0)}")
        print(f"   Локальных ответов: {stats.get('local_responses', 0)}")
        print(f"   Hugging Face ответов: {stats.get('huggingface_responses', 0)}")
        
        print("\n🤖 HUGGING FACE API:")
        print(f"   Всего запросов: {ai_stats.get('total_requests', 0)}")
        print(f"   Успешных: {ai_stats.get('successful_requests', 0)}")
        print(f"   Неудачных: {ai_stats.get('failed_requests', 0)}")
        print(f"   Успешность: {ai_stats.get('success_rate', 0):.1f}%")
        print(f"   Хитов кэша: {ai_stats.get('cache_hits', 0)}")
        print(f"   Эффективность кэша: {ai_stats.get('cache_hit_rate', 0):.1f}%")
        print(f"   Использовано токенов: {ai_stats.get('tokens_used', 0)}")
        
        print("\n💰 ЭКОНОМИЯ:")
        total_requests = ai_stats.get('total_requests', 1)
        local_responses = stats.get('local_responses', 0)
        savings_percent = (local_responses / total_requests * 100) if total_requests > 0 else 0
        
        print(f"   Экономия запросов: {savings_percent:.1f}%")
        print(f"   Сэкономлено токенов: {stats.get('tokens_saved', 0)}")
        
        print("\n💾 ПРИВАТНОСТЬ:")
        privacy_stats = self.privacy_engine.get_statistics()
        print(f"   Обработано запросов: {privacy_stats.get('total_processed', 0)}")
        print(f"   Найдено конфиденц. данных: {privacy_stats.get('sensitive_found', 0)}")
        print(f"   Применено правил: {privacy_stats.get('rules_applied', 0)}")
        
        print("="*50)
    
    def _handle_stats(self, args) -> int:
        """
        Показать статистику
        """
        if args.detailed:
            self._show_detailed_stats()
        else:
            stats = self.knowledge_base.get_statistics()
            ai_stats = self.ai_engine.get_statistics()
            
            print("📊 СТАТИСТИКА JARVIS AI")
            print("="*40)
            print(f"Всего записей в базе: {stats.get('total_entries', 0)}")
            print(f"Использовано Hugging Face: {stats.get('huggingface_used', 0)}")
            print(f"Локальных ответов: {stats.get('local_responses', 0)}")
            print(f"Сэкономлено токенов: {stats.get('tokens_saved', 0)}")
            print(f"Успешность API: {ai_stats.get('success_rate', 0):.1f}%")
            print("="*40)
        
        return 0
    
    def _handle_config(self, args) -> int:
        """
        Управление настройками
        """
        if not hasattr(args, 'config_command') or args.config_command is None:
            print("📋 Текущие настройки:")
            print("="*40)
            
            token = self.ai_engine.api_token or "Не установлен"
            masked_token = token[:10] + "..." + token[-10:] if len(token) > 20 else token
            print(f"🤫 Hugging Face токен: {masked_token}")
            
            print("\n🤖 Модели по умолчанию:")
            for key, value in self.ai_engine.default_models.items():
                print(f"  {key}: {value}")
            
            print("\n🔧 Команды настройки:")
            print("  jarvis config show           - Показать настройки")
            print("  jarvis config set-token <t>  - Установить токен")
            print("  jarvis config set-model <t> <m> - Установить модель")
            return 0
        
        if args.config_command == 'show':
            self._handle_config(args)  # Рекурсивно вызываем показ настроек
            return 0
        
        elif args.config_command == 'set-token':
            token = args.token
            os.environ['HUGGINGFACE_TOKEN'] = token
            
            # Сохраняем в файл конфигурации
            config_dir = os.path.expanduser("~/.jarvis")
            os.makedirs(config_dir, exist_ok=True)
            
            config_file = os.path.join(config_dir, "config.json")
            config = {}
            
            if os.path.exists(config_file):
                try:
                    with open(config_file, 'r') as f:
                        config = json.load(f)
                except:
                    pass
            
            config['huggingface_token'] = token
            
            with open(config_file, 'w') as f:
                json.dump(config, f, indent=2)
            
            # Обновляем токен в AI Engine
            self.ai_engine.api_token = token
            
            print(f"✅ Токен установлен: {token[:15]}...")
            print("📝 Перезапустите Jarvis для применения настроек")
            return 0
        
        elif args.config_command == 'set-model':
            model_type = args.model_type
            model_name = args.model_name
            
            if model_type in self.ai_engine.default_models:
                self.ai_engine.default_models[model_type] = model_name
                print(f"✅ Модель {model_type} установлена: {model_name}")
            else:
                print(f"❌ Неизвестный тип модели: {model_type}")
                print(f"   Доступные типы: {list(self.ai_engine.default_models.keys())}")
                return 1
            
            return 0
        
        return 0
    
    def _handle_file(self, args) -> int:
        """
        Обработка команд файлового менеджера
        """
        if not hasattr(args, 'file_command') or args.file_command is None:
            print("📁 Файловый менеджер - доступные команды:")
            print("  jarvis file list          - Список файлов")
            print("  jarvis file pwd           - Текущая директория")
            print("  jarvis file mkdir <name>  - Создать директорию")
            print("  jarvis file touch <name>  - Создать файл")
            return 0
        
        if args.file_command == 'list':
            files = self.file_manager.list_directory()
            for file in files:
                print(file)
            return 0
        
        elif args.file_command == 'pwd':
            print(self.file_manager.get_current_directory())
            return 0
        
        elif args.file_command == 'mkdir':
            dirname = args.dirname
            result = self.file_manager.create_directory(dirname)
            if result:
                print(f"✅ Директория создана: {dirname}")
            else:
                print(f"❌ Не удалось создать директорию: {dirname}")
            return 0
        
        elif args.file_command == 'touch':
            filename = args.filename
            result = self.file_manager.create_file(filename)
            if result:
                print(f"✅ Файл создан: {filename}")
            else:
                print(f"❌ Не удалось создать файл: {filename}")
            return 0
        
        return 0
    
    def _handle_knowledge(self, args) -> int:
        """
        Обработка команд базы знаний
        """
        if not hasattr(args, 'knowledge_command') or args.knowledge_command is None:
            print("🧠 База знаний - доступные команды:")
            print("  jarvis knowledge search <query> - Поиск в базе знаний")
            print("  jarvis knowledge clear         - Очистить базу знаний")
            return 0
        
        if args.knowledge_command == 'search':
            query = args.query
            results = self.knowledge_base.search(query)
            
            if not results:
                print(f"🔍 По запросу '{query}' ничего не найдено")
                return 0
            
            print(f"🔍 Найдено {len(results)} результатов по запросу '{query}':")
            print("="*60)
            
            for i, result in enumerate(results, 1):
                print(f"\n📝 Результат #{i}:")
                print(f"   Вопрос: {result.get('question', 'N/A')}")
                print(f"   Ответ: {result.get('answer', 'N/A')[:200]}...")
                print(f"   Источник: {result.get('source', 'N/A')}")
                print(f"   Дата: {result.get('created_at', 'N/A')}")
                print("-"*40)
            
            return 0
        
        elif args.knowledge_command == 'clear':
            confirm = input("⚠️  Вы уверены, что хотите очистить базу знаний? (y/N): ")
            if confirm.lower() == 'y':
                self.knowledge_base.clear_database()
                print("✅ База знаний очищена")
            else:
                print("❌ Отменено")
            return 0
        
        return 0
    
    def _handle_screenshots(self, args) -> int:
        """
        Управление скриншотами
        """
        if args.cleanup:
            deleted = self.screen_analyzer.cleanup_old_screenshots()
            print(f"🗑️  Удалено {deleted} старых скриншотов")
            return 0
        
        screenshots = self.screen_analyzer.get_recent_screenshots(limit=args.limit)
        
        if not screenshots:
            print("📸 Скриншотов не найдено")
            return 0
        
        print(f"📸 Последние {len(screenshots)} скриншотов:")
        print("="*80)
        
        for i, screenshot in enumerate(screenshots, 1):
            print(f"\n#{i}:")
            print(f"  Файл: {screenshot.get('filename', 'N/A')}")
            print(f"  Дата: {screenshot.get('created_at', 'N/A')}")
            print(f"  Путь: {screenshot.get('full_path', 'N/A')}")
            
            if 'problem_description' in screenshot:
                desc = screenshot['problem_description']
                if len(desc) > 100:
                    desc = desc[:100] + "..."
                print(f"  Описание: {desc}")
            
            # Размер файла
            import os
            path = screenshot.get('full_path')
            if path and os.path.exists(path):
                size = os.path.getsize(path) / 1024  # KB
                print(f"  Размер: {size:.1f} KB")
        
        return 0
    
    def _handle_models(self, args) -> int:
        """
        Управление моделями ИИ
        """
        if args.list:
            print("🤖 Популярные модели Hugging Face:")
            print("="*60)
            
            models = [
                ("microsoft/DialoGPT-large", "Диалоговый ИИ", "Текст"),
                ("gpt2", "Базовая генерация", "Текст"),
                ("deepset/roberta-base-squad2", "Вопрос-ответ", "Текст"),
                ("facebook/bart-large-cnn", "Суммаризация", "Текст"),
                ("Helsinki-NLP/opus-mt-en-ru", "Перевод EN->RU", "Текст"),
                ("Helsinki-NLP/opus-mt-ru-en", "Перевод RU->EN", "Текст"),
                ("distilbert-base-uncased-finetuned-sst-2-english", "Классификация", "Текст"),
                ("nlptown/bert-base-multilingual-uncased-sentiment", "Анализ настроения", "Текст"),
                ("Salesforce/blip-image-captioning-large", "Анализ изображений", "Изображение"),
            ]
            
            for model_name, description, model_type in models:
                print(f"\n📦 {model_name}")
                print(f"   📝 {description}")
                print(f"   🏷️  Тип: {model_type}")
            
            print("\n💡 Для использования модели:")
            print("   jarvis ask --model 'имя_модели' 'ваш_вопрос'")
            
            return 0
        
        elif args.search:
            query = args.search
            print(f"🔍 Поиск моделей по запросу: '{query}'")
            
            try:
                models = self.ai_engine.get_available_models(limit=10)
                
                if not models:
                    print("❌ Не удалось получить список моделей")
                    return 1
                
                # Фильтрация по запросу
                filtered_models = [
                    model for model in models 
                    if query.lower() in model.get('id', '').lower() or 
                       query.lower() in model.get('pipeline_tag', '').lower()
                ][:10]
                
                if not filtered_models:
                    print(f"❌ Модели по запросу '{query}' не найдены")
                    return 0
                
                print(f"\n✅ Найдено {len(filtered_models)} моделей:")
                print("="*60)
                
                for model in filtered_models:
                    model_id = model.get('id', 'N/A')
                    downloads = model.get('downloads', 0)
                    tags = model.get('tags', [])
                    
                    print(f"\n📦 {model_id}")
                    print(f"   ⬇️  Загрузок: {downloads:,}")
                    print(f"   🏷️  Теги: {', '.join(tags[:3])}")
            
            except Exception as e:
                print(f"❌ Ошибка поиска моделей: {e}")
            
            return 0
        
        else:
            print("🤖 Управление моделями ИИ:")
            print("  jarvis models --list           - Показать популярные модели")
            print("  jarvis models --search <query> - Поиск моделей")
            return 0

def main():
    """
    Точка входа в приложение
    """
    try:
        cli = JarvisCLI()
        return cli.run()
    except KeyboardInterrupt:
        print("\n\n👋 До свидания!")
        return 130
    except Exception as e:
        print(f"❌ Критическая ошибка: {e}")
        logger.error(f"Критическая ошибка: {e}", exc_info=True)
        return 1

if __name__ == "__main__":
    sys.exit(main())