# jarvis_ai/core/ai_engine.py
import os
import sys
import requests
import json
import time
import base64
from pathlib import Path
from typing import Optional, Dict, Any, List, Union
import logging
from datetime import datetime

# Настройка логирования
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

class AIEngine:
    """
    Основной движок ИИ для работы с Hugging Face API
    """
    
    def __init__(self, api_token: Optional[str] = None):
        """
        Инициализация AI Engine для работы с Hugging Face API
        
        Args:
            api_token: Токен Hugging Face API
        """
        self.api_token = api_token or os.getenv("HUGGINGFACE_TOKEN")
        
        if not self.api_token:
            logger.warning("HUGGINGFACE_TOKEN не установлен. Установите токен командой: setx HUGGINGFACE_TOKEN \"ваш_токен\"")
        
        # Endpoint для Hugging Face Inference API
        self.base_url = "https://api-inference.huggingface.co"
        self.models_url = "https://huggingface.co/api/models"
        
        # Популярные модели для разных задач
        self.default_models = {
            "text_generation": "microsoft/DialoGPT-large",  # Диалоговый ИИ
            "text_generation_fallback": "gpt2",  # Фолбэк модель
            "question_answering": "deepset/roberta-base-squad2",  # Вопрос-ответ
            "summarization": "facebook/bart-large-cnn",  # Суммаризация
            "translation_en_ru": "Helsinki-NLP/opus-mt-en-ru",  # Перевод EN->RU
            "translation_ru_en": "Helsinki-NLP/opus-mt-ru-en",  # Перевод RU->EN
            "text_classification": "distilbert-base-uncased-finetuned-sst-2-english",  # Классификация текста
            "sentiment": "nlptown/bert-base-multilingual-uncased-sentiment",  # Анализ настроения
        }
        
        # Кэш для хранения результатов
        self.cache = {}
        self.cache_size = 100  # Максимальное количество записей в кэше
        
        # Статистика использования
        self.stats = {
            "total_requests": 0,
            "successful_requests": 0,
            "failed_requests": 0,
            "cache_hits": 0,
            "tokens_used": 0,
        }
    
    def _get_cache_key(self, model: str, inputs: Any, params: Dict) -> str:
        """Создание ключа для кэша"""
        import hashlib
        data_str = f"{model}:{str(inputs)}:{json.dumps(params, sort_keys=True)}"
        return hashlib.md5(data_str.encode()).hexdigest()
    
    def _add_to_cache(self, key: str, result: Any) -> None:
        """Добавление результата в кэш"""
        if len(self.cache) >= self.cache_size:
            # Удаляем самую старую запись
            oldest_key = next(iter(self.cache))
            del self.cache[oldest_key]
        
        self.cache[key] = {
            "result": result,
            "timestamp": datetime.now().isoformat()
        }
    
    def query_huggingface(
        self,
        model: str,
        inputs: Union[str, Dict, List],
        parameters: Optional[Dict] = None,
        retry_count: int = 3
    ) -> Optional[Dict]:
        """
        Основной метод запроса к Hugging Face API
        
        Args:
            model: Имя модели (например, "microsoft/DialoGPT-large")
            inputs: Входные данные (текст, изображение, etc)
            parameters: Дополнительные параметры
            retry_count: Количество попыток при ошибке 503
            
        Returns:
            Ответ от API или None при ошибке
        """
        if not self.api_token:
            logger.error("HUGGINGFACE_TOKEN не установлен")
            return {"error": "API токен не установлен"}
        
        self.stats["total_requests"] += 1
        
        # Проверяем кэш
        cache_key = self._get_cache_key(model, inputs, parameters or {})
        if cache_key in self.cache:
            self.stats["cache_hits"] += 1
            logger.debug(f"Кэш хит для модели {model}")
            return self.cache[cache_key]["result"]
        
        # Подготавливаем URL
        url = f"{self.base_url}/models/{model}"
        
        # Подготавливаем заголовки
        headers = {
            "Authorization": f"Bearer {self.api_token}",
            "Content-Type": "application/json",
        }
        
        # Подготавливаем данные
        data = {"inputs": inputs}
        if parameters:
            data["parameters"] = parameters
        
        try:
            logger.info(f"Запрос к модели {model}")
            response = requests.post(
                url,
                headers=headers,
                json=data,
                timeout=30  # Таймаут 30 секунд
            )
            
            # Обработка ответа
            if response.status_code == 200:
                result = response.json()
                self.stats["successful_requests"] += 1
                
                # Сохраняем в кэш
                self._add_to_cache(cache_key, result)
                
                # Примерная оценка использованных токенов
                if isinstance(inputs, str):
                    self.stats["tokens_used"] += len(inputs.split()) // 0.75  # Приблизительный расчет
                
                return result
                
            elif response.status_code == 503:
                # Модель загружается
                logger.warning(f"Модель {model} загружается...")
                
                if retry_count > 0:
                    # Ждем и пробуем снова
                    wait_time = 10  # секунд
                    logger.info(f"Ждем {wait_time} секунд перед повторной попыткой...")
                    time.sleep(wait_time)
                    return self.query_huggingface(model, inputs, parameters, retry_count - 1)
                else:
                    logger.error(f"Модель {model} не загрузилась после нескольких попыток")
                    self.stats["failed_requests"] += 1
                    return {"error": "Модель не загружена"}
                    
            elif response.status_code == 401:
                logger.error(f"Ошибка аутентификации: неверный токен")
                self.stats["failed_requests"] += 1
                return {"error": "Неверный API токен"}
                
            elif response.status_code == 404:
                logger.error(f"Модель {model} не найдена")
                self.stats["failed_requests"] += 1
                return {"error": f"Модель {model} не найдена"}
                
            elif response.status_code == 429:
                logger.error(f"Превышен лимит запросов")
                self.stats["failed_requests"] += 1
                return {"error": "Превышен лимит запросов. Попробуйте позже."}
                
            elif response.status_code == 410:
                logger.error(f"API endpoint больше не поддерживается")
                self.stats["failed_requests"] += 1
                return {"error": "API endpoint больше не поддерживается"}
                
            else:
                logger.error(f"Ошибка API: {response.status_code} - {response.text[:200]}")
                self.stats["failed_requests"] += 1
                return {"error": f"Ошибка API: {response.status_code}"}
                
        except requests.exceptions.Timeout:
            logger.error("Таймаут при запросе к Hugging Face")
            self.stats["failed_requests"] += 1
            return {"error": "Таймаут соединения"}
            
        except requests.exceptions.ConnectionError:
            logger.error("Ошибка подключения к Hugging Face")
            self.stats["failed_requests"] += 1
            return {"error": "Ошибка подключения к интернету"}
            
        except Exception as e:
            logger.error(f"Неожиданная ошибка: {e}")
            self.stats["failed_requests"] += 1
            return {"error": f"Неожиданная ошибка: {str(e)}"}
    
    def generate_text(
        self,
        prompt: str,
        max_length: int = 200,
        temperature: float = 0.7,
        top_p: float = 0.9,
        repetition_penalty: float = 1.0,
        model: Optional[str] = None
    ) -> str:
        """
        Генерация текста на основе промпта
        
        Args:
            prompt: Текст для продолжения
            max_length: Максимальная длина ответа
            temperature: Температура (креативность)
            top_p: Top-p sampling
            repetition_penalty: Штраф за повторения
            model: Имя модели (если None, используется дефолтная)
            
        Returns:
            Сгенерированный текст
        """
        if model is None:
            model = self.default_models["text_generation"]
        
        parameters = {
            "max_length": max_length,
            "temperature": temperature,
            "top_p": top_p,
            "repetition_penalty": repetition_penalty,
            "do_sample": True,
            "return_full_text": False,  # Не возвращать промпт в ответе
        }
        
        result = self.query_huggingface(model, prompt, parameters)
        
        if isinstance(result, dict) and "error" in result:
            # Пробуем фолбэк модель
            logger.warning(f"Ошибка с основной моделью, пробуем фолбэк: {result['error']}")
            fallback_model = self.default_models["text_generation_fallback"]
            if model != fallback_model:
                return self.generate_text(
                    prompt, max_length, temperature, top_p, 
                    repetition_penalty, fallback_model
                )
            return f"Ошибка: {result['error']}"
        
        # Обработка разных форматов ответа
        if isinstance(result, list):
            if len(result) > 0:
                if "generated_text" in result[0]:
                    return result[0]["generated_text"]
                elif isinstance(result[0], dict) and "generated_text" in result[0]:
                    return result[0]["generated_text"]
                elif isinstance(result[0], str):
                    return result[0]
        elif isinstance(result, dict) and "generated_text" in result:
            return result["generated_text"]
        elif isinstance(result, str):
            return result
        
        # Если формат не распознан
        logger.warning(f"Неизвестный формат ответа: {type(result)}")
        return str(result)
    
    def question_answer(
        self,
        context: str,
        question: str,
        model: Optional[str] = None
    ) -> str:
        """
        Ответ на вопрос на основе контекста
        
        Args:
            context: Контекст для поиска ответа
            question: Вопрос
            model: Имя модели
            
        Returns:
            Ответ на вопрос
        """
        if model is None:
            model = self.default_models["question_answering"]
        
        inputs = {
            "question": question,
            "context": context
        }
        
        result = self.query_huggingface(model, inputs)
        
        if isinstance(result, dict) and "error" in result:
            return f"Ошибка: {result['error']}"
        
        if isinstance(result, dict) and "answer" in result:
            return result["answer"]
        elif isinstance(result, list) and len(result) > 0:
            if isinstance(result[0], dict) and "answer" in result[0]:
                return result[0]["answer"]
        
        return str(result)
    
    def summarize_text(
        self,
        text: str,
        max_length: int = 130,
        min_length: int = 30,
        model: Optional[str] = None
    ) -> str:
        """
        Суммаризация текста
        
        Args:
            text: Текст для суммаризации
            max_length: Максимальная длина суммы
            min_length: Минимальная длина суммы
            model: Имя модели
            
        Returns:
            Суммаризованный текст
        """
        if model is None:
            model = self.default_models["summarization"]
        
        parameters = {
            "max_length": max_length,
            "min_length": min_length,
            "do_sample": False,
        }
        
        result = self.query_huggingface(model, text, parameters)
        
        if isinstance(result, dict) and "error" in result:
            return f"Ошибка: {result['error']}"
        
        if isinstance(result, list) and len(result) > 0:
            if "summary_text" in result[0]:
                return result[0]["summary_text"]
            elif isinstance(result[0], dict) and "summary_text" in result[0]:
                return result[0]["summary_text"]
            elif isinstance(result[0], str):
                return result[0]
        
        return str(result)
    
    def translate_text(
        self,
        text: str,
        source_lang: str = "ru",
        target_lang: str = "en",
        model: Optional[str] = None
    ) -> str:
        """
        Перевод текста
        
        Args:
            text: Текст для перевода
            source_lang: Исходный язык
            target_lang: Целевой язык
            model: Имя модели
            
        Returns:
            Переведенный текст
        """
        if model is None:
            if source_lang == "en" and target_lang == "ru":
                model = self.default_models["translation_en_ru"]
            elif source_lang == "ru" and target_lang == "en":
                model = self.default_models["translation_ru_en"]
            else:
                return f"Перевод с {source_lang} на {target_lang} не поддерживается"
        
        result = self.query_huggingface(model, text)
        
        if isinstance(result, dict) and "error" in result:
            return f"Ошибка: {result['error']}"
        
        if isinstance(result, list) and len(result) > 0:
            if "translation_text" in result[0]:
                return result[0]["translation_text"]
            elif isinstance(result[0], dict) and "translation_text" in result[0]:
                return result[0]["translation_text"]
            elif isinstance(result[0], str):
                return result[0]
        
        return str(result)
    
    def analyze_sentiment(
        self,
        text: str,
        model: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Анализ настроения текста
        
        Args:
            text: Текст для анализа
            model: Имя модели
            
        Returns:
            Результаты анализа настроения
        """
        if model is None:
            model = self.default_models["sentiment"]
        
        result = self.query_huggingface(model, text)
        
        if isinstance(result, dict) and "error" in result:
            return {"error": result["error"]}
        
        # Обработка результатов анализа настроения
        if isinstance(result, list):
            return {
                "label": result[0][0]["label"],
                "score": result[0][0]["score"],
                "text": text
            }
        
        return {"result": result}
    
    def analyze_image_with_text(
        self,
        image_path: str,
        prompt: str,
        model: str = "Salesforce/blip-image-captioning-large"
    ) -> str:
        """
        Анализ изображения с текстовым промптом
        
        Args:
            image_path: Путь к изображению
            prompt: Текстовый промпт
            model: Модель для анализа изображений
            
        Returns:
            Результат анализа
        """
        try:
            # Кодируем изображение в base64
            with open(image_path, "rb") as image_file:
                image_data = base64.b64encode(image_file.read()).decode("utf-8")
            
            inputs = {
                "image": image_data,
                "text": prompt
            }
            
            result = self.query_huggingface(model, inputs)
            
            if isinstance(result, dict) and "error" in result:
                return f"Ошибка анализа изображения: {result['error']}"
            
            if isinstance(result, list) and len(result) > 0:
                if "generated_text" in result[0]:
                    return result[0]["generated_text"]
                elif isinstance(result[0], str):
                    return result[0]
            
            return str(result)
            
        except FileNotFoundError:
            return f"Файл не найден: {image_path}"
        except Exception as e:
            return f"Ошибка обработки изображения: {str(e)}"
    
    def get_model_info(self, model_name: str) -> Optional[Dict]:
        """
        Получение информации о модели
        
        Args:
            model_name: Имя модели
            
        Returns:
            Информация о модели
        """
        try:
            url = f"{self.models_url}/{model_name}"
            response = requests.get(url, timeout=10)
            
            if response.status_code == 200:
                return response.json()
        except Exception as e:
            logger.error(f"Ошибка получения информации о модели: {e}")
        
        return None
    
    def get_available_models(self, limit: int = 20) -> List[Dict]:
        """
        Получение списка доступных моделей
        
        Args:
            limit: Количество моделей для возврата
            
        Returns:
            Список моделей
        """
        try:
            params = {
                "sort": "downloads",
                "direction": -1,
                "limit": limit,
                "full": "true"
            }
            
            response = requests.get(self.models_url, params=params, timeout=10)
            
            if response.status_code == 200:
                return response.json()
        except Exception as e:
            logger.error(f"Ошибка получения списка моделей: {e}")
        
        return []
    
    def get_statistics(self) -> Dict[str, Any]:
        """
        Получение статистики использования
        
        Returns:
            Статистика
        """
        stats = self.stats.copy()
        stats["cache_size"] = len(self.cache)
        stats["cache_hit_rate"] = (
            stats["cache_hits"] / stats["total_requests"] * 100 
            if stats["total_requests"] > 0 else 0
        )
        stats["success_rate"] = (
            stats["successful_requests"] / stats["total_requests"] * 100 
            if stats["total_requests"] > 0 else 0
        )
        return stats
    
    def clear_cache(self) -> None:
        """Очистка кэша"""
        self.cache.clear()
        logger.info("Кэш очищен")