import re
import logging
from typing import Dict, List, Tuple, Optional
from dataclasses import dataclass

logger = logging.getLogger(__name__)

@dataclass
class PrivacyRule:
    pattern: str
    replacement: str
    description: str

class PrivacyEngine:
    """Движок для очистки конфиденциальных данных"""
    
    def __init__(self):
        self.rules = self._build_privacy_rules()
        self.context_placeholders = {}
        
    def _build_privacy_rules(self) -> List[PrivacyRule]:
        """Строим правила для очистки конфиденциальных данных"""
        return [
            # Электронная почта
            PrivacyRule(
                r'\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b',
                '[EMAIL]',
                "Email адреса"
            ),
            # Номера телефонов
            PrivacyRule(
                r'(\+7|7|8)?[\s\-]?\(?[0-9]{3}\)?[\s\-]?[0-9]{3}[\s\-]?[0-9]{2}[\s\-]?[0-9]{2}',
                '[PHONE]',
                "Номера телефонов"
            ),
            # Пути файлов Windows
            PrivacyRule(
                r'[A-Za-z]:\\[\\\S|*\S]?\S*?',
                '[FILEPATH]',
                "Пути к файлам Windows"
            ),
            # Пути файлов Linux/Mac
            PrivacyRule(
                r'/(home|Users|tmp|var|etc)/[^\s]*',
                '[FILEPATH]',
                "Пути к файлам Unix"
            ),
            # IP адреса
            PrivacyRule(
                r'\b(?:\d{1,3}\.){3}\d{1,3}\b',
                '[IP_ADDRESS]',
                "IP адреса"
            ),
            # Кредитные карты
            PrivacyRule(
                r'\b(?:\d{4}[- ]?){3}\d{4}\b',
                '[CREDIT_CARD]',
                "Номера кредитных карт"
            ),
            # Документы
            PrivacyRule(
                r'\b\d{4}[- ]?\d{6}\b',
                '[DOCUMENT]',
                "Номера документов"
            ),
            # Имена пользователей
            PrivacyRule(
                r'\b(user|username|login):?\s*\w+\b',
                '[USERNAME]',
                "Имена пользователей"
            ),
            # Домены и URL
            PrivacyRule(
                r'\b(https?://[^\s]+|www\.[^\s]+|[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})\b',
                '[URL]',
                "Веб-адреса"
            ),
        ]
    
    def clean_query(self, query: str, user_context: Dict = None) -> Tuple[str, Dict]:
        """Очищает запрос от конфиденциальных данных"""
        logger.info("🛡️ Начинаем очистку запроса от конфиденциальных данных")
        
        cleaned_query = query
        context_map = {}
        placeholder_counter = 1
        
        for rule in self.rules:
            matches = list(re.finditer(rule.pattern, cleaned_query, re.IGNORECASE))
            
            for match in matches:
                original_text = match.group()
                placeholder = f"[{rule.description.upper()}_{placeholder_counter}]"
                
                context_map[placeholder] = {
                    'original': original_text,
                    'type': rule.description,
                    'rule': rule.pattern
                }
                
                cleaned_query = cleaned_query.replace(original_text, placeholder)
                placeholder_counter += 1
                
                logger.debug(f"🛡️ Заменено: '{original_text}' -> '{placeholder}'")
        
        cleaned_query = self._contextual_cleaning(cleaned_query, user_context)
        
        logger.info(f"🛡️ Очистка завершена. Заменено {len(context_map)} элементов")
        return cleaned_query, context_map
    
    def _contextual_cleaning(self, query: str, user_context: Dict = None) -> str:
        """Дополнительная контекстная очистка"""
        personal_patterns = [
            r'^(привет|здравствуйте|добрый день),?\s*',
            r'\b(пожалуйста|спасибо|извините)\b',
        ]
        
        for pattern in personal_patterns:
            query = re.sub(pattern, '', query, flags=re.IGNORECASE)
        
        return query.strip()
    
    def restore_context(self, generic_response: str, context_map: Dict) -> str:
        """Восстанавливает персонализированный контекст в ответе"""
        if not context_map:
            return generic_response
        
        personalized_response = generic_response
        
        for placeholder, context_info in context_map.items():
            original_value = context_info['original']
            
            if self._should_restore(placeholder, generic_response):
                personalized_response = personalized_response.replace(
                    placeholder, original_value
                )
                logger.debug(f"🛡️ Восстановлено: '{placeholder}' -> '{original_value}'")
        
        return personalized_response
    
    def _should_restore(self, placeholder: str, response: str) -> bool:
        """Определяет, нужно ли восстанавливать конкретное значение в ответе"""
        sensitive_placeholders = ['CREDIT_CARD', 'DOCUMENT', 'PASSWORD']
        
        for sensitive in sensitive_placeholders:
            if sensitive in placeholder:
                return False
        
        return True
    
    def get_privacy_report(self) -> Dict:
        """Отчет о примененных правилах приватности"""
        return {
            "total_rules": len(self.rules),
            "rules_applied": [rule.description for rule in self.rules],
            "cleaned_items_count": len(self.context_placeholders)
        }