import os
import pyautogui
from datetime import datetime
from pathlib import Path
import re
import logging

logger = logging.getLogger(__name__)

class ScreenAnalyzer:
    def __init__(self, screenshots_dir: str = None):
        if screenshots_dir is None:
            self.screenshots_dir = Path.home() / ".jarvis_screenshots"
        else:
            self.screenshots_dir = Path(screenshots_dir)
        
        # Создаем директорию если не существует
        self.screenshots_dir.mkdir(exist_ok=True)
        logger.info(f"📁 Директория скриншотов: {self.screenshots_dir}")
    
    def _create_safe_filename(self, problem_description: str) -> str:
        """Создает безопасное имя файла из описания проблемы"""
        try:
            # Удаляем недопустимые символы для имен файлов
            safe_name = re.sub(r'[<>:"/\\|?*]', '_', problem_description)
            safe_name = re.sub(r'[@]', '_at_', safe_name)  # Заменяем @
            safe_name = re.sub(r'[\s]', '_', safe_name)    # Заменяем пробелы
            safe_name = safe_name[:50]  # Ограничиваем длину
            
            # Добавляем timestamp для уникальности
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"problem_{safe_name}_{timestamp}.png"
            
            logger.info(f"📝 Создано имя файла: {filename}")
            return filename
            
        except Exception as e:
            logger.error(f"❌ Ошибка создания имени файла: {e}")
            # Возвращаем имя по умолчанию
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            return f"problem_{timestamp}.png"
    
    def capture_problem_screenshot(self, problem_description: str) -> dict:
        """Создает скриншот проблемы и возвращает информацию о нем"""
        try:
            # Создаем безопасное имя файла
            filename = self._create_safe_filename(problem_description)
            screenshot_path = self.screenshots_dir / filename
            
            logger.info(f"📸 Создание скриншота: {screenshot_path}")
            
            # Создаем скриншот
            screenshot = pyautogui.screenshot()
            screenshot.save(str(screenshot_path))
            
            logger.info(f"✅ Скриншот сохранен: {screenshot_path}")
            
            return {
                "success": True,
                "screenshot_path": str(screenshot_path),
                "filename": filename,
                "message": f"Скриншот создан: {filename}"
            }
            
        except Exception as e:
            error_msg = f"Ошибка при создании скриншота: {str(e)}"
            logger.error(error_msg)
            return {
                "success": False,
                "error": error_msg,
                "screenshot_path": None
            }
    
    def analyze_ui_problem(self, problem_description: str) -> str:
        """Анализирует проблему с UI"""
        try:
            screenshot_result = self.capture_problem_screenshot(problem_description)
            
            if screenshot_result["success"]:
                return f"✅ Скриншот проблемы создан: {screenshot_result['filename']}\n" \
                       f"📍 Путь: {screenshot_result['screenshot_path']}\n" \
                       f"📝 Описание: {problem_description}"
            else:
                return f"❌ Не удалось создать скриншот: {screenshot_result['error']}"
                
        except Exception as e:
            return f"❌ Ошибка анализатора: {str(e)}"
    
    def get_recent_screenshots(self, limit: int = 10) -> list:
        """Возвращает список последних скриншотов"""
        try:
            screenshots = []
            for file in self.screenshots_dir.glob("*.png"):
                if file.is_file():
                    screenshots.append({
                        "name": file.name,
                        "path": str(file),
                        "size": file.stat().st_size,
                        "created": datetime.fromtimestamp(file.stat().st_ctime)
                    })
            
            # Сортируем по дате создания (новые сначала)
            screenshots.sort(key=lambda x: x["created"], reverse=True)
            return screenshots[:limit]
            
        except Exception as e:
            logger.error(f"Ошибка получения списка скриншотов: {e}")
            return []
    
    def is_available(self) -> bool:
        """Проверяет, доступен ли анализатор экрана"""
        try:
            # Простая проверка - пытаемся получить размер экрана
            pyautogui.size()
            return True
        except Exception as e:
            logger.error(f"Анализатор экрана недоступен: {e}")
            return False