"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cx_api_1 = require("@aws-cdk/cx-api");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 *
 * @stability stable
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlanBase extends core_1.Resource {
    /**
     * Adds an ApiKey.
     *
     * @param apiKey the api key to associate with this usage plan.
     * @param options options that control the behaviour of this method.
     * @stability stable
     */
    addApiKey(apiKey, options) {
        let id;
        const prefix = 'UsagePlanKeyResource';
        if (core_1.FeatureFlags.of(this).isEnabled(cx_api_1.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID)) {
            id = `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}`;
        }
        else {
            // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodified.
            id = this.node.tryFindChild(prefix) ? `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}` : prefix;
        }
        const resource = new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
        if (options === null || options === void 0 ? void 0 : options.overrideLogicalId) {
            resource.overrideLogicalId(options === null || options === void 0 ? void 0 : options.overrideLogicalId);
        }
    }
}
/**
 * @stability stable
 */
class UsagePlan extends UsagePlanBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.any({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Import an externally defined usage plan using its ARN.
     *
     * @param scope the construct that will "own" the imported usage plan.
     * @param id the id of the imported usage plan in the construct tree.
     * @param usagePlanId the id of an existing usage plan.
     * @stability stable
     */
    static fromUsagePlanId(scope, id, usagePlanId) {
        class Import extends UsagePlanBase {
            constructor() {
                super(scope, id);
                this.usagePlanId = usagePlanId;
            }
        }
        return new Import();
    }
    /**
     * Adds an apiStage.
     *
     * @stability stable
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
_a = JSII_RTTI_SYMBOL_1;
UsagePlan[_a] = { fqn: "@aws-cdk/aws-apigateway.UsagePlan", version: "1.126.0" };
//# sourceMappingURL=data:application/json;base64,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